<?php

/**
 * \file Traitement.class.inc
 * \brief Traitement.class.inc \n \n Ce fichier contient la classe php Traitement.
 *
 * Cette classe virtuelle permet de générer les éléments nécessaires à la création d'un rapport ou
 * à l'éxécution d'un traitement FME.
 *
 * Cette classe permet de générer des fichiers textes, csv ou d'images pour la création de rapport.
 * Elle permet aussi de lancer la création d'un rapport ou l'éxécution d'un traitement FME.
 *
 * \author Fabien Marty <fabien.marty@veremes.com>
 * \update Matthieu Ambrosy <matthieu.ambrosy@veremes.com>
 */
require_once ("vmlib/phpUtil.inc");
require_once ("vmlib/logUtil.inc");
require_once ("vmlib/stringUtil.inc");
require_once ("gtf_lib/GtfFmwParser.class.inc");
require_once 'string.inc';

class Traitement {

    //Variable stockant l'url du log FME
    var $sLogFme = "";

    /**
     * Identifiant de la demande.
     */
    var $iDemandeId;

    /**
     * Fichier de log.
     */
    var $sRobotLogFile;

    /**
     * Objet de la classe Bd. Connexion à la base de données
     */
    var $oBd;

    /**
     * Boolean spécifiant si le traitement est en erreur ou pas.
     */
    var $bErreur = false;

    /**
     * Boolean spécifiant si FME a subit un crash.
     */
    var $bFmeCrash = false;

    /**
     * Message d'erreur.
     */
    var $sMessageErreur;

    /**
     * Tableau des paramètres
     */
    var $aParametre;

    /**
     * Chaine contenant les parametres
     */
    var $sParams;

    /**
     * Emplacement du fichier projet FMW
     */
    var $sFmwFileName;

    /**
     * Tableau des propriétés provenant du fichier properties_robot.inc
     */
    var $aProperties;

    /**
     * Tableau des extensions associées à une source de données GUI
     */
    var $aExtension;
    var $sDestination;
    var $sSource;
    var $sSourceTemp = "";

    /**
     * Cette méthode permet d'initialiser les variables.
     * \param $oBd Connexion à la base de données.
     * \param $iDemandeId Identifiant de la demande.
     * \param $sRobotLogFile Fichier de log.
     */
    function __construct($oBd, $iDemandeId, $sRobotLogFile, $sParams, $aProperties, $sFmwFileName) {
        $this->oBd = $oBd;
        $this->iDemandeId = $iDemandeId;
        $this->sRobotLogFile = $sRobotLogFile;
        $this->sFmwFileName = $sFmwFileName;
        $this->aParametre = $this->dbParamsAsArray($sParams);
        $this->sParams = $sParams;
        $this->aProperties = $aProperties;
        // table de correspondance entre GTF et le FMW utiliser dans les fonction d'encodage 
        $this->aReplaceGTF = array("\r\n", " ", "(", ")", "$", "&", "@", "°", "'", ",", "\"", "{", "}", "[", "]", ";", "/", "\\", "é", "à");
        $this->aReplaceFME = array("<lf>", "<space>", "<openparen>", "<closeparen>", "<dollar>", "<amp>", "<at>", "<u00b0>", "<apos>", "<comma>", "<quote>", "<opencurly>", "<closecurly>", "<openbracket>", "<closebracket>", "<semicolon>", "<solidus>", "<backslash>", "<u00e9>", "<u00e0>");

        writeToLog(str_replace('[this->sFmwFileName]', $this->sFmwFileName, str_replace('[sParams]', $sParams, INFO_RESOURCES)), $this->sRobotLogFile);
    }

    /*
     * Initialisation du traitement.
     * \param $oBd Connexion à la base de données.
     * \param $iDemandeId Identifiant de la demande.
     * \param $sLogFile Fichier de log.
     * \param $sParams Liste des paramètres saisis par l'utilisateur à la création de la demande.
     * \param $aProperties Tableau des properties associés à l'application
     * \param $sFmwFileName Nom du fichier FMW associé à la demande.
     */

    /**
     * Cette méthode permet de générer un nom de fichier unique.
     * \return une chaîne de caractères.
     */
    function UniqFileName() {
        $sUniqFileName = date('YmdHis') . rand(1, 100000);
        return $sUniqFileName;
    }

    /**
     * Cette méthode permet de générer un tableau de paramètres à partir d'une chaîne dont les délimiteurs sont "|" et "=".
     * \param $sAllParams Chaine composé de tous les paramètres.
     * \return un tableau contenant les paramètres.
     */
    function dbParamsAsArray($sAllParams) {
        $aAllParams = Array();
        $aLigne = explode("|", $sAllParams);
        //$aLigne = stringExplode($sAllParams, "|", "\"", "\"");
        foreach ($aLigne as $sValeur) {
            $aValeurs = explode("=", $sValeur, 2);
            if ($aValeurs[1] == "") {
                writeToLog(INFO_NULL_ELEMENT . $aValeurs[0] . ".", $this->sRobotLogFile);
            }
            if (!empty($aValeurs[2])) {
                $sTemp = $aValeurs[1] . "=" . $aValeurs[2];
                $aValeurs[1] = $sTemp;
            }
            $aAllParams[$aValeurs[0]] = $aValeurs[1];
            writeToLog(str_replace('[aValeurs[0]]', $aValeurs[0], INFO_DB_PARAM) . $aValeurs[1], $this->sRobotLogFile);
        }
        return $aAllParams;
    }

    /**
     * Cette méthode permet d'éxécuter une ligne de commande et de stocker les paramètres dans un fichier
     * \param $sCommand Ligne de commande à éxécuter.
     */
    function execute($sCommand, $sFMELogFile) {
        $sExeName = "FME";
        if (!$this->bErreur) {
            set_time_limit(0);
            //$sResultCommande = exec('"'.$this->aProperties["fme_path"].'" APPLY_SETTINGS "Python/Python Interpreter"', $aResult, $iResult);
            $sResultCommande = exec('"' . $this->aProperties["fme_path"] . '" APPLY_SETTINGS "Python/Use Custom Python" false', $aResult, $iResult);
            //writeToLog(INFO_INFORM.$sExeName."| ".'"'.$this->aProperties["fme_path"].'" APPLY_SETTINGS "Python/Python Interpreter"', $this->sRobotLogFile);
            //writeToLog(INFO_INFORM.$sExeName."| ".$sResultCommande, $this->sRobotLogFile);
            // $sResultCommande = exec('"'.$this->aProperties["fme_path"].'" APPLY_SETTINGS "Python/Python Interpreter" "C:/apps/FME-2015.1-b15477-32/fmepython27/python27.dll"', $aResult, $iResult);
            // writeToLog(INFO_INFORM.$sExeName."| APPLY_SETTINGS \"Python/Python Interpreter\" \"C:/apps/FME-2015.1-b15477-32/fmepython27\"", $this->sRobotLogFile);
            // writeToLog(INFO_INFORM.$sExeName."| ".$sResultCommande, $this->sRobotLogFile);
            $sResultCommande = exec($sCommand, $aResult, $iResult);
            if ($sResultCommande == "") {
                $aResultCommande[0] = true;
            } else {
                $aResultCommande = explode("|", $sResultCommande);
            }
            //Récupération d'une erreur.
			if ($iResult !=0){
            //if ($iResult || ($aResultCommande[0] != 0)) {
                $this->bErreur = true;
                if (sizeOf($aResultCommande) == 1)
                    $this->bFmeCrash = false;
            }
            //Écriture dans le log
            if ($this->bErreur) {
                if (!empty($aResultCommande[1])) {
                    if ($this->aProperties["convert_utf8"]) {
                        $aResultCommande[1] = utf8_decode($aResultCommande[1]);
                    }
                    $this->sMessageErreur = "Erreur lors de l exécution de la commande." . $aResultCommande[1] . ".";
                } else {
                    $this->sMessageErreur = "Erreur lors de l exécution de la commande. Consultez le log du robot.";
                }
				$bExistFMELogFile = true;
				 if (file_exists($sFMELogFile) === false) {
					$bExistFMELogFile = false;
				}
                foreach ($aResult as $sValeur) {
                    writeToLog(INFO_ERROR . "   " . "| " . $sValeur, $this->sRobotLogFile);
					if ($bExistFMELogFile === false){
						$monfichier = fopen($sFMELogFile, 'a');
						fputs($monfichier, $sValeur. "\n");
						fclose($monfichier);
					}
                }
                writeToLog(INFO_COMMAND_ERROR . $sCommand . ".", $this->sRobotLogFile);
            } else {
                foreach ($aResult as $sValeur) {
                    if (mb_detect_encoding($sValeur, "UTF-8, ISO-8859-1") != "UTF-8") {
                        $sValeur = iconv("ISO-8859-1", "UTF-8", $sValeur);
                    }
                    writeToLog(INFO_INFORM . $sExeName . "| " . $sValeur, $this->sRobotLogFile);
                }
                writeToLog(INFO_SUCCESSFUL_TREATMENT, $this->sRobotLogFile);
            }
        }
    }

    /**
     * Cette méthode permet de 7zipper des fichiers.
     * \param $sZipDir Emplacement des fichiers à zipper.
     * \param $sZipUrl URL du fichier zip.
     * \param $sZipFileName Nom du fichier zip à compresser.
     * \return L'url du fichier zip.
     */
    function zipFile($sZipDir, $sZipFileDir, $sZipUrl, $sZipFileName) {
        writeToLog(str_replace('[sZipDir]', $sZipDir, INFO_FILE_COMPRESSION), $this->sRobotLogFile);
        if (createZip($sZipDir, $sZipFileDir . '/' . $sZipFileName . '.zip', 'log')) {
            $sZipFile = $sZipUrl . "/" . $sZipFileName . ".zip";
            writeToLog(INFO_COMPRESSED_FILES . $sZipFile . ".", $this->sRobotLogFile);
        } else {
            $this->bErreur = true;
            $sZipFile = "";
            writeToLog(INFO_TREATMENT_FAILURE, $this->sRobotLogFile);
            writeToLog(str_replace('[sZipFileDir]', $sZipFileDir, str_replace('[sZipDir]', $sZipDir, INFO_FILES_COMPRESSION_FAILURE)) . $sZipFileName, $this->sRobotLogFile);
            $this->sMessageErreur = "Le traitement n\'a produit aucun résultat. Consultez son log.";
        }
        return $sZipFile;
    }

    function unZipSource($sParam) {
        $this->sSource = $sParam;
        // Si la chaine contient un accent, la chaine sera en UTF-8, la fonction file_exists et la fonction de dézippage ne gèrent pas utf-8, il faut donc convertir la chaine en ISO-8859-1
        $bUtf8Param = false;
        if (mb_check_encoding($sParam, 'UTF-8')) {
            $sParam = utf8_decode($sParam);
            $bUtf8Param = true;
        }
        // Si le fichier n'est pas trouvé alors il se trouve dans le répertoire upload
        if (!file_exists($sParam))
            $sParam = $this->aProperties["upload_dir"] . "/" . $sParam;

        // If faut gérer l'ensemble des zip (.zip, .7z, .rar, bzip2, gzip, tar, wim, xz)
        if (mb_eregi("(.ZIP)$", strtoupper($sParam), $aOccurence)) {
            // Décompression du fichier zip.
            $sDir = $this->unZipFile($sParam);
        } else { // Le fichier uploadé n'est pas un fichier zippé
            $sDir = $sParam;
            $this->sSourceTemp = "";
        }
        // Si le paramètre passé était encodé en UTF-8 il faut le réencoder sinon problème avec les accents.
        if ($bUtf8Param)
            $sDir = utf8_encode($sDir);
        return $sDir;
    }

    /**
     * Cette méthode permet de dézipper un fichier.
     * \param $sZipFile Le fichier zip à décompresser.
     * \return L'emplacement des fichiers qui ont été extraits.
     */
    function unZipFile($sZipFile) {
        if (!$this->bErreur) {
            writeToLog(INFO_DECOMPRESSING_FILE . $sZipFile . ".", $this->sRobotLogFile);
            // Les backslashes du chemin de la variable d'environnement TEMP sont remplacés par des slashes
            $this->aProperties["extract_dir"] = str_replace("\\", "/", $this->aProperties["extract_dir"]);
            $sExtractDir = $this->aProperties["extract_dir"] . "/" . $this->UniqFileName() . "/";
            $this->sSourceTemp = $sExtractDir;
            if (unZip($sZipFile, $sExtractDir)) {
                writeToLog(INFO_FILE_COMPRESSION_DIRECTORY . $sExtractDir . ".", $this->sRobotLogFile);
            } else {
                $this->bErreur = true;
                writeToLog(INFO_TREATMENT_FAILURE, $this->sRobotLogFile);
                writeToLog(str_replace('[sZipFile]', $sZipFile, INFO_FILE_COMPRESSION_FAILURE) . $sExtractDir . ".", $this->sRobotLogFile);
                $this->sMessageErreur = "Impossible de dézipper le fichier. Consultez le log du robot.";
                $sExtractDir = "";
            }
            return $sExtractDir;
        }
    }

    /*
     * Fonction qui se charge de la mise à jour du paramètre "source" s'il existe
     * $oGtfFmwParser : Objet GtfFmwParser contenant les GUI
     * $oGui : Objet GUI de la source
     * $sDirSource : Chaine de caractère contenant le chemin d'accés à la source de données
     */

    function updateSourceParam($oGui, $oGtfFmwParser, $sDirSource, $bUsePattern) {
        $sdelimiterFichier = '"""';
        $sdelimiterParam = '"';
        $sSourceMacro = $sdelimiterFichier;
		$aSourceMacro = array();
        $i = 0;

        if (is_dir($sDirSource)) {
            $this->aExtension = $oGui->getExtensionArrayNoLower($oGui->sFiltreSource);
            $iExtension = count($this->aExtension);
            if (($this->aExtension[0] == "*.*" and $oGui->sType == "FILE_OR_URL") or $oGui->sType == "DIRNAME_SRC" or $oGui->sType == "SOURCE_GEODATABASE") {
                writeToLog(INFO_VALID_DATA_SOURCE_FILE_EXTENSION . $this->aExtension[0], $this->sRobotLogFile);
                $sSourceMacro.= $sdelimiterParam . $sDirSource . '/**/' . $sdelimiterParam;
            } else {
				if ($bUsePattern){
					writeToLog(INFO_USE_PATTERN, $this->sRobotLogFile);
					foreach ($this->aExtension as $sExtension) {
						if ($sExtension != "*.*" or ( $sExtension == "*.*" and $iExtension == 1)) { // Si le filtre est différent de *.* (ex : *.mdb) ou si le filtre est seulement *.*. Si le filtre est multiple et que l'un d'eux est *.* (ex : *.*, *.mdb...), il ne sera pas pris en compte
							$aExtension = explode(".", $sExtension);
							if ($this->countFilesExtension($sDirSource, $aExtension[1]) > 0) {
								if ($i != 0)
									$sSourceMacro.= "<space>";
								writeToLog(INFO_VALID_DATA_SOURCE_FILE_EXTENSION . $sExtension, $this->sRobotLogFile);
								$sSourceMacro.= $sdelimiterParam . $sDirSource . '/**/' . $sExtension . $sdelimiterParam;
								$i++;
							}
						}
					}
				}else{
					writeToLog(INFO_NO_USE_PATTERN, $this->sRobotLogFile);
					foreach ($this->aExtension as $sExtension) {
						if ($sExtension != "*.*" or ( $sExtension == "*.*" and $iExtension == 1)) { // Si le filtre est différent de *.* (ex : *.mdb) ou si le filtre est seulement *.*. Si le 
							$aExtension = explode(".", $sExtension);
							$aSourceMacroTmp = array_merge($aSourceMacro, $this->returnFileListToString($sDirSource, $aExtension[1]));
							$aSourceMacro = array_unique($aSourceMacroTmp);
							$i++;
						}
					}
					foreach ($aSourceMacro as $source) {
						if ($sSourceMacro != $sdelimiterFichier){
							$sSourceMacro.= "<space>";
						}
						$sSourceMacro.= $sdelimiterParam . $source . $sdelimiterParam;
					}
                }
            }
        }else {
            $sSourceMacro.= $sdelimiterParam . $sDirSource . $sdelimiterParam;
        }
        $sSourceMacro .= $sdelimiterFichier;
        $this->aParametre[$oGui->sDefault_Macro] = $sSourceMacro;
		
        return $sSourceMacro;
    }

    // Compte le nombre de fichier
    // $sDirSource : Répertoire de base
    // $sExtension : Filtre
    function countFilesExtension($sDirSource, $sExtension) {
        $num = 0;
        if (!is_dir($sDirSource)) {
            $path_parts = pathinfo($sDirSource);
            if (is_file($sDirSource) && $path_parts['extension'] == $sExtension)
                return 1;
            else
                return false;
        }
        foreach (scandir($sDirSource) AS $entry) {
            if (!in_array($entry, array('..', '.')))
                $num += $this->countFilesExtension($sDirSource . "/" . $entry, $sExtension);
        }
        return $num;
    }
	
	function returnFileListToString($sDirSource, $sExtension) {
        $aFileList = array();
        if (!is_dir($sDirSource)) {
            $path_parts = pathinfo($sDirSource);
            if (is_file($sDirSource) && strtolower($path_parts['extension']) == strtolower($sExtension)){
                array_push($aFileList,$sDirSource);
			}
            
        }else{
			foreach (scandir($sDirSource) AS $entry) {
				if (!in_array($entry, array('..', '.'))){
					$aFileListTmp = $aFileList;
					$aFileList = array_merge($aFileListTmp,$this->returnFileListToString($sDirSource . "/" . $entry, $sExtension));
				}
				
			}
		}
        return $aFileList;
    }

    /*
     * Fonction qui se charge de la mise à jour du paramètre "source" s'il existe
     * $oGtfFmwParser : Objet GtfFmwParser contenant les GUI
     * $oGui : Objet GUI de la source
     * $sDirSource : Chaine de caractère contenant le chemin d'accés à la source de données
     */

    function updateDestParam($oGui, $oGtfFmwParser, $sNewDir, $sDest) {
        $this->aExtension = $oGui->getExtensionArray($oGui->sFiltreDest);
        if ($sDest == "" && $oGui->sType != "DIRNAME") {
            $sDest = $sNewDir;
        }
        $this->aParametre[$oGui->sDefault_Macro] = "";
        $sDest = normalizeString($sDest);
        $aDest = explode(".", $sDest);
        if ($oGui->sType == "DIRNAME" or $oGui->sType == "DEST_GEODATABASE") {
            $this->aParametre[$oGui->sDefault_Macro] = $this->aProperties["dir_export"] . "/gtf/" . $sNewDir . "/" . $sDest;
            writeToLog(INFO_OUTPUT_DATA_DIRECTORY . $this->aParametre[$oGui->sDefault_Macro], $this->sRobotLogFile);
        }
        if ($oGui->sType == "FILENAME") {
            if (count($aDest) > 1) { // Dans GTF, la valeur du champ [nom de fichier en sortie] comporte une extension. Ex : result.xls (le fichier résultat sera de l'excel et sera nommé result.
                $sResult = $sDest;
            } else {
                $sResult = str_replace("*", $sDest, $this->aExtension[0]);
            }
            $this->aParametre[$oGui->sDefault_Macro].= $this->aProperties["dir_export"] . "/gtf/" . $sNewDir . '/' . $sResult;
            $this->sDestination = $sResult;

            writeToLog(INFO_OUTPUT_DATA_FILE . $this->aParametre[$oGui->sDefault_Macro], $this->sRobotLogFile);
        }
        else if ($oGui->sType == "FILENAME_EXISTING")
            $this->aParametre[$oGui->sDefault_Macro] = $sDest;
        return $this->aParametre[$oGtfFmwParser->sDestMacro];
    }

    /*     * *************************************************************************************** */
    /*                                          Codage                                        */

    function unichr($u) {
        return mb_convert_encoding('&#' . intval($u) . ';', 'UTF-8', 'HTML-ENTITIES');
    }

    function ordutf8($string, &$offset) {
        $code = ord(substr($string, $offset, 1));
        if ($code >= 128) {        //otherwise 0xxxxxxx
            if ($code < 224)
                $bytesnumber = 2;                //110xxxxx
            else if ($code < 240)
                $bytesnumber = 3;        //1110xxxx
            else if ($code < 248)
                $bytesnumber = 4;    //11110xxx
            $codetemp = $code - 192 - ($bytesnumber > 2 ? 32 : 0) - ($bytesnumber > 3 ? 16 : 0);
            for ($i = 2; $i <= $bytesnumber; $i++) {
                $offset ++;
                $code2 = ord(substr($string, $offset, 1)) - 128;        //10xxxxxx
                $codetemp = $codetemp * 64 + $code2;
            }
            $code = $codetemp;
        }
        $offset += 1;
        if ($offset >= strlen($string))
            $offset = -1;
        return $code;
    }

    function encoderFME($sStr) {
        //table de correcpondance des symboles UTF-8 avec les codes FME/GTF
        // ATTENTION : laisser < > en premier sinon replace remplace les balises par d'autre balise
        $aReplaceGTFToFMW = array("<" => "<lt>", ">" => "<gt>");
        $sReplaced = $sStr;

        //on gere les chevrons simultanèment pour ne pas gener l'encodage après
        if (strpos($sReplaced, "<") !== false && strpos($sReplaced, ">") !== false) {
            $sReplaced = strtr($sReplaced, $aReplaceGTFToFMW);
        }

        //on remplace tous les encodage par leurs symboles
        for ($i = 0; $i < count($this->aReplaceGTF); $i++) {
            if (strpos($sReplaced, $this->aReplaceGTF[$i]) !== false) {
                $sReplaced = str_replace($this->aReplaceGTF[$i], $this->aReplaceFME[$i], $sReplaced);
            }
        }

        // on encode les balises UTF-8 		

        $offset = 0;
        $iPos = 0;

        while ($offset >= 0) {
            $iCode = $this->ordutf8($sReplaced, $offset);
            if ($iCode > 127) {
                $cLetter = dechex($iCode);
                while (strlen($sLetter) < 4) {
                    $cLetter = "0" . $cLetter;
                }
                $sReplaced = str_replace($sReplaced[$iPos], "<u" . $cLetter . ">", $sReplaced);
            }
            $iPos += 1;
        }

        return $sReplaced;
    }

    function decoderFME($sStr) {
        //table de correcpondance des symboles UTF-8 avec les codes FME/GTF
        // ATTENTION : laisser < > en premier sinon replace remplace les balises par d'autre balise
        $aReplaceFMWToGTF = array("<lt>" => "<", "<gt>" => ">");
        $sReplaced = $sStr;

        //on gere les chevrons simultanèment pour ne pas gener l'encodage après
        if (strpos($sReplaced, "<lt>") !== false && strpos($sReplaced, "<gt>") !== false) {
            $sReplaced = strtr($sReplaced, $aReplaceFMWToGTF);
        }
        //on remplace tous les encodage par leurs symboles
        for ($i = 0; $i < count($this->aReplaceGTF); $i++) {
            if (strpos($sReplaced, $this->aReplaceFME[$i]) !== false) {
                $sReplaced = str_replace($this->aReplaceFME[$i], $this->aReplaceGTF[$i], $sReplaced);
            }
        }

        // on decode les balises UTF-8 		
        $matches = array();

        preg_match_all("/<u[0-9a-fA-F]{4}>/", $sStr, $matches);

        for ($i = 0; $i < count($matches[0]); $i++) {
            $iCode = hexdec(substr($matches[0][$i], 2, 4));
            $cChr = $this->unichr($iCode);
            $sReplaced = str_replace($matches[0][$i], $cChr, $sReplaced);
        }
        return $sReplaced;
    }

    function isDecode($sStr) {
        for ($i = 0; $i < count($this->aReplaceFME); $i++) {
            if (strpos($sStr, $this->aReplaceFME[$i]) !== false && strpos($sStr, $this->aReplaceGTF[$i]) === false) {
                return false;
            }
        }
        return true;
    }

    function isEncode($sStr) {
        for ($i = 0; $i < count($this->aReplaceFME); $i++) {
            if (strpos($sStr, $this->aReplaceFME[$i]) === false && strpos($sStr, $this->aReplaceGTF[$i]) !== false) {
                return false;
            }
        }
        return true;
    }

    /*     * *************************************************************************************** */

    /*
     *  Fonction qui permet de concaténer tous les paramètres nécessaire pour l'appel du TCL
     *  et qui retourne $sChaine, la chaine des paramètres à passer au TCL
     *  $oGtfFmwParser : Objet GtfFmwParser contenant les GUI
     */

    function getTclParams($oGtfFmwParser, $sNewDir) {
        $aTGui = array();
        $sChaine = "";
		$bUsePattern = false;
        foreach ($oGtfFmwParser->aGuiObject as $aGui) {
            if ($aGui->sDefault_Macro != "") {
                array_push($aTGui, $aGui);
            }
			if ($aGui->sDefault_Macro == "GTF_FORCE_PATTERN") {
				if (strtolower($aGui->sDefaultValue) == "true"){
					$bUsePattern = true;
				}
			}
        }
        foreach ($aTGui as $oGui) {
            //Teste si le paramètre est une source
            if ($oGui->bIsSource) {
                if ($this->aParametre[$oGui->sDefault_Macro] != "" && substr($this->aParametre[$oGui->sDefault_Macro], 0, 7) != "<quote>") {
                    $sDirSource = $this->unZipSource($this->aParametre[$oGui->sDefault_Macro]);
                    $this->updateSourceParam($oGui, $oGtfFmwParser, $sDirSource, $bUsePattern);
                }
            }
            if ($oGui->bIsDest) {
                $sDest = $this->aParametre[$oGui->sDefault_Macro];
                $this->updateDestParam($oGui, $oGtfFmwParser, $sNewDir, $sDest);
            }

            if ($this->aParametre[$oGui->sDefault_Macro] != "") {

                if (!$oGui->bIsSource) {
                    // Protection des guillemets de chaque valeur pour qu'ils soient correctement envoyé à traitement.tcl
                    // Cette protection s'effectue uniquement pour les paramètre qui ne sont pas sources (un paramètre source est de ce type : """"param1"" ""param2"" ""param3"""" : les guillemets seraient donc protégé et cela fait planter l'exécution)
                    //$this->aParametre[$oGui->sDefault_Macro] = str_replace('"', '""', $this->aParametre[$oGui->sDefault_Macro]);
                }
                $aBaliseToReplace = array('<GTF_EQUAL>', '<GTF_PIPE>');
                $aReplacedBalise = array('=', '|');
                $sGTFOutput = str_replace($aBaliseToReplace, $aReplacedBalise, $this->aParametre[$oGui->sDefault_Macro]);
                switch ($oGui->sType) {
                    case "FLOAT":
                        $sGTFOutput = str_replace(",", ".", $sGTFOutput);
                        if (!$this->isDecode($sGTFOutput))
                            $sGTFOutput = $this->decoderFME($sGTFOutput);
                        break;
                    case "LOOKUP_CHOICE":
                        if (!$this->isEncode($sGTFOutput))
                            $sGTFOutput = $this->encoderFME($sGTFOutput);
                        break;
                    case "TEXT_EDIT" :
                    case "TEXT_EDIT_OR_NUM" :
                        if (!$this->isEncode($sGTFOutput)) {
                            //$sGTFOutput = $this->decoderFME($sGTFOutput);
                            $sGTFOutput = $this->encoderFME($sGTFOutput);
                        }
                        break;
                    case "CHOICE":
                    case "LOOKUP_LISTBOX":
                        $sGTFOutput = str_replace('"', '""', $sGTFOutput);
                        if (!$this->isDecode($sGTFOutput))
                            $sGTFOutput = $this->decoderFME($sGTFOutput);
                        break;
                    default :
                        //$sGTFOutput = str_replace('"', '""', $sGTFOutput);
                        if (!$this->isDecode($sGTFOutput))
                            $sGTFOutput = $this->decoderFME($sGTFOutput);
                        break;
                }
                // Echappement des doubles quotes pour FME.
                $sGTFOutput = str_replace('"', '\"', $sGTFOutput);
                //
                $sChaine .= " --" . $oGui->sDefault_Macro . " \"" . $sGTFOutput . "\"";
            }
        }

        // $sChaine = substr($sChaine, 0, -1)."\"";
        //$sChaine = substr($sChaine, 0, -1);
        return $sChaine;
    }

    function listDir($sDossier, $iDemandeId) {
        $ouverture = @opendir($sDossier);
        if (!$ouverture)
            return false;
        while ($sFichier = readdir($ouverture)) {
            if ($sFichier == '.' || $sFichier == '..')
                continue;
            if (is_dir($sDossier . $sFichier)) {
                closedir($ouverture);
                return true;
            } else {
                $path_parts = pathinfo($sDossier . $sFichier);
                if ($sFichier != "fme_" . $iDemandeId . ".log") {
                    closedir($ouverture);
                    return true;
                }
            }
        }
        closedir($ouverture);
        return False;
    }

    /*
     * Fonction qui retourne le nom du fichier résultat qui sera zippé.
     *  $sDossier correspond au chemin complet dans lequel les fichiers/dossiers résultats seront générés
     *  $sNewDir correspond au numéro unique du dossier dans lequel les fichiers/dossiers résultats seront générés
     *  $iDemandeId correspond au numéro de la demande
     * \return Retourne NULL (s'il y a plusieurs fichiers résultat à zipper) ou le fichier résultat qui sera zippé sous la forme 20140324141501876/ventes.xlsx
     */

    function CheckResult($sDossier, $sNewDir, $iDemandeId) {
        $ouverture = @opendir($sDossier);
        $NbFile = 0;
        while ($sFichier = readdir($ouverture)) {
            if ($sFichier == '.' || $sFichier == '..')
                continue;
            if (is_dir($sDossier . $sFichier)) {
                $NbFile = $NbFile + 2;
            } else {
                $path_parts = pathinfo($sDossier . $sFichier);
                if ($sFichier != "fme_" . $iDemandeId . ".log") {
                    $NbFile = $NbFile + 1;
                }
            }
        }
        closedir($ouverture);
        $ouverture = @opendir($sDossier);
        if ($NbFile != 1) {
            return "";
        } else {
            $pattern = '/^' . str_replace("*", ".+", str_replace(".", "\\.", $this->aProperties['not_compressed_extension'])) . '$/';
            while ($sFichier = readdir($ouverture)) {
                if ($sFichier == '.' || $sFichier == '..')
                    continue;
                if ($sFichier != "fme_" . $iDemandeId . ".log") {
                    if (preg_match($pattern, $sFichier)) {
                        return $sNewDir . "/" . $sFichier;
                    } else {
                        return "";
                    }
                }
            }
        }
        closedir($ouverture);
    }

    /*
     * fonction qui lance la génération de la ligne de commande, l'éxécution de la demande et la mise en forme du résultat
     * \return Retourne les fichiers résultant zippés
     */

    function Process() {
        $oGtfFmwParser = new GtfFmwParser($this->aProperties["workspace_dir"] . "/" . $this->sFmwFileName);
        // Création du répertoire de sortie des données
        $sNewDir = $this->UniqFileName();
		 if (!file_exists($this->aProperties["dir_export"] . "/gtf/")){
			mkdir($this->aProperties["dir_export"] . "/gtf/");
		}
        $sDestDir = $this->aProperties["dir_export"] . "/gtf/" . $sNewDir;
        mkdir($sDestDir);
        // Création de l'URL de téléchargement des données
        $sDestUrl = $this->aProperties["url_export"] . "/gtf/" . $sNewDir;

        //1 Initialisation & calcul du chemin d'accés du fichier TCL
        $sCommandLine = '"' . $this->aProperties["fme_path"] . '" "' . getenv("GTF_HOME") . "/" . utf8_decode($this->sFmwFileName) . '"';



        //3 Ajout des paramètres propres au traitement
        $sTclParams = $this->getTclParams($oGtfFmwParser, $sNewDir);
        if ($sTclParams != "")
            $sCommandLine .= $sTclParams;

		$aTGui = array();
        $sChaine = "";
		  // Ajout des paramètres GTF_ s'ils sont associés à un paramètre publié du traitement
        foreach ($oGtfFmwParser->aGuiObject as $aGui) {
			switch($aGui->sDefault_Macro) {
				 case 'GTF_ORDER_ID':
					$sCommandLine .= " --GTF_ORDER_ID \"" . $this->iDemandeId . "\"";
					break;
				 case 'GTF_SHARED_DIR':
					$sCommandLine .= " --GTF_SHARED_DIR \"" . $this->aProperties["shared_dir"] . "\"";
					break;
				 case 'GTF_CONNECTION_STRING':
					if ($aGui->sType == "PASSWORD"){
						 $sCommandLine .= "  --GTF_CONNECTION_STRING \"" . $this->aProperties["server"] . "," . $this->aProperties["port"] . "," . $this->aProperties["database"] . "," . $this->aProperties["login_scheduler"] . "," . $this->aProperties["password_scheduler"] . "\"";
					}
					 break;
				 case 'GTF_REST_URL':
					 $sCommandLine .= "  --GTF_REST_URL \"" . $this->aProperties["web_server_name"] . "/" . $this->aProperties["services_alias"] . "/\"";
					 break;
				 //case 'GTF_FORCE_PATTERN':
			}
           
        }
        /*
         * 2 Passage des paramètres généraux
         * - FMWFILENAME : fichier Fmw
         * - LOG_ROBOT : fichier log du robot
         * - LOG_FILENAME : fichier log pour FME
         */
        $sCommandLine.= " -LOG_FILENAME \"" . $sDestDir . "/fme_" . $this->iDemandeId . ".log\" 2>&1";
        writeToLog(INFO_GTF_HOME . getenv("GTF_HOME"), $this->sRobotLogFile);
        writeToLog(INFO_FME_PATH . getenv("FME_PATH"), $this->sRobotLogFile);
        writeToLog(INFO_TREATMENT_COMMAND_LINE . $sCommandLine, $this->sRobotLogFile);

        //4 Lancement de la ligne de commande
        $this->sLogFme = $sNewDir . "/fme_" . $this->iDemandeId . ".log";
        $sSql = "UPDATE " . $this->aProperties['schema_gtf'] . ".order SET log_url='" . $this->sLogFme . "' WHERE order_id=" . $this->iDemandeId;
        $this->oBd->execute($sSql);
        $this->execute($sCommandLine, $sDestDir . "/fme_" . $this->iDemandeId . ".log");
        writeToLog(INFO_TREATMENT_COMMAND_LINE . $sCommandLine, $this->sRobotLogFile);
        if (file_exists($sDestDir . "/fme_" . $this->iDemandeId . ".log") == false) {
            $this->sLogFme = "";
        }
        $bHasFile = $this->listDir($sDestDir, $this->iDemandeId);
        if ($this->sSourceTemp != "" && $this->aProperties["debug_mode"] == false) {
            clearDir($this->sSourceTemp);
        }
        if ($bHasFile) {
            $sIsToZip = $this->CheckResult($sDestDir, $sNewDir, $this->iDemandeId);

            if ($sIsToZip == "") {
                //Compression en zip des fichiers résultant. Le parametres Array() permet de filtrer les extensions à compresser. Exemples : Array("shp","shx","dbf") ; Array("*")		
                $sZipFile = $this->zipFile($sDestDir, $sDestDir, $sDestUrl, $this->iDemandeId);
                $sZipFile = str_replace($sDestUrl, $sNewDir, $sZipFile);
            } else {
                $sZipFile = str_replace($sDestUrl, $sNewDir, $sIsToZip);
            }
        } else {
            $sZipFile = "";
        }
        return $sZipFile;
    }

}

?>
