"use strict";

// Tableau de retour (succès ou échec)
var REQUEST_SUCCESS = 1;
var REQUEST_ERROR = 0;

// Calendrier
var FIRST_DAY_WEEK = 1; // 0 = dimanche, 1 = lundi (défaut)

// RGraph
var CANVAS_ID = 'statsGraph'; // Id de l'élément html <canvas>
var BORDER_WIDTH = 100; // Bord à enlever à la largeur max du graphique (ne pas compter la largeur du form. de stats)
var BORDER_HEIGHT = 130; //  Bord à enlever à la hauteur max du graphique
//var GRAPH_COLORS = ['FireBrick', 'Red', 'Tomato', 'DarkOrange', 'Gold', 'Yellow', 'YellowGreen'];
var GRAPH_COLORS = {"palette1": ['YellowGreen', 'DarkGreen', 'LightSkyBlue', 'Maroon', 'MediumSeaGreen', 'Khaki', 'Chocolate', 'MediumBlue', 'BlueViolet', 'Aqua', 'Red', 'Lime', 'Blue', 'Darkorange'],
    "palette2": ['Red', 'OliveDrab', 'OrangeRed', 'DodgerBlue', 'DarkOrange', 'Blue', 'Gold', 'Indigo', 'Yellow', 'DarkOrchid', 'GreenYellow', 'Brown', 'Silver', 'Black']};

var TITLE_POS_Y = 15; // Position verticale du titre
var GUTTER_LEFT = 45; // Marge gauche pour les graphiques (sinon chiffres invisibles)
var GUTTER_TOP = 40; // Marge haute pour le titre
var TITLE_BOLD = true; // Titre du graphique en gras ?
var TITLE_SIZE = 10; // Taille de la police du titre
var TEXT_SIZE = 10; // Taille de la police de 'label', 'key' et des données
var KEY_TEXT_SIZE = 10; // Taille de la police de la légende
var SCATTER_X_SPACING = 3; // Espacement des données pour le graphique en chandelier
var UNIT_ID_DECIMALS = '{"unit_sec":0, "unit_min":0, "unit_hour":1, "unit_pc":2}'; // Nombre de décumales suivant l'unité
var STATS_PER_DETAIL = true; // Affichage d'un graph de stats suvant la période de la colonne cliquée (année -> un mois, mois -> un jour...)

// Paramètres des graphiques pour chaque variable
var JSON_GRAPH_ARRAY = '{"var_tjl":{"nb_graph":2, "graph_type": ["Bar","Pie"]}, "var_ujl":{"nb_graph":1, "graph_type": ["Scatter"]}, "var_jn":{"nb_graph":2, "graph_type": ["Bar","Pie"]}}';
var JSON_GRAPH_TYPE_ARRAY = '{"Bar": "detail", "Pie": "general"}';
var DEFAULT_MAX_ELEMENTS = 5; // Nb éléments max. à afficher par défaut
var DEFAULT_MAX_LABELS_CHARACTERS = 50; // Longueur max. des labels

// Paramètres de la légende du graphique en chandelier
var SCATTER_POS_Y = 50; // Position verticale de la légende
var SCATTER_LEGEND_WIDTH = 20; // Largeur
var SCATTER_LEGEND_HEIGHT = 30; // Hauteur
var SCATTER_LEGEND_LEFT_MARGIN = 4; // Marge gauche par rapport au graphique
var SCATTER_LEGEND_COLOR_UP = 'black'; // Couleur de la barre max (haut)
var SCATTER_LEGEND_COLOR_DOWN = 'white'; // // Couleur de la barre min (bas)

//
var CANVAS_PARENT_ID = "double_form_right_section_statistics_gtf_statistics"; // Element ou seront affichés les canvas
var TABLE_ID = 'statsTable'; // Id de l'élément html contenant les stats en mode tableau 
var TABLE_TITLE = '<p id="statsTableTitle">[title]</p>';
var TABLE_HTML = '<table id="' + TABLE_ID + '"><thead><tr><th>[legend]</th></tr></thead><tfoot><tr></tr></tfoot><tbody></tbody></table>';
var ERROR_MESSAGE_ELEMENT_ID = "statistics-error-message";
//var AJAX_LOADING_IMG = "images/treeview/load-root.gif"; // Image à afficher pendant le chargement/affichage des stats

/***********************************
 Charge les données d'un graphique
 ***********************************/
function loadGraphData(iGraphIndex)
{
// Crée une chaine de caractères avec les champs du formulaire pour la requête ajax
    var aFormParams = $("form").serializeArray();
    // Dédoublonnage (".serializeArray()" et ".serialize()" retournent les paramètres en doubles ?).
    var aFilteredFormParams = [], sUrlParams = "";
    for (var i = 0; i < aFormParams.length; i++) {
        if (typeof(aFilteredFormParams[aFormParams[i]["name"]]) == "undefined") {
            aFilteredFormParams[aFormParams[i]["name"]] = aFormParams[i]["value"];
            if (sUrlParams != "")
                sUrlParams += "&";
            sUrlParams += aFormParams[i]["name"] + "=" + aFormParams[i]["value"];
        }
    }    
    //
    var promise = $.Deferred();

    //var dataStats;
    $.ajax({
        type: "GET",
        url: sessionStorage["statistics_web_service_url"],
        async: false,
        cache: false,
        dataType: "json", // transforme auto. le tableau en json
        Accept: "application/json",
        data: sUrlParams,
        "headers": {
             "Token": sessionStorage["session_token"]
        },
        error: function (jqXHR, textStatus, errorThrown) {
            alert('Erreur Ajax : ' + textStatus);
        },
        success: function (data)
        {
            // Traduction des libellés retournés par le web service.
            var sStatisticsResult = JSON.stringify(data);
            var aTranslationsId = sStatisticsResult.match(/([A-Z_]+_STATISTICS_STATISTICS)+/g);
            var $translate = angular.element(vitisApp.appMainDrtv).injector().get(["$translate"]);
            $translate(aTranslationsId, data["statistics"]["translation_values"]).then(function (oTranslations) {
                var i = 0;
                var aKeys = Object.keys(oTranslations);
                var regExp;
                while (aKeys.length > i) {
                    //sStatisticsResult = eval("sStatisticsResult.replace(/" + aKeys[i] + "/g, oTranslations[aKeys[i]])");
                    regExp = new RegExp(aKeys[i], "g");
                    sStatisticsResult = sStatisticsResult.replace(regExp, oTranslations[aKeys[i]]);
                    i++;
                }
                //
                data = JSON.parse(sStatisticsResult);
                if (data["status"] == 1) {
                    // Erreur renvoyée par la classe php ?
                    if (data["statistics"]["status"] == REQUEST_ERROR)
                        showErrorMsg(data["statistics"]["message"]);
                } else
                    showErrorMsg(data["message"]);
                //
                promise.resolve(data["statistics"], iGraphIndex);
            });
        }
    });
    return promise;
}

/***********************************
 Affiche un message d'erreur
 ***********************************/
function showErrorMsg(message)
{
    $("#" + ERROR_MESSAGE_ELEMENT_ID).text(message);
    $("#" + ERROR_MESSAGE_ELEMENT_ID).show();
}

/***********************************
 Supprime un message d'erreur
 ***********************************/
function cleanErrorMsg()
{
    $("#" + ERROR_MESSAGE_ELEMENT_ID).hide();
    $("#" + ERROR_MESSAGE_ELEMENT_ID).empty();
}

/***********************************
 Affiche un graphique (Bar)
 - aRGraphDatas : Tableau json contenant les paramètres graphiques pour RGraph (retour classe php)
 - sCanvas : Id de l'élément html (<canvas>) contenant le graphique
 ***********************************/
function displayBarGraph(aJsonRGraphDatas, sCanvas)
{
    var myGraph = new RGraph["Bar"](sCanvas, aJsonRGraphDatas["data"]);

// Type de regroupement et séparation	
    myGraph["Set"]('grouping', 'stacked');	// Plusieurs colonnes par grain (voir demo : bar-stacked.html)
    myGraph["Set"]('strokestyle', 'white'); // Couleur de la ligne de séparation entre les barres (sur un même label)
    myGraph["Set"]('linewidth', 1); // Largeur de la ligne de séparation

    myGraph["Set"]('chart.title.y', TITLE_POS_Y);

// Marges	
    myGraph["Set"]('gutter.left', GUTTER_LEFT); // Marge gauche pour le graphique
    myGraph["Set"]('gutter.top', GUTTER_TOP); // Marge haute pour le graphique (par rapport au titre)
//	myGraph["Set"]('chart.key.position', 'gutter'); // Position de la légende (key)

// Largeur et position de la légende suivant la longueur de la chaine la plus longue * taille de la police
    myGraph["Set"]('chart.key.position.y', TEXT_SIZE + 5);
    var iMaxLengthStr = getMaxStrLength(aJsonRGraphDatas["key"]);
    if (aJsonRGraphDatas["legend"].length > iMaxLengthStr) // Le titre de la légende est plus long que les valeurs de la légende ?
        iMaxLengthStr = aJsonRGraphDatas["legend"].length;
    var iKeyWidth = iMaxLengthStr * (KEY_TEXT_SIZE - 4); // Largeur de la légende
    var iKeyPositionX = ($("#" + sCanvas).width() - 50) - iKeyWidth;
    myGraph["Set"]('chart.key.position.x', iKeyPositionX);
    myGraph["Set"]('gutter.right', iKeyWidth + 60); // Marge droite (pour la légende)

// Titre
    myGraph["Set"]('chart.title', aJsonRGraphDatas["title"]); // Titre du graphique
    myGraph["Set"]('chart.title.bold', TITLE_BOLD); // Titre en gras ?
    myGraph["Set"]('chart.title.size', TITLE_SIZE); // Taille de la police du titre

// Tooltips	(infobulles)
    myGraph["Set"]('chart.tooltips', function (i) {
        return aJsonRGraphDatas.key[(i % aJsonRGraphDatas["key"].length)];
    }); //
    myGraph["Set"]('chart.tooltips.event', 'onmousemove'); // onmousemove : passage de la souris / onclick
    myGraph["Set"]('chart.tooltips.effect', 'none'); // Effet pour l'affichage du tooltip
    myGraph["canvas"].onmouseout = function (e) {
        RGraph["HideTooltip"]();
    }; // Efface les infobulles

// Si clic sur une des colonnes : affiche un graph sur la période de cette colonne
    if (STATS_PER_DETAIL)
        myGraph["Set"]('chart.events.click', displayStatsPerDetail);

// Labels et keys (légende)
    myGraph["Set"]('chart.labels', aJsonRGraphDatas["labels"]); // Libellé par colonne (lundi, mardi...)
    myGraph["Set"]('labels.above', true); // Afficher le total des données au dessus
    myGraph["Set"]('chart.text.size', TEXT_SIZE);	// Taille de la police des labels
    myGraph["Set"]('chart.key', aJsonRGraphDatas["key"]); // Légende (projets FME, utilisateurs...)
    myGraph["Set"]('chart.key.text.size', KEY_TEXT_SIZE);	// Taille de la police des keys

//	myGraph["Set"]('bevel', true);	// Effet '3D' --> ralenti si affichage avec animation
//	myGraph["Set"]('chart.shadow', true); // Ombre portée --> ralenti si affichage avec animation

// Menu (bouton droit) sur le graphique
//	myGraph["Set"]('chart.contextmenu', [['Image', RGraph.showPNG]]);
//	myGraph["Set"]('chart.contextmenu', [['Image', function(){showGraphImage(sCanvas)}]]);
    myGraph["Set"]('chart.contextmenu', [['Image', function () {
                showGraphImage(sCanvas)
            }], ['Zoom Out', function () {
                displayStatsZoomOut()
            }]]);

// Couleurs (Palette par défaut sauf si une palette est passée en paramètre par la classe php)
    if (aJsonRGraphDatas["colors"] != undefined)
        myGraph["Set"]('colors', aJsonRGraphDatas["colors"]);	// Couleurs des barres (keys / légende)
    else
        myGraph["Set"]('colors', GRAPH_COLORS[document.getElementById("graph_colors").value]);	// Couleurs des barres (keys / légende)
    myGraph["Set"]('chart.background.color', 'white');
    //myGraph["Set"]('chart.colors', $.parseJSON(getColorGradient(aJsonRGraphDatas.key.length))); // Génération d'un dégradé

// Nb décimales en fonction de unit_id (heure, min, sec, %)	
    var aUnitIdDecimals = $.parseJSON(UNIT_ID_DECIMALS);
    myGraph["Set"]('chart.scale.decimals', aUnitIdDecimals[document.getElementById("unit_id").value]);
    myGraph["Set"]('chart.labels.above.decimals', aUnitIdDecimals[document.getElementById("unit_id").value]);
    myGraph["Set"]('chart.scale.point', ',');	// // Caractère précédent les décimales
    myGraph["Set"]('chart.scale.thousand', '');	// // Caractère après mille

// Unité des ordonnées (si présente : augmente la marge de gauche)
    if (aJsonRGraphDatas["data_unit"] != undefined)
    {
        myGraph["Set"]('chart.units.post', aJsonRGraphDatas["data_unit"]);
        myGraph["Set"]('gutter.left', GUTTER_LEFT + (aJsonRGraphDatas["data_unit"].length * (TEXT_SIZE - 3))); // Marge gauche pour le graphique
    }
    //myGraph.ondraw = function (obj){displayBarData(obj)}; // Affiche les valeurs pour chaque colonne	

// Affichage (sans animation)
    myGraph["Draw"]();

// Affichage du titre de la légende
    if (aJsonRGraphDatas["legend"] != '')
    {
        var oText = new RGraph["Drawing"]["Text"](sCanvas, iKeyPositionX + 5, TEXT_SIZE + 2, aJsonRGraphDatas["legend"]);
        oText["Set"]('chart.bold', true);
        oText["Draw"]();
    }

// Affichage avec animation	
//	RGraph.Effects.Bar.Grow(myGraph);
//	RGraph.Effects.Bar.Wave(myGraph);
}

/***********************************
 Affiche un graphique (Pie)
 - aRGraphDatas : Tableau json contenant les paramètres graphiques pour RGraph (retour classe php)
 - sCanvas : Id de l'élément html (<canvas>) contenant le graphique
 ***********************************/
function displayPieGraph(aJsonRGraphDatas, sCanvas)
{
    var myGraph = new RGraph["Pie"](sCanvas, aJsonRGraphDatas["data"][0]);
    myGraph["Set"]('chart.variant', 'donut'); // donut ou pie (camenbert)
    myGraph["Set"]('gutter.top', GUTTER_TOP); // Marge haute pour le graphique (par rapport au titre)

// Séparation	
    myGraph["Set"]('strokestyle', 'white'); // Couleur de la ligne de séparation entre les barres (sur un même label)
    myGraph["Set"]('linewidth', 1); // Largeur de la ligne de séparation

// Largeur et position de la légende suivant la longueur de la chaine la plus longue * taille de la police
    myGraph["Set"]('chart.key.position.y', TEXT_SIZE + 5);
    var iMaxLengthStr = getMaxStrLength(aJsonRGraphDatas["key"]);
    if (aJsonRGraphDatas["legend"].length > iMaxLengthStr) // Le titre de la légende est plus long que les valeurs de la légende ?
        iMaxLengthStr = aJsonRGraphDatas["legend"].length;
    var iKeyWidth = iMaxLengthStr * (KEY_TEXT_SIZE - 4); // Largeur de la légende
    var iKeyPositionX = ($("#" + sCanvas).width() - 50) - iKeyWidth;
    myGraph["Set"]('chart.key.position.x', iKeyPositionX);
    myGraph["Set"]('gutter.right', iKeyWidth + 60); // Marge droite (pour la légende)

// Titre	
    myGraph["Set"]('chart.title.y', TITLE_POS_Y);
    myGraph["Set"]('chart.title', aJsonRGraphDatas["title"]);
    myGraph["Set"]('chart.title.bold', TITLE_BOLD);	// Titre en gras ?
    myGraph["Set"]('chart.title.size', TITLE_SIZE);	// Taille de la police du titre

// Key (légende)
    myGraph["Set"]('chart.key', aJsonRGraphDatas["key"]); // Légende (projets FME, utilisateurs...)
    myGraph["Set"]('chart.key.text.size', KEY_TEXT_SIZE);	// Taille de la police des keys
    myGraph["Set"]('chart.text.size', TEXT_SIZE);	// Taille de la police des labels

// Tooltips	(infobulles)
    myGraph["Set"]('chart.tooltips', function (i) {
        return aJsonRGraphDatas["key"][(i % aJsonRGraphDatas["key"].length)];
    }); //
    myGraph["Set"]('chart.tooltips.event', 'onmousemove'); // onmousemove : passage de la souris / onclick
    myGraph["Set"]('chart.tooltips.effect', 'none'); // Effet pour l'affichage du tooltip
    myGraph["canvas"].onmouseout = function (e) {
        RGraph["HideTooltip"]();
    }; // Efface les infobulles
    //myGraph["Set"]('chart.exploded', 3); // Espace entre les tranches

// Couleurs (Palette par défaut sauf si une palette est passée en paramètre par la classe php)
    if (aJsonRGraphDatas["colors"] != undefined)
        myGraph["Set"]('colors', aJsonRGraphDatas["colors"]);	// Couleurs des barres (keys / légende)
    else
        myGraph["Set"]('colors', GRAPH_COLORS[[document.getElementById("graph_colors").value]]); // Couleurs des barres
    myGraph["Set"]('chart.background.color', 'white');
    //myGraph["Set"]('chart.colors', $.parseJSON(getColorGradient(aJsonRGraphDatas.key.length))); // Génération d'un dégradé

// Nb décimales en fonction de unit_id (heure, min, sec, %)	
    var aUnitIdDecimals = $.parseJSON(UNIT_ID_DECIMALS);
    myGraph["Set"]('chart.scale.decimals', aUnitIdDecimals[document.getElementById("unit_id").value]);
    myGraph["Set"]('chart.scale.point', ',');	// // Caractère précédent les décimales
    myGraph["Set"]('chart.scale.thousand', '');	// // Caractère après mille

// Unité des ordonnées	
    if (aJsonRGraphDatas["data_unit"] != undefined)
        myGraph["Set"]('chart.units.post', aJsonRGraphDatas["data_unit"]);

// Ombrage	
    //myGraph["Set"]('chart.shadow', true); // Ombre portée --> ralenti si affichage avec animation
    //myGraph["Set"]('chart.shadow.offsetx', 5);
    //myGraph["Set"]('chart.shadow.offsety', 5);

// Menu (bouton droit) sur le graphique
//	myGraph["Set"]('chart.contextmenu', [['Image', RGraph.showPNG]]);
    myGraph["Set"]('chart.contextmenu', [['Image', function () {
                showGraphImage(sCanvas)
            }]]);

    RGraph["Clear"](myGraph["canvas"], 'white'); // This effectively sets the background color to be white

// Affichage (sans animation)
    //myGraph.Draw();

// Affichage avec animation	
    RGraph["Effects"]["Pie"]["RoundRobin"](myGraph, {'radius': false, frames: 30}); // Effet d'affichage

// Affichage du titre de la légende	
    if (aJsonRGraphDatas["legend"] != '')
    {
        var oText = new RGraph["Drawing"]["Text"](sCanvas, iKeyPositionX + 5, TEXT_SIZE + 2, aJsonRGraphDatas["legend"]);
        oText["Set"]('chart.bold', true);
        oText["Draw"]();
    }
}

/***********************************
 Affiche un graphique (Scatter)
 - aRGraphDatas : Tableau json contenant les paramètres graphiques pour RGraph (retour classe php)
 - sCanvas : Id de l'élément html (<canvas>) contenant le graphique
 ***********************************/
function displayScatterGraph(aJsonRGraphDatas, sCanvas)
{
    var myGraph = new RGraph["Scatter"](sCanvas, aJsonRGraphDatas["data"]);

// Marges	
    myGraph["Set"]('gutter.top', GUTTER_TOP); // Marge haute pour le titre du graphique
    myGraph["Set"]('gutter.left', GUTTER_LEFT); // Marge gauche pour la légende des ordonnées
    myGraph["Set"]('chart.gutter.bottom', 150); // Marge pour les labels ?

// Titre	
    myGraph["Set"]('chart.title.y', TITLE_POS_Y);
    myGraph["Set"]('chart.title', aJsonRGraphDatas["title"]);
    myGraph["Set"]('chart.title.bold', TITLE_BOLD);	// Titre en gras ?
    myGraph["Set"]('chart.title.size', TITLE_SIZE);	// Taille de la police du titre

    myGraph["Set"]('chart.labels', aJsonRGraphDatas["labels"]);
    myGraph["Set"]('chart.text.size', TEXT_SIZE);	// Taille de la police des labels

// Labels	
    myGraph["Set"]('chart.tooltips.event', 'onmousemove'); // onmousemove : passage de la souris / onclick
    myGraph["Set"]('chart.tooltips.effect', 'none'); // Effet pour l'affichage du tooltip

    myGraph["Set"]('chart.boxplot.capped', false);
    myGraph["Set"]('chart.boxplot.width', 1);

    myGraph["Set"]('chart.text.angle', 30);

// Unité des ordonnées (si présente : augmente la marge de gauche)
    if (aJsonRGraphDatas.data_unit != undefined)
    {
        myGraph["Set"]('chart.units.post', aJsonRGraphDatas["data_unit"]);
        myGraph["Set"]('gutter.left', GUTTER_LEFT + (aJsonRGraphDatas["data_unit"].length * (TEXT_SIZE - 3))); // Marge gauche pour le graphique
    }

// Menu (bouton droit) sur le graphique
//	myGraph["Set"]('chart.contextmenu', [['Image', RGraph.showPNG]]);
    myGraph["Set"]('chart.contextmenu', [['Image', function () {
                showGraphImage(sCanvas)
            }]]);
    myGraph["Set"]('chart.background.color', 'white');

// Nb décimales en fonction de unit_id (heure, min, sec, %)	
    var aUnitIdDecimals = $.parseJSON(UNIT_ID_DECIMALS);
    myGraph["Set"]('chart.scale.decimals', aUnitIdDecimals[document.getElementById("unit_id").value]);

    myGraph["Set"]('chart.scale.point', ",");	// Caractère précédent les décimales
    myGraph["Set"]('chart.scale.thousand', '');	// // Caractère après mille
    var xmax = aJsonRGraphDatas["labels"].length * SCATTER_X_SPACING;
    if (aJsonRGraphDatas["labels"].length == 1)
        xmax += SCATTER_X_SPACING; // Si une seule barre : ajoute un espace sinon décalage
    myGraph["Set"]('chart.xmax', xmax); // Nb total barres * marge entre chaque barre

    myGraph["Set"]('chart.numxticks', 0);

// Affiche le graphique	
    myGraph["Draw"]();

    /*
     // Insère l'image de la légende (max, moy, min)
     imgPosX = ($(document).width() - $("#form_edit_first").width() - BORDER_WIDTH) - 110;
     var img = new RGraph["Drawing"]["Image"](sCanvas, imgPosX, 40, 'images/statistics/legend.png');
     img["Draw"]();	
     */

// Légende (min, moy, max)	
    displayScatterLegend(aJsonRGraphDatas, sCanvas);
}

/***********************************
 Affiche la légende (min, moy, max) pour 
 le graphique en chandelier (traitements unitaires)
 ***********************************/
function displayScatterLegend(aJsonRGraphDatas, sCanvas)
{
    var iPosX = $("#" + sCanvas).width() - 110;

// Rectangle du haut	
    var oRect = new RGraph["Drawing"]["Rect"](sCanvas, iPosX, SCATTER_POS_Y, SCATTER_LEGEND_WIDTH, SCATTER_LEGEND_HEIGHT);
    oRect["Set"]('chart.strokestyle', 'black');
    oRect["Set"]('chart.fillstyle', SCATTER_LEGEND_COLOR_UP);
    oRect["Draw"]();

// Rectangle du bas	
    oRect = new RGraph["Drawing"]["Rect"](sCanvas, iPosX, (SCATTER_POS_Y + SCATTER_LEGEND_HEIGHT), SCATTER_LEGEND_WIDTH, SCATTER_LEGEND_HEIGHT);
    oRect["Set"]('chart.strokestyle', 'black');
    oRect["Set"]('chart.fillstyle', SCATTER_LEGEND_COLOR_DOWN);
    oRect["Draw"]();

// Légende	
    var oText = new RGraph["Drawing"]["Text"](sCanvas, (iPosX + SCATTER_LEGEND_WIDTH + SCATTER_LEGEND_LEFT_MARGIN), (SCATTER_POS_Y + 7), aJsonRGraphDatas["scatter_legend"]["max"])["Draw"]();
    //oText.Set('chart.bold', true);
    oText = new RGraph["Drawing"]["Text"](sCanvas, (iPosX + SCATTER_LEGEND_WIDTH + SCATTER_LEGEND_LEFT_MARGIN), (SCATTER_POS_Y + (SCATTER_LEGEND_HEIGHT * 2) + 6), aJsonRGraphDatas["scatter_legend"]["min"])["Draw"]();
    oText = new RGraph["Drawing"]["Text"](sCanvas, (iPosX + SCATTER_LEGEND_WIDTH + SCATTER_LEGEND_LEFT_MARGIN), (SCATTER_POS_Y + SCATTER_LEGEND_HEIGHT + 6), aJsonRGraphDatas["scatter_legend"]["avg"])["Draw"]();
}

/***********************************
 Affiche l'image d'un graphique dans une nouvelle fenêtre
 ***********************************/
function showGraphImage(sCanvas)
{
    var canvas = document.getElementById(sCanvas);

// Pas de pixel transparent
    fillCanvasAlpha(sCanvas);
    /*
     var image_data = canvas.toDataURL("image/png");
     $.post("forms/statistics/save_image.phtml", { src: image_data } );
     */
    var my_win = window.open("", "Image", "menubar=no, status=no, scrollbars=no, menubar=no, width=" + (canvas.width + 50) + ", height=" + (canvas.height + 50) + "");
    var img = new Image();
    img.src = canvas.toDataURL("image/png");
    my_win.document.documentElement.innerHTML = '<head></head><body><img src="' + img.src + '"></body>';
}

/***********************************
 Colorie en blanc tous les pixels transparents du canvas
 ***********************************/
function fillCanvasAlpha(sCanvas)
{
    var canvas = document.getElementById(sCanvas);
    var context = canvas["getContext"]('2d');
    var imageData = context["getImageData"](0, 0, canvas.width, canvas.height);
    var pixelArray = imageData["data"];
    var i = 0;
    while (i < pixelArray.length)
    {
        // Colorie en blanc les pixels transparents
        if (pixelArray[i + 3] === 0 && pixelArray[i] === 0 && pixelArray[i + 1] === 0 && pixelArray[i + 2] === 0)
        {
            pixelArray[i] = 255;
            pixelArray[i + 1] = 255;
            pixelArray[i + 2] = 255;
            pixelArray[i + 3] = 255;
        }
        i += 4;
    }
    context["putImageData"](imageData, 0, 0);
}

/***********************************
 Affiche un graphique de stats suivant la période de la colonne cliquée:
 ex: An -> un mois / mois -> un jour...
 e : the (standard) event object
 shape : array of coordinates that describe the shape
 ***********************************/
function displayStatsPerDetail(e, shape)
{
// Supprime les infobulles
    RGraph["HideTooltip"]();

// Crée un objet date à partir de la date sélectionné dans le calendrier (notation GMT standard)
    var my_date = new Date($("#per_date").data("DateTimePicker")["date"]());

// Période sélectionnée
    var per_type = $("#per_type_save").val();

// Selection de la période
    var iSetPerDate = false;
    switch (per_type)
    {
        // Par semaine -> jour
        case "per_week":
            // Nouvelle période : par jour
            $("#per_type").val('per_day');

            // Modification de la date avec le jour cliqué
            my_date.setDate(my_date.getDate() + shape["dataset"]);

            //	
            iSetPerDate = true;
            break;

            // Par mois -> semaine
        case "per_month":
            // Nouvelle période : par semaine
            $("#per_type").val('per_week');

            // Modification de la date avec le jour cliqué
            my_date.setDate(my_date.getDate() + shape["dataset"]);

            //	
            iSetPerDate = true;
            break;

            // Par an -> mois
        case "per_year":
            // Nouvelle période : par mois	
            $("#per_type").val('per_month');

            // Modification de la date avec le mois cliqué
            my_date.setMonth(shape["dataset"]);

            //	
            iSetPerDate = true;
            break;
    }

// Remplir le champs de form. avec la nouvelle date ?
    if (iSetPerDate)
    {
        // Mise à jour et vérification de la nouvelle date	
        $("#per_date").data("DateTimePicker")["date"](my_date);
        checkPeriodDate();

        // Simule un click sur le bouton pour lancer la requête de stats	
        $('#btn_start_stats').click();
    }
}

/***********************************
 Affiche un graphique de stats sur la période supérieure:
 ex: jour -> semaine / semaine -> mois...
 ***********************************/
function displayStatsZoomOut()
{
// Supprime les infobulles
    RGraph["HideTooltip"]();

// Période sélectionnée
    var per_type = $("#per_type_save").val();

// Selection de la période
    var iSetPerDate = false;
    switch (per_type)
    {
        // Par jour -> semaine
        case "per_day":
            $("#per_type").val('per_week');
            iSetPerDate = true;
            break;
            // Par semaine -> mois
        case "per_week":
            $("#per_type").val('per_month');
            iSetPerDate = true;
            break;
            // Par mois -> an
        case "per_month":
            $("#per_type").val('per_year');
            iSetPerDate = true;
            break;
    }
// Remplir le champs de form. avec la nouvelle date ?
    if (iSetPerDate)
    {
        // Vérification de la nouvelle date	
        checkPeriodDate();

        // Simule un click sur le bouton pour lancer la requête de stats	
        $('#btn_start_stats').click();
    }
}

/***********************************
 Affiche un tableau de données
 - aRGraphDatas : Tableau json contenant les paramètres graphiques pour RGraph (retour classe php)
 - sElement : Id de l'élément html contenant le tableau
 ***********************************/
function displayTableStats(aJsonRGraphDatas, sElement)
{
// Affiche le titre du tableau
    var title = TABLE_TITLE;
    title = title.replace("[title]", aJsonRGraphDatas.title);
    $("#" + sElement).append(title);

// Création du tableau
    var sMyTableHtml = TABLE_HTML.replace("[legend]", aJsonRGraphDatas["legend"]); // Légende (projet FME, utilisateur...)
    $("#" + sElement).append(sMyTableHtml);

// Si traitements unitaires : affichage différent	
    if ($('#variable_id').val() == 'var_ujl')
    {
        // Insère les labels (titres) dans le thead (th)
        $("#" + TABLE_ID + ">thead>tr").append("<th>" + aJsonRGraphDatas["scatter_legend"]["min"] + "</th>");
        $("#" + TABLE_ID + ">thead>tr").append("<th>" + aJsonRGraphDatas["scatter_legend"]["avg"] + "</th>");
        $("#" + TABLE_ID + ">thead>tr").append("<th>" + aJsonRGraphDatas["scatter_legend"]["max"] + "</th>");

        // Insère les données pour chaque colonne (min, moy, max) dans le tbody (td)
        var i = 0;
        var nb_cols = 3; // min, moy, max
        while (i < aJsonRGraphDatas["labels"].length)
        {
            // Insère une nouvelle ligne + la légende (label)	
            $("#" + TABLE_ID + ">tbody").append("<tr></tr>");
            $("#" + TABLE_ID + ">tbody>tr:last-of-type").append("<td>" + aJsonRGraphDatas["labels"][i][0] + "</td>");

            // Insère les données : min, moy, max	
            var aData = aJsonRGraphDatas["data"][i][1];
            $("#" + TABLE_ID + ">tbody>tr:last-of-type").append("<td>" + aData[1] + "</td>");
            $("#" + TABLE_ID + ">tbody>tr:last-of-type").append("<td>" + aData[2] + "</td>");
            $("#" + TABLE_ID + ">tbody>tr:last-of-type").append("<td>" + aData[3] + "</td>");
            i++;
        }
    } else
    {

        // Insère les labels (titres) dans le thead (th)
        var i = 0;
        while (i < aJsonRGraphDatas["labels"].length)
        {
            $("#" + TABLE_ID + ">thead>tr").append("<th>" + aJsonRGraphDatas["labels"][i] + "</th>");
            i++;
        }

        // Insère les données pour chaque légende dans le tbody (td)
        var i = 0;
        var nb_cols = aJsonRGraphDatas["labels"].length;
        while (i < aJsonRGraphDatas["key"].length)
        {
            // Insère une nouvelle ligne + la légende (key)	
            $("#" + TABLE_ID + ">tbody").append("<tr></tr>");
            $("#" + TABLE_ID + ">tbody>tr:last-of-type").append("<td>" + aJsonRGraphDatas["key"][i] + "</td>");

            // Insère les données de la légende
            var j = 0;
            while (j < aJsonRGraphDatas["labels"].length)
            {
                $("#" + TABLE_ID + ">tbody>tr:last-of-type").append("<td>" + aJsonRGraphDatas["data"][j][i] + "</td>");
                j++;
            }
            i++;
        }
    }

// Calcul de la largeur des colonnes
//	var iColsWidth = parseInt(($("#" + TABLE_ID).width() - ($('#max_labels_characters').val() * 10))); //
    var iKeyWidth = parseInt(($("#" + TABLE_ID).width() * 25) / 100); // Largeur du titre pour chaque ligne : 15% du total
    var iColsWidth = parseInt($("#" + TABLE_ID).width() - iKeyWidth); //
    iColsWidth = parseInt(iColsWidth / nb_cols);
    $("#" + TABLE_ID + ">thead>tr th").css("width", iColsWidth);
    $("#" + TABLE_ID + ">tbody>tr td").css("width", iColsWidth);
    $("#" + TABLE_ID + ">thead>tr>th:first-of-type").css("width", "");
    $("#" + TABLE_ID + ">tbody>tr>td:first-of-type").css("width", "");
}

/***********************************
 Affiche le(s) graphique(s) ou un tableau de données d'une requête de stats
 ***********************************/
function displayStatsRequest()
{
    var promise;
// Supprime tous les graphiques sauvegardés par RGraph (sinon ralentissements)	
    $("canvas").each(function () {
        RGraph["Reset"](document.getElementById($(this).attr('id')));
    });

// Vide l'élément parent du/des canvas (élément des graphiques) ou d'un tableau
    $('#' + CANVAS_PARENT_ID).empty();

// Supprime le message d'erreur
    cleanErrorMsg();

// Affichage sous forme de graphique ou de tableau	?
    if (document.getElementById("radio_view_id_0").checked)
    {
        // Combien de graphiques à afficher pour la variable sélectionnée ?	
        var variable_id = $("#variable_id").val();
        var aGraph = $.parseJSON(JSON_GRAPH_ARRAY);
        var agraph_type = $.parseJSON(JSON_GRAPH_TYPE_ARRAY);

        // Largeur et hauteur max des graphiques
        var canvas_width = $("#" + CANVAS_PARENT_ID).width();
        var canvas_height = parseInt($("#" + CANVAS_PARENT_ID).height() / aGraph[variable_id]["nb_graph"]);
        /*
         if ($(document).width() < 1200){
         canvas_width = 1200 - $("#form_edit_first").width() - BORDER_WIDTH; // Enlève la largeur du menu de stats + une valeur fixe (BORDER_WIDTH)*
         }else{
         canvas_width = $(document).width() - $("#form_edit_first").width() - BORDER_WIDTH; // Enlève la largeur du menu de stats + une valeur fixe (BORDER_WIDTH)*
         }
         */
        //canvas_height = parseInt(($(document).height() - BORDER_HEIGHT) / aGraph[variable_id]["nb_graph"]);

        // Affichage du ou des graphiques
        var i = 0;
        while (i < aGraph[variable_id]["nb_graph"])
        {
            // Type de données à récupérer : générales ou détaillées (pie ou bar)
            $("#graph_type").val(agraph_type[aGraph[variable_id]["graph_type"][i]]);

            // Envoi une requête ajax (demande de graph. de stats à la classe php)	
            //var data = loadGraphData();
            promise = loadGraphData(i);
            $.when(promise).done(function (data, iGraphIndex) {
                // Si la classe php renvoie 'success' : affichage du graphique
                if (data["status"] == REQUEST_SUCCESS)
                {
                    // Force les libellés de la légende en type "string".
                    if (Array.isArray(data["key"])) {
                        data["key"].forEach(function(sKey, iIndex){
                                data["key"][iIndex] = String(sKey);
                        });
                    }
                    // Elément canvas où sera affiché le graphique
                    $('#' + CANVAS_PARENT_ID).append('<canvas id="' + CANVAS_ID + iGraphIndex + '" width="' + canvas_width + '" height="' + canvas_height + '">[No canvas support]</canvas>');

                    // Un sous-titre à afficher ?
                    if (data["subtitle"] != '')
                        data["title"] += '\n' + data["subtitle"];

                    // Type de graphique à afficher
                    switch (aGraph[variable_id]["graph_type"][iGraphIndex])
                    {
                        case "Bar":
                            displayBarGraph(data, CANVAS_ID + iGraphIndex);
                            break;

                        case "Pie":
                            displayPieGraph(data, CANVAS_ID + iGraphIndex);
                            break;

                        case "Scatter":
                            displayScatterGraph(data, CANVAS_ID + iGraphIndex);
                            break;
                    }
                }
            });
            i++;
        }
    } else
    {
        // Type de données à récupérer : générales ou détaillées
        $("#graph_type").val("detail");

        // Envoi une requête ajax (demande de graph. de stats à la classe php)	
        //var data = loadGraphData();
        promise = loadGraphData();
        $.when(promise).done(function (data) {
            // Un sous-titre à afficher ?
            if (data["subtitle"] != '')
                data["title"] = '<b>' + data["title"] + '</b>' + '<br>' + data["subtitle"];

            // Si la classe php renvoie 'success' : affichage du graphique
            if (data["status"] == REQUEST_SUCCESS)
            {
                // Affiche un tableau de données	
                displayTableStats(data, CANVAS_PARENT_ID);
            }
        });
    }
}

/***********************************
 Affiche les valeurs dans chaque barre
 - obj : l'objet RGraph
 ***********************************/
function displayBarData(obj)
{
    for (var i = 0; i < obj["coords"].length; ++i)
    {
        obj["context"]["fillStyle"] = 'white';
        RGraph["Text"](obj["context"], 'Verdana', 10, obj["coords"][i][0] + (obj["coords"][i][2] / 2), obj["coords"][i][1] + (obj["coords"][i][3] / 2), obj["data_arr"][i].toString(), 'center', 'center', null, null, null, true);
    }
}

/***********************************
 Vérifie si la date est valide suivant le type de la période (sinon modification de la date)
 - Semaine : dimanche ou lundi
 - Mois : 1er jour du mois
 - An : 1er janvier
 ***********************************/
function checkPeriodDate() // sDateText, oDatePicker
{
    //if (document.getElementById("per_date").value != window.sessionStorage["stats_per_date"]) {
    // Crée un objet date à partir de la date sélectionné dans le calendrier (notation GMT standard)
    if ($("#per_date").data("DateTimePicker")["date"]() != null)
        var my_date = new Date($("#per_date").data("DateTimePicker")["date"]());
    else
        var my_date = new Date();
    var per_type = $("#per_type").val();

    // Vérifie que la date sélectionné soit correcte pour le type de période sélectionné
    switch (per_type)
    {
        // Par semaine : date = lundi ou dimanche
        case "per_week":
            var jour = my_date.getDay() - FIRST_DAY_WEEK;
            if (jour < 0)
                jour += 7;
            my_date.setDate(my_date.getDate() - jour); // 1er jour de la semaine
            break;

            // Par mois : date = 1er jour du mois
        case "per_month":
            my_date.setDate(1); // 1er jour du mois
            break;

            // Par an : date = 1er jour de l'année
        case "per_year":
            my_date.setDate(1); // 1er jour du mois
            my_date.setMonth(0); // 1er mois de l'année (janvier = 0)
            break;
    }

    // Convertion de la date pour la requête sql (yyyy-mm-dd -> ISO 8601) et sauvegarde dans le champ caché 'per_date_sql'
    var day = my_date.getDate();
    if (day < 10)
        day = '0' + day;
    var month = my_date.getMonth();
    month += 1;
    if (month < 10)
        month = '0' + month;
    $("#per_date_sql").val(my_date.getFullYear() + '-' + month + '-' + day);

    // Enlève le focus (sinon pas d'affichage du calendrier au prochain click)
    //$("#per_date").blur();

    // Sauve la nouvelle date dans la session.       
    window.sessionStorage["stats_per_date"] = document.getElementById("per_date").value;

    // Rempli le champs de form. avec la nouvelle date
    $("#per_date").data("DateTimePicker")["date"](my_date);
    //$("#per_date").datepicker("setDate", my_date);
    //}
}

/***********************************
 Crée un dégradé de couleurs au format TSL (Teinte Saturation Lumière)
 ou HSL (Hue Saturation Light)
 ***********************************/
function getColorGradient(nb_colors)
{
    var i = 0;
    var hue_start = 0, hue_end = 120; // 0 120
    var light_start = 50, light_end = 50;
    var nb_hue = (hue_end - hue_start) / nb_colors;
    var nb_light = (light_end - light_start) / nb_colors;
    var aColors = '[';
    while (i < nb_colors)
    {
        if (i > 0)
            aColors += ',';
        aColors += '"hsl(' + parseInt((i * nb_hue) + hue_start) + ', 100%, ' + parseInt((i * nb_light) + light_start) + '%)"';
        i++;
    }
    aColors += ']';
    return (aColors)
}

/***********************************
 Retourne la longueur de la chaine de caractère
 la plus longue du tableau
 - aStr : Tableau contenant des chaines de caractères
 ***********************************/
function getMaxStrLength(aStr)
{
    var i = 0, iMax = 0, sMyStr;
    while (i < aStr.length)
    {
        if (aStr)
            sMyStr = aStr[i];
        if (sMyStr.length > iMax)
            iMax = sMyStr.length;
        i++;
    }
    return (iMax);
}

/***********************************
 Initialisation
 ***********************************/
function initStatistics() {
    window.sessionStorage["stats_per_date"] = null;
// Vide le message d'erreur et cache l'élément
    //cleanErrorMsg();

// Champ texte de la date non modifiable
    //document.getElementById("per_date").readonly = true;
    $('#per_date').attr('readonly', 'readonly');

// Paramètres du calendrier.        
    var date_format = document.getElementById("date_format").value;
    date_format = date_format.toLowerCase();
    date_format = date_format.replace('d', 'dd');
    date_format = date_format.replace('m', 'mm');
    date_format = date_format.replace('y', 'yyyy');

// 1er jour de la semaine.
    moment["locale"]($("#per_date").data("DateTimePicker")["locale"](), {
        week: {dow: FIRST_DAY_WEEK}
    });

// Si changement de date -> vérification suivant la période sélectionnée.
    $("#per_date").on("dp.change", function (e) {
        if (document.getElementById("per_date").value != window.sessionStorage["stats_per_date"])
            checkPeriodDate();
    });

// Si sélection d'un nouveau type de période : vérification de la date
    $("#per_type").change(function () {
        checkPeriodDate()
    });

// Si changement de sélection de la variable : plusieurs conditions à vérifier
    $('#variable_id').change(function ()
    {
        // Si 'nombre de traitements' est sélectionné ou non : désactive ou réactive le menu déroulant de l'unité
        if ($(this).val() == 'var_jn')
        {
            $("#unit_id").attr("disabled", "disabled");
            $("#unit_id").val('unit_sec');
        } else
            $("#unit_id").removeAttr("disabled");

        // Si 'durée des traitements unitaires' est sélectionné : légende = 'Projet FME' uniquement et nb max elements désactivé
        if ($(this).val() == 'var_ujl')
        {
            //$('input[type=radio][name="gb_id"][value="gb_workspace"]').click();
            $("#gb_id").val('gb_workspace');
            $("#max_elements").attr("disabled", "disabled");
        } else
            $("#max_elements").removeAttr("disabled");

        // Unité en pourcentage uniquement pour 'Durée totale de traitement'
        if ($('#unit_id').val() == 'unit_pc' && $(this).val() != 'var_tjl')
            $('#unit_id').val('unit_sec');
    });

// Si 'durée des traitements unitaires' est sélectionné : légende = 'Projet FME' uniquement
    //$('input[type=radio][name="gb_id"]').change(function()
    $("#gb_id").change(function ()
    {
        if ($('#variable_id').val() == 'var_ujl')
            $(this).val('gb_workspace'); //$(this)
        //$("#gb_id").val('gb_workspace');
        //$('input[type=radio][name="gb_id"][value="gb_workspace"]').click();
    });

// Unité en pourcentage uniquement pour 'Durée totale de traitement'
    $('#unit_id').change(function ()
    {
        if ($(this).val() == 'unit_pc' && $('#variable_id').val() != 'var_tjl')
            $(this).val('unit_sec');
    });

// Nb max d'éléments : si la valeur entrée est incorrecte : valeur par défaut
    $('#max_elements').change(function ()
    {
        // La valeur entrée est un nombre ?	
        if (isNaN($(this).val()))
            $(this).val(DEFAULT_MAX_ELEMENTS);
    });

// Nb caractères max des libellés : si la valeur entrée est incorrecte : valeur par défaut
    $('#max_labels_characters').change(function ()
    {
        // La valeur entrée est un nombre ?	
        if (isNaN($(this).val()))
            $(this).val(DEFAULT_MAX_LABELS_CHARACTERS);
        else if ($(this).val() == 0)
            $(this).val(DEFAULT_MAX_LABELS_CHARACTERS);
    });

// Bouton de lancement des stats	
    $('#btn_start_stats').click(function ()
    {
        /*
         // Affiche le gif animé de chargement de la requête ajax (et positionnement au milieu de l'écran)
         document.getElementById("gif_ajax").style.visibility = "visible";
         document.getElementById("gif_ajax").style.left = (parseInt($(document).width() / 2) + "px");
         document.getElementById("gif_ajax").style.top = (parseInt($(document).height() / 2) + "px");
         */
        // Sauve la période sélectionnée
        document.getElementById("per_type_save").value = document.getElementById("per_type").value;

        // Lance la requête et l'affichage des stats
        displayStatsRequest();

        // Cache le gif animé de chargement de la requête ajax
        //document.getElementById("gif_ajax").style.visibility = "hidden"; //hidden;
    });

// Initialise les valeurs par défaut ou simulation de click / changement
    $('#max_labels_characters').change();				// 'Nombre d'éléments max' par défaut
    checkPeriodDate(); 									// Initialise le champ date vide
    $("#first_day_week").val(FIRST_DAY_WEEK); 	// 1er jour de la semaine dans le champ caché (pour la classe php)
    /* $("#PHPSESSID").val(sSessId);	 */			// Session php
    /*	
     // Gif animé pendant le chargement des stats
     $("body").append('<div id="gif_ajax"><img src="' + AJAX_LOADING_IMG + '"></div>');
     document.getElementById("gif_ajax").style.left = (parseInt($(document).width() / 2) + "px");
     document.getElementById("gif_ajax").style.top = (parseInt($(document).height() / 2) + "px");
     */
// Lance une requête de stats à l'affichage de la page
    $('#btn_start_stats').click();
}