<?php
require_once("vmlib/logUtil.inc");

/**
 * \file FmeCloud.php
 * \brief Main class to use Fme Cloud services.
 *
 * \author Frederic Carretero <frederic.carretero@veremes.com>
 */

Class FmeCloud {
    public $sUrl;
    public $aLastCurlRequestInfo;
    public $sLogFilePath;   // Chemin optionnel vers le fichier de log pour Fme Cloud.
    private $sToken;
    
    /**
     * construct
     * @param {string} $sUrl Url of the fme server.
     * @param {string} $sToken Token of the FME Cloud API.
     */
    function __construct ($sUrl, $sToken) {
        $this->sUrl = $sUrl;
        $this->sToken = $sToken;    // Token de l'API FME Cloud.
    }
    
    /**
     * Curl request
     * @param {string} $sUrl Url of the Curl request.
     * @param {string} $sType Type of the Curl request (GET, POST).
     * @param {array} $aData Data of the Curl request.
     * @param {array} $aHeaders Headers of the Curl request.
     * @return Request result
     */
    // 
    function curlRequest($sUrl, $sType, $aData = array(), $aHeaders = array()) {
        //
        $this->aLastCurlRequestInfo = '';
        //
        $ch = curl_init();
        $sType = strtoupper($sType);
        // Force la méthode de requête utilisée (GET, POST...).
        curl_setopt($ch, CURLOPT_CUSTOMREQUEST, $sType);
        // Url à utiliser.
        if (($sType == 'GET' || $sType == 'DELETE') && !empty($aData))
            $sUrl .= '?' . http_build_query($aData);
        curl_setopt($ch, CURLOPT_URL, $sUrl);
        // Retour sous forme de texte. 
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        // Requête POST.
        if ($sType == 'POST') {
            curl_setopt($ch, CURLOPT_POST, true);
            // Chaîne de requête en encodage URL.
            if (is_array($aData))
                $aData = http_build_query($aData);
            // Données de la requête.
            curl_setopt($ch, CURLOPT_POSTFIELDS, $aData);
            //
            curl_setopt($ch, CURLOPT_SAFE_UPLOAD, true);
            // Entête pour la requête en POST.
            //$aHeaders[] = 'Content-Type: application/x-www-form-urlencoded';
        }
        // Entête pour la requête
        $aHeaders[] = 'Accept: application/json';
        if (!empty($this->sToken))
            $aHeaders[] = 'Authorization: Bearer ' . $this->sToken; // Token obligatoire pour éxécuter la requête.
        curl_setopt($ch, CURLOPT_HTTPHEADER, $aHeaders);
        // Durée max. de la requête.
        //curl_setopt($ch, CURLOPT_TIMEOUT, 120);
        // Exécute la session CURL.
        // Log.
        /*
        $handle = fopen("curl_log.txt", "a");
        fwrite($handle, PHP_EOL . '-----------------------' . PHP_EOL);
        curl_setopt($ch, CURLOPT_VERBOSE, true);
        curl_setopt($ch, CURLOPT_STDERR , $handle);
        */
        curl_setopt($ch, CURLINFO_HEADER_OUT, true);
        // Curl error: SSL certificate problem: unable to get local issuer certificate
        // Curl error n°60
        curl_setopt ($ch, CURLOPT_SSL_VERIFYPEER, false);      
        //
        $output = curl_exec($ch);
        // Erreur de la requête CURL.
        if(curl_errno($ch)) {
            writeToErrorLog('Curl error: ' . curl_error($ch));
            writeToErrorLog('Curl error n°' . curl_errno($ch));
        }
        // Informations de la requête.
        $aCurlInfo = curl_getinfo($ch);
        $this->aLastCurlRequestInfo = $aCurlInfo;
        //file_put_contents('C:/svn/gtf_cloud/gtf.engines/log/response.log', print_r($aCurlInfo, true));
        //writeToErrorLog(print_r(curl_getinfo($ch), true));
        
        // Ferme la session CURL.
        curl_close($ch);
        //
        //fclose($handle);
        //
        return $output;
    }
    
    /**
     * Send a request to the REST API.
     * @param {string} $sService Name of the service.
     * @param {string} $sMethod Method of the service (GET, POST...).
     * @param {array} $aData Data of the Curl request.
     * @return Request result
     */
    function serviceRequest($sService, $sMethod = 'GET', $aData = array()) {
        // Url vers le serveur fme.
        $sUrl = $this->sUrl;
        // Url vers le service d'information.
        $sUrl = $this->sUrl . '/v1/' . $sService;
        // Transfert cURL
        $sRequestResponse = $this->curlRequest($sUrl, $sMethod, $aData);
        if ($this->aLastCurlRequestInfo['http_code'] >= 400) {
            $oRequestResponse = json_decode($sRequestResponse);
            if (is_object($oRequestResponse) && !empty($oRequestResponse->exception) && !empty($oRequestResponse->exception->message))
                $this->writeToErrorLog('|' . $sService . '|' . ' ' . $oRequestResponse->message);
            else
                $this->writeToErrorLog('|' . $sService . '|' . ' ' . $sRequestResponse);
            return false;
        }
        else
            return json_decode($sRequestResponse);
    }
    
    /**
     * Start a FME Server Instance that is currently paused.
     * @param {integer} $iId Id of the Instance.
     * @return Request result
     */
    function startInstance($iId) {
        // Url vers le serveur fme.
        $sUrl = $this->sUrl;
        // Url vers le service d'information.
        $sUrl = $this->sUrl . '/v1/instances/' . $iId . '/start';
        // Transfert cURL
        $sRequestResponse = $this->curlRequest($sUrl, 'PUT');
        if ($this->aLastCurlRequestInfo['http_code'] >= 400) {
            $oRequestResponse = json_decode($sRequestResponse);
            if (is_object($oRequestResponse) && !empty($oRequestResponse->exception) && !empty($oRequestResponse->exception->message))
                $this->writeToErrorLog($oRequestResponse->exception->message);
            else
                $this->writeToErrorLog($sRequestResponse);
            return false;
        }
        else
            return json_decode($sRequestResponse);
    }
    
    /**
     * Pause a FME Server Instance that is currently running.
     * @param {integer} $iId Id of the Instance.
     * @return Request result
     */
    function pauseInstance($iId) {
        // Url vers le serveur fme.
        $sUrl = $this->sUrl;
        // Url vers le service d'information.
        $sUrl = $this->sUrl . '/v1/instances/' . $iId . '/pause';
        // Transfert cURL
        $sRequestResponse = $this->curlRequest($sUrl, 'PUT');
        if ($this->aLastCurlRequestInfo['http_code'] >= 400) {
            $oRequestResponse = json_decode($sRequestResponse);
            if (is_object($oRequestResponse) && !empty($oRequestResponse->exception) && !empty($oRequestResponse->exception->message))
                $this->writeToErrorLog($oRequestResponse->exception->message);
            else
                $this->writeToErrorLog($sRequestResponse);
            return false;
        }
        else
            return json_decode($sRequestResponse);
    }
    
    /**
     * Get the credentials for the instance.
     * @param {integer} $iId Id of the Instance.
     * @return Request result
     */
    function getCredentials($iId) {
        // Url vers le serveur fme.
        $sUrl = $this->sUrl;
        // Url vers le service d'information.
        $sUrl = $this->sUrl . '/v1/instances/' . $iId . '/credentials';
        // Transfert cURL
        $sRequestResponse = $this->curlRequest($sUrl, 'GET');
        if ($this->aLastCurlRequestInfo['http_code'] >= 400) {
            $oRequestResponse = json_decode($sRequestResponse);
            if (is_object($oRequestResponse) && !empty($oRequestResponse->exception) && !empty($oRequestResponse->exception->message))
                $this->writeToErrorLog($oRequestResponse->exception->message);
            else
                $this->writeToErrorLog($sRequestResponse);
            return false;
        }
        else
            return json_decode($sRequestResponse);
    }
    
    /**
     * Write a message to the error log file.
     * @param {string} $sMessage Message to write to the log file.
     */
    function writeToErrorLog($sMessage) {
        $aDebugBacktrace = debug_backtrace();
        $sLogMessage = '|ERROR|' . $aDebugBacktrace[1]['class'] . '::' . $aDebugBacktrace[1]['function'] . '| ' . $sMessage;
        if (empty($this->sLogFilePath))
            writeToErrorLog($sLogMessage);
        else
            writeToLog($sLogMessage, $this->sLogFilePath);
    }
}
?>