<?php

/**
 * \file imap_veremes.php
 * \brief class Imap pour se connecter à un serveur imap et lire les mails.
 *
 * \author Anthony Borghi <anthony.borghi@veremes.com>
 */

class Imap {

    /**
     * imap connection
     */
    protected $imap = false;
    
    /**
     * mailbox url string
     */
    protected $mailbox = "";
    
    /**
     * currentfolder
     */
    protected $folder = "Inbox";
    
    /**
     * initialize imap connection
     *
     * @return void
     * @param $mailbox imap_open string
     * @param $port
     * @param $username
     * @param $password
     * @param $encryption ssl or tls
     */
    public function __construct($mailbox, $port, $username, $password, $encryption = false) {
        $enc = '';
        $more = '';
        if($encryption!=null && isset($encryption) && $encryption=='ssl'){
            $enc = '/imap/ssl/novalidate-cert';
        } else if($encryption!=null && isset($encryption) && $encryption=='tls'){
            $enc = '/imap/tls/novalidate-cert';
        } else if($encryption!=null && isset($encryption) && $encryption=='gmail'){
            $enc = '/imap/ssl';
            $more = 'INBOX';
        }
        $this->mailbox = "{" . $mailbox . ":" . $port . $enc . "}";
        $this->imap = @imap_open($this->mailbox, $username, $password);
    }

    /**
     * returns true after successfull connection
     *
     * @return bool true on success
     */
    public function isConnected() {
        return $this->imap !== false;
    }
    
    
    /**
     * returns last imap error
     *
     * @return string error message
     */
    public function getLastError() {
        return imap_last_error();
    }

    /**
     * select given folder
     *
     * @return bool successfull opened folder
     * @param $folder name
     */
    public function selectFolder($folder) {
        $result = imap_reopen($this->imap, $this->mailbox . $folder);
        if($result === true)
            $this->folder = $folder;
        return $result;
    }
    
    
    /**
     * returns all available folders
     *
     * @return array with foldernames
     */
    public function getFolders() {
        $folders = imap_list($this->imap, $this->mailbox, "*");
        return str_replace($this->mailbox, "", $folders);
    }

    /**
     * returns the number of messages in the current folder
     *
     * @return int message count
     */
    public function countMessages() {
        return imap_num_msg($this->imap);
    }
    
    
    /**
     * returns the number of unread messages in the current folder
     *
     * @return int message count
     */
    public function countUnreadMessages() {
        $result = imap_search($this->imap, 'UNSEEN');
        if($result===false)
            return 0;
        return count($result);
    }


    /**
     * returns unseen emails in the current folder
     *
     * @return array messages
     * @param $withbody without body
     */
    public function getUnreadMessages($withbody=true){
        $emails = array();;
        $result = imap_search($this->imap, 'UNSEEN');
        if($result){
            foreach($result as $k=>$i){
                $emails[]= $this->formatMessage($i, $withbody);
            }
        }
        return $emails;
    }

    /**
     * returns all emails in the current folder
     *
     * @return array messages
     * @param $withbody without body
     */
    public function getMessages($withbody = true) {
        $count = $this->countMessages();
        $emails = array();
        for($i=1;$i<=$count;$i++) {
            $emails[]= $this->formatMessage($i, $withbody);
        }

        // sort emails descending by date
        // usort($emails, function($a, $b) {
        // try {
        // $datea = new \DateTime($a['date']);
        // $dateb = new \DateTime($b['date']);
        // } catch(\Exception $e) {
        // return 0;
        // }
        // if ($datea == $dateb)
        // return 0;
        // return $datea < $dateb ? 1 : -1;
        // });

        return $emails;
    }

    /**
     * returns email by given id
     *
     * @return array messages
     * @param $id
     * @param $withbody without body
     */
    public function getMessage($id, $withbody = true) {
        return $this->formatMessage($id, $withbody);
    }

    /**
     * @param $id
     * @param bool $withbody
     * @return array
     */
    protected function formatMessage($id, $withbody=true){
        $header = imap_headerinfo($this->imap, $id);

        // fetch unique uid
        $uid = imap_uid($this->imap, $id);
    
        // get email data
        $subject = '';
        if ( isset($header->subject) && strlen($header->subject) > 0 ) {
            foreach(imap_mime_header_decode($header->subject) as $obj){
                $subject .= $obj->text;
            }
        }
        $subject = $this->convertToUtf8($subject);
        $email = array(
            'to'       => isset($header->to) ? $this->arrayToAddress($header->to) : '',
            'from'     => $this->toAddress($header->from[0]),
            'date'     => $header->date,
            'subject'  => $subject,
            'uid'       => $uid,
            'unread'   => strlen(trim($header->Unseen))>0,
            'answered' => strlen(trim($header->Answered))>0,
            'deleted' => strlen(trim($header->Deleted))>0,
            'id' => $id
        );
        if(isset($header->cc))
            $email['cc'] = $this->arrayToAddress($header->cc);

        // get email body
        if($withbody===true) {
            $body = $this->getBody($uid);
            $email['body'] = $body['body'];
            $email['html'] = $body['html'];
        }

        // get attachments
        $mailStruct = imap_fetchstructure($this->imap, $id);
        //error_log(print_r($mailStruct, true));
        $attachments = $this->attachments2name($this->getAttachments($this->imap, $id, $mailStruct, ""));
        if(count($attachments)>0) {

            foreach ($attachments as $val) {
                foreach ($val as $k=>$t) {
                    if ($k == 'name') {
                        $decodedName = imap_mime_header_decode($t);
                        $t = $this->convertToUtf8($decodedName[0]->text);
                    }
                $arr[$k] = $t;
                }
            $email['attachments'][] = $arr;
            }
        }
        return $email;
    }

    public function getMailFullStructure($id, $withbody=true){
        $header = imap_headerinfo($this->imap, $id);

        // fetch unique uid
        $uid = imap_uid($this->imap, $id);
    
        // get email data
        $subject = '';
        if ( isset($header->subject) && strlen($header->subject) > 0 ) {
            foreach(imap_mime_header_decode($header->subject) as $obj){
                $subject .= $obj->text;
            }
        }
        $subject = $this->convertToUtf8($subject);
        $email = array(
            'to'       => isset($header->to) ? $this->arrayToAddress($header->to) : '',
            'from'     => $this->toAddress($header->from[0]),
            'date'     => $header->date,
            'subject'  => $subject,
            'uid'       => $uid,
            'unread'   => strlen(trim($header->Unseen))>0,
            'answered' => strlen(trim($header->Answered))>0,
            'deleted' => strlen(trim($header->Deleted))>0,
            'id' => $id
        );
        if(isset($header->cc))
            $email['cc'] = $this->arrayToAddress($header->cc);

        // get email body
        if($withbody===true) {
            $body = $this->getBody($uid);
            $email['body'] = $body['body'];
            $email['html'] = $body['html'];
        }

        $structure  = imap_fetchstructure($this->imap, $id);

        $attachments = array();
       if(isset($structure->parts) && count($structure->parts)) {
            //error_log("1");
         for($i = 0; $i < count($structure->parts); $i++) {
            //error_log("2");
            //error_log(print_r($structure->parts[$i], true));
           if (isset($structure->parts[$i]->disposition) && isset($structure->parts[$i]->type)) {
            if ((strtolower($structure->parts[$i]->disposition) == "attachment" || strtolower($structure->parts[$i]->disposition) == "inline") 
                && $structure->parts[$i]->type === 3) {
            //error_log("3");
                $attachment= array(
                    "is_attachment" => false,
                    "name"    => '',
                    //"partNum" => $partNum,
                    "enc"     => $structure->parts[$i]->encoding,
                    "size"    => $structure->parts[$i]->bytes,
                    "content" => ''
                );
                if($structure->parts[$i]->ifdparameters) {
                     foreach($structure->parts[$i]->dparameters as $object) {
                       if(strtolower($object->attribute) == 'filename') {
                         $attachment['is_attachment'] = true;
                         $attachment['filename'] = $object->value;
                       }
                     }
                   }

                if($structure->parts[$i]->ifparameters) {
                     foreach($structure->parts[$i]->parameters as $object) {
                       if(strtolower($object->attribute) == 'name') {
                         $attachment['is_attachment'] = true;
                         $attachment['name'] = $object->value;
                       }
                     }
                   }
                
                if($attachment['is_attachment']) {
                     $attachment['content'] = imap_fetchbody($this->imap, $id, $i+1);
                     if($structure->parts[$i]->encoding == 3) { // 3 = BASE64
                       $attachment['content'] = base64_decode($attachment['content']);
                     }
                     elseif($structure->parts[$i]->encoding == 4) { // 4 = QUOTED-PRINTABLE
                       $attachment['content'] = quoted_printable_decode($attachment['content']);
                     }
                     array_push($attachments, $attachment);
                   }             
                 } // for($i = 0; $i < count($structure->parts); $i++)
               } // if(isset($structure->parts) && count($structure->parts))
            }
        }

       $email['attachments'] = $attachments;


       return $email;
    }
    
    /**
     * delete given message
     *
     * @return bool success or not
     * @param $id of the message
     */
    public function deleteMessage($id) {
        return $this->deleteMessages(array($id));
    }
    
    
    /**
     * delete messages
     *
     * @return bool success or not
     * @param $ids array of ids
     */
    public function deleteMessages($ids) {
        if( imap_mail_move($this->imap, implode(",", $ids), $this->getTrash(), CP_UID) == false)
            return false;
        return imap_expunge($this->imap);
    }
    
    
    /**
     * move given message in new folder
     *
     * @return bool success or not
     * @param $id of the message
     * @param $target new folder
     */
    public function moveMessage($id, $target) {
        return $this->moveMessages(array($id), $target);
    }
    
    
    /**
     * move given message in new folder
     *
     * @return bool success or not
     * @param $ids array of message ids
     * @param $target new folder
     */
    public function moveMessages($ids, $target) {
        if(imap_mail_move($this->imap, implode(",", $ids), $target, CP_UID)===false)
            return false;
        return imap_expunge($this->imap);
    }
    
    
    /**
     * mark message as read
     *
     * @return bool success or not
     * @param $id of the message
     * @param $seen true = message is read, false = message is unread
     */
    public function setUnseenMessage($id) {
        return imap_setflag_full($this->imap, $id, "\\Seen \\Flagged", ST_UID);

    }
    
    
    /**
     * return content of messages attachment
     *
     * @return binary attachment
     * @param $id of the message
     * @param $index of the attachment (default: first attachment)
     */
    public function getAttachment($id, $index = 0) {
        //error_log(print_r("getAttachment called with " . $id . " and " . $index , true));
        // find message
        $attachments = false;
        $messageIndex = imap_msgno($this->imap, $id);
        $header = imap_headerinfo($this->imap, $messageIndex);
        $mailStruct = imap_fetchstructure($this->imap, $messageIndex);
        $attachments = $this->getAttachments($this->imap, $messageIndex, $mailStruct, "");

        // if attachments are ine the same part (inline disposition case)
        if(isset($attachments[0][0])){
            $attachments = $attachments[0];
        }

        if($attachments==false)
            return false;
        
        // find attachment
        if($index > count($attachments)){
            error_log(print_r("you are asking for index : " . $index . "but we have only " . count($attachments) . ' attachments', true));
            return false;
        }

        $attachment = $attachments[$index];

        //error_log("IMAP classe : getAttachment(integer, integer)");
        //error_log($attachment['name'] . " fait " . $attachment['size']);

        
        // get attachment body
        //$partStruct = imap_bodystruct($this->imap, imap_msgno($this->imap, $id), $attachment['partNum']);
        $filename = $attachment['name'];
        $message = imap_fetchbody($this->imap, $id, $attachment['partNum'], FT_UID);
        //error_log(print_r(imap_fetchstructure($this->imap, $id, FT_UID), true));
        //error_log($message);

        switch ($attachment['enc']) {
            case 0:
            case 1:
            //echo $message."\n";
                $message = $message;
                //echo $message."\n";
                break;
            case 2:
                $message = imap_binary($message);
                break;
            case 3:
                $message = imap_base64($message);
                break;
            case 4:
                $message = quoted_printable_decode($message);
                break;
        }
     
        return array(
                "name" => $attachment['name'], 
                "size" => $attachment['size'],
                "content" => $message);
    }
    
    
    /**
     * add new folder
     *
     * @return bool success or not
     * @param $name of the folder
     * @param $subscribe immediately subscribe to folder
     */
    public function addFolder($name, $subscribe = false) {
        $success = imap_createmailbox($this->imap, $this->mailbox . $name);

        if ($success && $subscribe) {
            $success = imap_subscribe($this->imap, $this->mailbox . $name);
        }

        return $success;
    }
    
    
    /**
     * remove folder
     *
     * @return bool success or not
     * @param $name of the folder
     */
    public function removeFolder($name) {
        return imap_deletemailbox($this->imap, $this->mailbox . $name);
    }
    
    
    /**
     * rename folder
     *
     * @return bool success or not
     * @param $name of the folder
     * @param $newname of the folder
     */
    public function renameFolder($name, $newname) {
        return imap_renamemailbox($this->imap, $this->mailbox . $name, $this->mailbox . $newname);
    }
    
    
    /**
     * clean folder content of selected folder
     *
     * @return bool success or not
     */
    public function purge() {
        // delete trash and spam
        if($this->folder==$this->getTrash() || strtolower($this->folder)=="spam") {
            if(imap_delete($this->imap,'1:*')===false) {
                return false;
            }
            return imap_expunge($this->imap);
        
        // move others to trash
        } else {
            if( imap_mail_move($this->imap,'1:*', $this->getTrash()) == false)
                return false;
            return imap_expunge($this->imap);
        }
    }
    
    
    /**
     * returns all email addresses
     *
     * @return array with all email addresses or false on error
     */
    public function getAllEmailAddresses() {
        $saveCurrentFolder = $this->folder;
        $emails = array();
        foreach($this->getFolders() as $folder) {
            $this->selectFolder($folder);
            foreach($this->getMessages(false) as $message) {
                $emails[] = $message['from'];
                $emails = array_merge($emails, $message['to']);
                if(isset($message['cc']))
                    $emails = array_merge($emails, $message['cc']);
            }
        }
        $this->selectFolder($saveCurrentFolder);
        return array_unique($emails);
    }
    
    
    /**
     * save email in sent
     *
     * @return void
     * @param $header
     * @param $body
     */
    public function saveMessageInSent($header, $body) {
        return imap_append($this->imap, $this->mailbox . $this->getSent(), $header . "\r\n" . $body . "\r\n", "\\Seen");
    }
    
    
    /**
     * explicitly close imap connection
     */
    public function close() {
        if($this->imap!==false)
            imap_close($this->imap);
    }
    
    
    
    // protected helpers
    
    
    /**
     * get trash folder name or create new trash folder
     *
     * @return trash folder name
     */
    protected function getTrash() {
        foreach($this->getFolders() as $folder) {
            if(strtolower($folder)==="trash" || strtolower($folder)==="papierkorb")
                return $folder;
        }
        
        // no trash folder found? create one
        $this->addFolder('Trash');
        
        return 'Trash';
    }
    
    
    /**
     * get sent folder name or create new sent folder
     *
     * @return sent folder name
     */
    protected function getSent() {
        foreach($this->getFolders() as $folder) {
            if(strtolower($folder)==="sent" || strtolower($folder)==="gesendet")
                return $folder;
        }
        
        // no sent folder found? create one
        $this->addFolder('Sent');
        
        return 'Sent';
    }
    
    
    /**
     * fetch message by id
     *
     * @return header
     * @param $id of the message
     */
    protected function getMessageHeader($id) {
        $count = $this->countMessages();
        for($i=1;$i<=$count;$i++) {
            $uid = imap_uid($this->imap, $i);
            if($uid==$id) {
                $header = imap_headerinfo($this->imap, $i);
                return $header;
            }
        }
        return false;
    }
    
    
    /**
     * convert attachment in array(name => ..., size => ...).
     *
     * @return array
     * @param $attachments with name and size
     */
    protected function attachments2name($attachments) {
        $names = array();
        if(isset($attachments[0][0])){
            $attachments = $attachments[0];
        }
        foreach($attachments as $attachment) {
            $names[] = array(
                'name' => $attachment['name'],
                'size' => $attachment['size']
            );
        }
        return $names;
    }
    
    
    /**
     * convert imap given address in string
     *
     * @return string in format "Name <email@bla.de>"
     * @param $headerinfos the infos given by imap
     */
    protected function toAddress($headerinfos) {
        $email = "";
        $name = "";
        if(isset($headerinfos->mailbox) && isset($headerinfos->host)) {
            $email = $headerinfos->mailbox . "@" . $headerinfos->host;
        }

        if(!empty($headerinfos->personal)) {
            $name = imap_mime_header_decode($headerinfos->personal);
            $name = $name[0]->text;
        } else {
            $name = $email;
        }
        
        $name = $this->convertToUtf8($name);
        
        return $name . " <" . $email . ">";
    }

    
    /**
     * converts imap given array of addresses in strings
     *
     * @return array with strings (e.g. ["Name <email@bla.de>", "Name2 <email2@bla.de>"]
     * @param $addresses imap given addresses as array
     */
    protected function arrayToAddress($addresses) {
        $addressesAsString = array();
        foreach($addresses as $address) {
            $addressesAsString[] = $this->toAddress($address);
        }
        return $addressesAsString;
    }

    
    /**
     * returns body of the email. First search for html version of the email, then the plain part.
     *
     * @return string email body
     * @param $uid message id
     */
    protected function getBody($uid) {
        $body = $this->get_part($this->imap, $uid, "TEXT/HTML");
        $html = true;
        // if HTML body is empty, try getting text body
        if ($body == "") {
            $body = $this->get_part($this->imap, $uid, "TEXT/PLAIN");
            $html = false;
        }
        $body = $this->convertToUtf8($body);
        return array( 'body' => $body, 'html' => $html);
    }
    
    
    /**
     * convert to utf8 if necessary.
     *
     * @return true or false
     * @param $string utf8 encoded string
     */
    function convertToUtf8($str) { 
        if(mb_detect_encoding($str, "UTF-8, ISO-8859-1, GBK")!="UTF-8")
            $str = utf8_encode($str);
        $str = iconv('UTF-8', 'UTF-8//IGNORE', $str);
        return $str; 
    } 
    
    
    /**
     * returns a part with a given mimetype
     * taken from http://www.sitepoint.com/exploring-phps-imap-library-2/
     *
     * @return string email body
     * @param $imap imap stream
     * @param $uid message id
     * @param $mimetype
     */
    protected function get_part($imap, $uid, $mimetype, $structure = false, $partNumber = false) {
        if (!$structure) {
               $structure = imap_fetchstructure($imap, $uid, FT_UID);
        }
        if ($structure) {
            if ($mimetype == $this->get_mime_type($structure)) {
                if (!$partNumber) {
                    $partNumber = 1;
                }
                $text = imap_fetchbody($imap, $uid, $partNumber, FT_UID | FT_PEEK);
                switch ($structure->encoding) {
                    case 3: return imap_base64($text);
                    case 4: return imap_qprint($text);
                    default: return $text;
               }
           }
     
            // multipart 
            if ($structure->type == 1) {
                foreach ($structure->parts as $index => $subStruct) {
                    $prefix = "";
                    if ($partNumber) {
                        $prefix = $partNumber . ".";
                    }
                    $data = $this->get_part($imap, $uid, $mimetype, $subStruct, $prefix . ($index + 1));
                    if ($data) {
                        return $data;
                    }
                }
            }
        }
        return false;
    }
    
    
    /**
     * extract mimetype
     * taken from http://www.sitepoint.com/exploring-phps-imap-library-2/
     *
     * @return string mimetype
     * @param $structure
     */
    protected function get_mime_type($structure) {
        $primaryMimetype = array("TEXT", "MULTIPART", "MESSAGE", "APPLICATION", "AUDIO", "IMAGE", "VIDEO", "OTHER");
     
        if ($structure->subtype) {
           return $primaryMimetype[(int)$structure->type] . "/" . $structure->subtype;
        }
        return "TEXT/PLAIN";
    }
    
    
    /**
     * get attachments of given email
     * taken from http://www.sitepoint.com/exploring-phps-imap-library-2/
     *
     * @return array of attachments
     * @param $imap stream
     * @param $mailNum email
     * @param $part
     * @param $partNum
     */
    protected function getAttachments($imap, $mailNum, $part, $partNum) {
        $attachments = array();
        //error_log(print_r($part,true));
        if (isset($part->parts)) {
            foreach ($part->parts as $key => $subpart) {
                if($partNum != "") {
                    $newPartNum = $partNum . "." . ($key + 1);
                } else {
                    $newPartNum = ($key+1);
                }
                $result = $this->getAttachments($imap, $mailNum, $subpart, $newPartNum);
                //error_log(print_r($result, true));
                if (count($result) != 0) {
                    array_push($attachments, $result);
                }
            }
        } else if (isset($part->disposition) /*&& isset($part->type)*/) {
            if ((strtolower($part->disposition) == "attachment") /*&& $part->type === 3*/) {
                $partStruct = imap_bodystruct($imap, $mailNum, $partNum);

                $partStruct = json_decode(json_encode($partStruct), True);

                //error_log(print_r(gettype($partStruct["ifparameters"]), true));

                $sFileName = false;
                if($partStruct["ifparameters"]){
                    for ($iParameters = 0 ; $iParameters < count($partStruct["parameters"]);$iParameters++){
                        if($partStruct["parameters"][$iParameters]["attribute"] === "NAME"){
                            $sFileName = $partStruct["parameters"][$iParameters]["value"];
                        }
                    }
                }

                if($partStruct["ifdparameters"]){
                    for ($iDParameters = 0 ; $iDParameters < count($partStruct["dparameters"]);$iDParameters++){
                        if($partStruct["dparameters"][$iDParameters]["attribute"] === "FILENAME"){
                            $sFileName = $partStruct["dparameters"][$iDParameters]["value"];
                        }
                    }
                }

                $aFileName = imap_mime_header_decode($sFileName);
                $sFileName = $aFileName[0]->text;

                //$sMessage = imap_fetchbody($imap, $mailNum, $partNum);
                //error_log(print_r(base64_decode($sMessage), true));
                $attachmentDetails = array(
                    "name"    => $sFileName,
                    "partNum" => $partNum,
                    "enc"     => $partStruct["encoding"],
                    "size"    => $partStruct["bytes"]
                );
                //error_log(print_r($attachmentDetails,true));

                return $attachmentDetails;
            } else if (strtolower($part->disposition) == "inline"){
                $partStruct = imap_bodystruct($imap, $mailNum, $partNum);

                $partStruct = json_decode(json_encode($partStruct), True);

              
                if($partStruct["ifparameters"] || $partStruct["ifdparameters"]){
                    $sFileName = false;
                    if($partStruct["ifparameters"]){
                        for ($iParameters = 0 ; $iParameters < count($partStruct["parameters"]);$iParameters++){
                            if($partStruct["parameters"][$iParameters]["attribute"] === "NAME"){
                                $sFileName = $partStruct["parameters"][$iParameters]["value"];
                            }
                        }
                    }

                    if($partStruct["ifdparameters"]){
                        for ($iDParameters = 0 ; $iDParameters < count($partStruct["dparameters"]);$iDParameters++){
                            if($partStruct["dparameters"][$iDParameters]["attribute"] === "FILENAME"){
                                $sFileName = $partStruct["dparameters"][$iDParameters]["value"];
                            }
                        }
                    }

                    $aFileName = imap_mime_header_decode($sFileName);
                    $sFileName = $aFileName[0]->text;

                    //$sMessage = imap_fetchbody($imap, $mailNum, $partNum);
                    //error_log(print_r(base64_decode($sMessage), true));
                    $attachmentDetails = array(
                        "name"    => $sFileName,
                        "partNum" => $partNum,
                        "enc"     => $partStruct["encoding"],
                        "size"    => $partStruct["bytes"]
                    );
                    //error_log(print_r($attachmentDetails,true));

                    return $attachmentDetails;
                }
            }
        }
        /*if(isset($attachments[0])) {
            $attachments = $attachments[0];
        }*/
        //error_log(print_r($attachments, true));
        return $attachments;
    }
    
    /**
     * Return general mailbox statistics
     *
     * @return bool | StdClass object
     */
    public function getMailboxStatistics() {
        return $this->isConnected() ? imap_mailboxmsginfo($this->imap) : false ;
    }

}

?>
