<?php

/**
 * \file imap_veremes.php
 * \brief class Imap pour se connecter à un serveur imap et lire les mails.
 *
 * \author Anthony Borghi <anthony.borghi@veremes.com>
 */

class ImapConnection {

    /**
     * imap connection
     */
    public $imap= false;
    
    /**
     * mailbox url string
     */
    protected $mailbox = "";
    
    /**
     * currentfolder
     */
    protected $folder = "Inbox";

	/**
     * initialize imap connection
     *
     * @return void
     * @param $mailbox imap_open string
     * @param $port
     * @param $username
     * @param $password
     * @param $encryption ssl or tls
     */
	public function __construct($mailbox, $port, $username, $password, $encryption = false) {
        $enc = '';
        $more = '';
        if($encryption!=null && isset($encryption) && $encryption=='ssl'){
            $enc = '/imap/ssl/novalidate-cert';
        } else if($encryption!=null && isset($encryption) && $encryption=='tls'){
            $enc = '/imap/tls/novalidate-cert';
        } else if($encryption!=null && isset($encryption) && $encryption=='gmail'){
            $enc = '/imap/ssl';
            $more = 'INBOX';
        }
        $this->mailbox = "{" . $mailbox . ":" . $port . $enc . "}";
        $this->imap = @imap_open($this->mailbox, $username, $password);
    }

    /**
     * returns true after successfull connection
     *
     * @return bool true on success
     */
    public function isConnected() {
        return $this->imap !== false;
    }

    /**
     * returns last imap error
     *
     * @return string error message
     */
    public function getLastError() {
        return imap_last_error();
    }

    /**
     * select given folder
     *
     * @return bool successfull opened folder
     * @param $sFolder name of the folder to select
     */
    public function selectFolder($sFolder) {
        $result = imap_reopen($this->imap, $this->mailbox . $sFolder);
        if($result === true)
            $this->folder = $sFolder;
        return $result;
    }
    
    
    /**
     * returns all available folders
     *
     * @return array with foldernames
     */
    public function getFolders() {
        $folders = imap_list($this->imap, $this->mailbox, "*");
        return str_replace($this->mailbox, "", $folders);
    }

    /**
     * returns the number of messages in the current folder
     *
     * @return int message count
     */
    public function countMessages() {
        return imap_num_msg($this->imap);
    }

    /**
     * returns the number of messages with the flag in parameter in the current folder
     *
     * @return int message count
     * @param $sFlag Flag to filter mail (ex: UNSEEN)
     */
    public function countFlagedMessages($sFlag) {
        $result = imap_search($this->imap, $sFlag);
        if($result === false)
            return 0;
        return count($result);
    }

    /**
     * returns all emails in the current folder
     *
     * @return array of uid messages
     */
    public function getMessagesUidList() {
    	$iCount = $this->countMessages();
    	$aEmails = array();

    	for($i=1 ; $i <= $iCount ; $i++) {
    		// fetch unique uid
        	$uid = imap_uid($this->imap, $i);
    		array_push($aEmails, $uid);
        }

        return $aEmails;
    }

    /**
     * returns email by given uid
     *
     * @return object messages
     * @param $id universal id in the mailBox
     */
    public function getMessage($uid) {
    	return imap_fetchstructure($this->imap, $uid, FT_UID);
    }

    /**
     * returns email by given uid
     *
     * @return object messages
     * @param $id universal id in the mailBox
     */
    /*public function getMessageHeader($uid) {
    	return imap_fetch_overview ($this->imap, $uid, FT_UID);
    }*/

    /**
     * delete given message
     *
     * @return bool success or not
     * @param $id of the message
     */
    public function deleteMessage($id) {
        return $this->deleteMessages(array($id));
    }
    
    
    /**
     * delete messages
     *
     * @return bool success or not
     * @param $ids array of ids separated by ,
     */
    public function deleteMessages($ids) {
        if( imap_mail_move($this->imap, implode(",", $ids), $this->getTrash(), CP_UID) == false)
            return false;
        return imap_expunge($this->imap);
    }

    /**
     * move given message in new folder
     *
     * @return bool success or not
     * @param $id of the message
     * @param $target new folder
     */
    public function moveMessage($id, $target) {
        return $this->moveMessages(array($id), $target);
    }
    
    
    /**
     * move given message in new folder
     *
     * @return bool success or not
     * @param $ids array of message ids separated by ,
     * @param $target new folder
     */
    public function moveMessages($ids, $target) {
        if(imap_mail_move($this->imap, implode(",", $ids), $target, CP_UID)===false)
            return false;
        return imap_expunge($this->imap);
    }

    /**
     * mark message as read
     *
     * @return bool success or not
     * @param $id of the message
     * @param $seen true = message is read, false = message is unread
     */
    public function setUnseenMessage($id) {
        return imap_setflag_full($this->imap, $id, "\\Seen \\Flagged", ST_UID);

    }

     /**
     * add new folder
     *
     * @return bool success or not
     * @param $name of the folder
     * @param $subscribe immediately subscribe to folder
     */
    public function addFolder($name, $subscribe = false) {
        $success = imap_createmailbox($this->imap, $this->mailbox . $name);

        if ($success && $subscribe) {
            $success = imap_subscribe($this->imap, $this->mailbox . $name);
        }

        return $success;
    }

    /**
     * add new folder if not exist
     *
     * @return bool folder created (False if error or if already exist)
     * @param $sFolderName of the folder
     * @param $subscribe immediately subscribe to folder
     */
    public function addFolderIfNotExists ($sFolderName, $subscribe = false){
    	$aBoxList = $this->getFolders();

		if(!in_array($sFolderName, $aBoxList)){
			return $this->addFolder($sFolderName, $subscribe); //return boolean
		}

		return false;
    }
    
    
    /**
     * remove folder
     *
     * @return bool success or not
     * @param $name of the folder
     */
    public function removeFolder($name) {
        return imap_deletemailbox($this->imap, $this->mailbox . $name);
    }
    
    
    /**
     * rename folder
     *
     * @return bool success or not
     * @param $name of the folder
     * @param $newname of the folder
     */
    public function renameFolder($name, $newname) {
        return imap_renamemailbox($this->imap, $this->mailbox . $name, $this->mailbox . $newname);
    }

    /**
     * clean folder content of selected folder
     *
     * @return bool success or not
     */
    public function purge() {
        // delete trash and spam
        if($this->folder==$this->getTrash() || strtolower($this->folder)=="spam") {
            if(imap_delete($this->imap,'1:*')===false) {
                return false;
            }
            return imap_expunge($this->imap);
        
        // move others to trash
        } else {
            if( imap_mail_move($this->imap,'1:*', $this->getTrash()) == false)
                return false;
            return imap_expunge($this->imap);
        }
    }

    /**
     * save email in sent
     *
     * @return void
     * @param $header
     * @param $body
     */
    public function saveMessageInSent($header, $body) {
        return imap_append($this->imap, $this->mailbox . $this->getSent(), $header . "\r\n" . $body . "\r\n", "\\Seen");
    }
    
    
    /**
     * explicitly close imap connection
     */
    public function close() {
        if($this->imap!==false)
            imap_close($this->imap);
    }

    /**
     * get trash folder name or create new trash folder
     *
     * @return trash folder name
     */
    protected function getTrash() {
        foreach($this->getFolders() as $folder) {
            if(strtolower($folder)==="trash" || strtolower($folder)==="papierkorb")
                return $folder;
        }
        
        // no trash folder found? create one
        $this->addFolder('Trash');
        
        return 'Trash';
    }
    
    
    /**
     * get sent folder name or create new sent folder
     *
     * @return sent folder name
     */
    protected function getSent() {
        foreach($this->getFolders() as $folder) {
            if(strtolower($folder)==="sent" || strtolower($folder)==="gesendet")
                return $folder;
        }
        
        // no sent folder found? create one
        $this->addFolder('Sent');
        
        return 'Sent';
    }
    
    
    /**
     * fetch message by id
     *
     * @return header
     * @param $id of the message
     */
    public function getMessageHeader($id) {
        $count = $this->countMessages();
        for($i=1;$i<=$count;$i++) {
            $uid = imap_uid($this->imap, $i);
            if($uid==$id) {
                $header = imap_headerinfo($this->imap, $i);
                return $header;
            }
        }
        return false;
    }

    /**
     * Return general mailbox statistics
     *
     * @return bool | StdClass object
     */
    public function getMailboxStatistics() {
        return $this->isConnected() ? imap_mailboxmsginfo($this->imap) : false ;
    }

    /**
     * returns all the mail in a string. First search for html version of the email, then the plain part.
     *
     * @return string email body
     * @param $iUid message id
     */
    public function getBodyString($iUid) {
    	return imap_body ($this->imap, $iUid, FT_UID);
    }

    /**
     * returns all parts at same level of an email.
     *
     * @return array email parts
     * @param $uid message id
     */
    public function gerPartArray($iUid){
        $oStructure = imap_fetchstructure($this->imap, $iUid, FT_UID);
        $aPart = $this->createPartArray($oStructure);

        return $aPart;
    }

    /**
     * create the parts array from a mail structure.
     *
     * @return array email parts
     * @param $oStructure mail structure content of imap_fetchstructure
     * @param $sPrefix Constant for the PartId 
     */
	protected function createPartArray($oStructure, $sPrefix="") {
	    //print_r($structure);
	    if (sizeof($oStructure->parts) > 0) {    // There some sub parts
	        foreach ($oStructure->parts as $count => $part) {
	            $this->addPartToArray($part, $sPrefix.($count+1), $aPart);
	        }
	    }else{    // Email does not have a seperate mime attachment for text
	        $aPart[] = array('part_number' => $sPrefix.'1', 'part_object' => $oStructure);
	    }
	   return $aPart;
	}

    /**
     * Recursive treatment of substruct to create the part array. Called only if the mail is in multi Part. fill the array in parameter.
     * Sub function for create_part_array(). Only called by create_part_array() and itself. 
     *
     * @param $oStructure mail structure content of imap_fetchstructure
     * @param $sPrefix Constant for the PartId 
     */
	protected function addPartToArray($oStructure, $sPartNumber, &$aPart) {
	    $aPart[] = array('part_number' => $sPartNumber, 'part_object' => $oStructure);
	    if ($oStructure->type == 2) { // Check to see if the part is an attached email message, as in the RFC-822 type
	        //print_r($oStructure);
	        if (sizeof($oStructure->parts) > 0) {    // Check to see if the email has parts
	            foreach ($oStructure->parts as $count => $part) {
	                // Iterate here again to compensate for the broken way that imap_fetchbody() handles attachments
	                if (sizeof($part->parts) > 0) {
	                    foreach ($part->parts as $count2 => $part2) {
	                        $this->addPartToArray($part2, $sPartNumber.".".($count2+1), $aPart);
	                    }
	                }else{    // Attached email does not have a seperate mime attachment for text
	                    $aPart[] = array('part_number' => $sPartNumber.'.'.($count+1), 'part_object' => $oStructure);
	                }
	            }
	        }else{    // Not sure if this is possible
	            $aPart[] = array('part_number' => $prefix.'.1', 'part_object' => $oStructure);
	        }
	    }else{    // If there are more sub-parts, expand them out.
            //error_log(print_r($oStructure,true));
            if(property_exists($oStructure, "parts")){
    	        if (sizeof($oStructure->parts) > 0) {
    	            foreach ($oStructure->parts as $count => $p) {
    	                $this->addPartToArray($p, $sPartNumber.".".($count+1), $aPart);
    	            }
    	        }
            }
	    }
	}

    /**
     * returns a part with a given mimetype
     *
     * @return string email body False if no structure parameter
     * @param $uid mail universal id to select a part
     * @param $iStructureEncoding encoding of the part to decode it
     * @param $partNumber part id to get the part form the mail
     */
    public function getPart($uid, $iStructureEncoding, $partNumber) {       
        $sContent = imap_fetchbody($this->imap, $uid, $partNumber, FT_UID | FT_PEEK);
        switch ($iStructureEncoding) {
            case 3: 
                return imap_base64($sContent);
            case 4: 
                return imap_qprint($sContent);
            default: 
                error_log("Check encodage " . $iStructureEncoding);
                return $sContent;
        }
    }

    /**
     * extract mimetype
     * taken from http://www.sitepoint.com/exploring-phps-imap-library-2/
     *
     * @return string mimetype
     * @param $structure
     */
    public function getMimeType($structure) {
        $primaryMimetype = array("TEXT", "MULTIPART", "MESSAGE", "APPLICATION", "AUDIO", "IMAGE", "VIDEO", "OTHER");
     
        if ($structure->subtype) {
           return $primaryMimetype[(int)$structure->type] . "/" . $structure->subtype;
        }
        return "TEXT/PLAIN";
    }

    /**
     * Returns formatted string for imap_search 
     * @return array uid
     * @param $aFilter
     */
    public function buildSearchString($aFilter){
        $sFilter = "";
        $aFilterKey = array(
            //"ALL" => array("type"=>"boolean"),
            "ANSWERED" => array("type"=>"boolean"),
            "BCC" => array("type"=>"string"),
            "BEFORE" => array("type"=>"date"),
            "BODY" => array("type"=>"string"),
            "CC" => array("type"=>"string"),
            "DELETED" => array("type"=>"boolean"),
            "FLAGGED" => array("type"=>"boolean"),
            "FROM" => array("type"=>"string"),
            "KEYWORD" => array("type"=>"string"),
            "NEW" => array("type"=>"boolean"),
            "OLD" => array("type"=>"boolean"),
            "ON" => array("type"=>"date"),
            "RECENT" => array("type"=>"boolean"),
            "SEEN" => array("type"=>"boolean"),
            "SINCE" => array("type"=>"date"),
            "SUBJECT" => array("type"=>"string"),
            "TEXT" => array("type"=>"string"),
            "TO" => array("type"=>"string"),
            "UNANSWERED" => array("type"=>"boolean"),
            "UNDELETED" => array("type"=>"boolean"),
            "UNFLAGGED" => array("type"=>"boolean"),
            "UNKEYWORD" => array("type"=>"string"),
            "UNSEEN" => array("type"=>"boolean")
        );

        foreach ($aFilter as $sKey => $mValue) {
            if(isset($aFilterKey[strtoupper($sKey)]) && !empty($mValue)){
                switch ($aFilterKey[strtoupper($sKey)]["type"]) {
                    case 'boolean':
                        if($mValue == true){
                            $sFilter .= strtoupper($sKey);
                            $sFilter .= " ";
                        }
                        break;
                    case 'string':
                        if(strpos($mValue, '"') > 0){
                            // cette chaine ne peut pas être chercher
                        } else {
                            $sFilter .= strtoupper($sKey);
                            $sFilter .= " ";
                            $sFilter .= "\"" . $mValue . "\"";
                            $sFilter .= " ";
                        }
                        break;
                    case 'date':
                        $sFilter .= strtoupper($sKey);
                        $sFilter .= " ";
                        $sFilter .= "\"" . date("j F Y", strtotime($mValue)) . "\"";
                        $sFilter .= " ";
                        break;
                }
            } else {
                // this key doesn't exist in filter
            }
        }
        return substr($sFilter, 0, -1);
    }


    /**
     * Returns array of uid for this search object
     * @return array uid
     * @param $aFilter array of filter 
     */
    public function searchInFolder($aFilter){
        $sFilter = $this->buildSearchString($aFilter);
        return imap_search($this->imap, $sFilter, SE_UID);
    }
}


?>