<?php

/**
 * \file imap_veremes.php
 * \brief class Imap pour se connecter à un serveur imap et lire les mails.
 *
 * \author Anthony Borghi <anthony.borghi@veremes.com>
 */

class MailStruct {

    /**
     * imap connection
     */
    protected $oConnection = false;
    
    /**
     * Sender name and address of this mail
     */
    public $aSender = array();

    /**
     * Receiver name and address of this mail
     */
    public $aReceiver = array();

    /**
     * Copy name and address of this mail (non testé)
     */
    public $aCC = array();

    /**
     * Objet DateTime sur la date de réception
     */
    protected $tReceptionDate = 0;

    /**
     * Objet DateTime sur la date d'envoi'
     */
    protected $tSendDate = 0;

    /**
     * Uid of the mail
     */
    public $iUid = 0;

    /**
     * Subject of the mail
     */
    public $sSubject = "";

    /**
     * Body of this mail
     */
    public $sBody = "";

    /**
     * Body of this mail is using HTML
     */
    public $bIsHTML = false;

    /**
     * Attachments list of this mail
     */
    public $aAttachments = array();

    /**
     * Status of this mail (si vide =false)
     */
    public $aStatus = array("unread"=>false, "answered"=>false, "deleted"=>false);

	/**
     * initialize mail object structure
     *
     * @return void
     * @param $oImapConnection an imap connection with a folder selected
     * @param $iUid Universal ID of the mail to get informations
     */
	public function __construct($oImapConnection, $iUid) {
        $this->oConnection = $oImapConnection;
        $this->iUid = $iUid;

        //Get Mail Info
        $oHeader = $this->oConnection->getMessageHeader($this->iUid);
        $this->headerExtractor($oHeader);

        //Get Mail Body
        $aPart = $this->oConnection->gerPartArray($this->iUid);
        $this->bodyExtractor($aPart);
        //get Mail Attachment
        $this->attachmentsExtractor($aPart);
        //error_log(print_r($this, true));
    }

    /**
     * Extract Body of the mail header and it type (HTML or PLAIN).
     * @param $aPart array of parts of this mail
     */
    protected function bodyExtractor($aPart){
        foreach ($aPart as $oPart) {
           if ($oPart["part_object"]->encoding == 3 || $oPart["part_object"]->encoding == 4) {
                $sMimeType = $this->oConnection->getMimeType($oPart["part_object"]);

                //$this->sBody = $this->oConnection->getPart($this->iUid, $oPart["part_object"]->encoding, $oPart["part_number"]);

               if ($sMimeType === "TEXT/HTML") {
                    //error_log(print_r($oPart, true));
                    $this->sBody = $this->convertToUtf8($this->oConnection->getPart($this->iUid, $oPart["part_object"]->encoding, $oPart["part_number"]));
                   $this->bIsHTML = true;
               } else if($sMimeType === "TEXT/PLAIN"){
                    //error_log(print_r($oPart, true));
                    $this->sBody = $this->convertToUtf8($this->oConnection->getPart($this->iUid, $oPart["part_object"]->encoding, $oPart["part_number"]));
                    $this->bIsHTML = false;
               }
               //error_log($this->sBody);
               //error_log(mb_detect_encoding($this->sBody));
            } 
        }
    }

    /**
     * Extract Body of the mail header and it type (HTML or PLAIN).
     * @param $aPart array of parts of this mail
     */
    protected function attachmentsExtractor($aPart, $bTakeInlinePicture = false){
        $aKeyFileName = array("NAME", "FILENAME");

        foreach ($aPart as $oPart) {
           if (isset($oPart["part_object"]->disposition)){
                $sMimeType = $this->oConnection->getMimeType($oPart["part_object"]);
                if($oPart["part_object"]->ifdparameters == true || $oPart["part_object"]->ifparameters == true){
                    $sFileName = false;
                    if($oPart["part_object"]->ifparameters){
                        for ($iParameters = 0; $iParameters < count($oPart["part_object"]->parameters); $iParameters++){
                            if(in_array($oPart["part_object"]->parameters[$iParameters]->attribute, $aKeyFileName)){
                                $sFileName = $oPart["part_object"]->parameters[$iParameters]->value;
                            }
                        }
                    }

                    if($oPart["part_object"]->ifdparameters){
                        for ($iDParameters = 0 ; $iDParameters < count($oPart["part_object"]->dparameters);$iDParameters++){
                            if(in_array($oPart["part_object"]->dparameters[$iDParameters]->attribute, $aKeyFileName)){
                                $sFileName = $oPart["part_object"]->dparameters[$iDParameters]->value;
                            }
                        }
                    }

                    $aFileName = imap_mime_header_decode($sFileName);
                    $sFileName = $aFileName[0]->text;

                    //$sContent = $this->oConnection->getPart($this->iUid, $oPart["part_object"]->encoding, $oPart["part_number"]);
                
                    $aAttachment = array("name" => $sFileName, "size" => $oPart["part_object"]->bytes, "mime" => $sMimeType, "isFile" => true, "part_number" => $oPart["part_number"], "encoding" => $oPart["part_object"]->encoding);

                    array_push($this->aAttachments, $aAttachment);
                } else if ($bTakeInlinePicture === true){
                    // to take all the attachments of the mail as pictures inline ...
                    $sMime = str_replace("/", "_", $sMimeType);
                    $sFileName = $sMime . "_" . $oPart["part_number"] . ".txt";
                    
                    //$sContent = $this->oConnection->getPart($this->iUid, $oPart["part_object"]->encoding, $oPart["part_number"]);
                
                    $aAttachment = array("name" => $sFileName, "size" => $oPart["part_object"]->bytes, "mime" => $sMimeType, "isFile" => false, "part_number" => $oPart["part_number"], "encoding" => $oPart["part_object"]->encoding);

                    array_push($this->aAttachments, $aAttachment);
                }

                
           }
        }
    }

    /**
     * Return the content of an attachment
     * @return string content of the attachment
     * @param $iaAttachment attachment struct to get it content
     */
    public function getAttachmentContent($aAttachment){
        return $this->oConnection->getPart($this->iUid, $aAttachment["encoding"], $aAttachment["part_number"]);
    }

    /**
     * Extract MetaData of the mail header.
     */
    protected function headerExtractor($oHeader){
        // remover date mail
        $aRemover = array(" (Paris, Madrid (heure d’été))", " (Paris, Madrid)", " (Paris, Madrid (heure d'été))");
        // get email data
        $this->sSubject = $this->getSubjectConverted($oHeader);
        $this->aReceiver = $this->toAddressArray($oHeader->to[0]);
        $this->aSender = $this->toAddressArray($oHeader->from[0]);

        $sDateReception = $oHeader->MailDate;
        $sDateEnvoi = $oHeader->date;
        for($i = 0; $i < count($aRemover); $i++){
            $sDateReception = str_replace($aRemover[$i], "", $sDateReception);
            $sDateEnvoi = str_replace($aRemover[$i], "", $sDateEnvoi);
        }

        $this->tReceptionDate = new DateTime($sDateReception);
        $this->tSendDate = new DateTime($sDateEnvoi);

        $this->aStatus = array(
            "unread" => strlen(trim($oHeader->Unseen))>0, 
            "answered" => strlen(trim($oHeader->Answered))>0, 
            "deleted" => strlen(trim($oHeader->Deleted))>0
        );

        if(isset($oHeader->cc)){
            //a finir
            //error_log(print_r($this->oHeader->cc, true));
            $aCC = $this->toAddressArray($oHeader->cc);
        }
    }

    protected function getSubjectConverted($oHeader){
        $sSubject = '';
        if ( isset($oHeader->subject) && strlen($oHeader->subject) > 0 ) {
            foreach(imap_mime_header_decode($oHeader->subject) as $obj){
                $sSubject .= $obj->text;
            }
        }
        return $this->convertToUtf8($sSubject);
    } 

    /**
     * convert to utf8 if necessary.
     *
     * @return true or false
     * @param $string utf8 encoded string
     */
    protected function convertToUtf8($str) { 
        if(mb_detect_encoding($str, "UTF-8, ISO-8859-1, GBK")!="UTF-8")
            $str = utf8_encode($str);
        $str = iconv('UTF-8', 'UTF-8//IGNORE', $str);
        return $str; 
    }

    /**
     * convert imap given address in string
     *
     * @return array with name and address
     * @param $headerinfos the infos given by imap
     */
    protected function toAddressArray($aHeaderArray) {
        $email = "";
        $name = "";
        if(isset($aHeaderArray->mailbox) && isset($aHeaderArray->host)) {
            $email = $aHeaderArray->mailbox . "@" . $aHeaderArray->host;
        }

        if(!empty($aHeaderArray->personal)) {
            $name = imap_mime_header_decode($aHeaderArray->personal);
            $name = $name[0]->text;
        } else {
            $name = $email;
        }
        
        $name = $this->convertToUtf8($name);
        
        return array("name" => $name, "mail" => $email);
    } 

    /**
     * convert output of toAddressArray in a string address formated
     *
     * @return array string formated as name<mail>
     * @param $aMailArray Array with name and mail
     */
    public function toAddressString($aMailArray) {
        return $aMailArray["name"] . " <" . $aMailArray["mail"] . ">";
    }


    /**
     * convert dateTime struct to a string formated
     *
     * @return string date formated
     * @param $sFormat string to formate dataTime class default : "d/m/Y H:i:s" see http://php.net/manual/fr/function.date.php
     */
    public function receptionDateFormater($sFormat = "d/m/Y H:i:s") {
        return $this->tReceptionDate->format($sFormat);
    }

    /**
     * convert dateTime struct to a string formated
     *
     * @return string date formated
     * @param $sFormat string to formate dataTime class default : "d/m/Y H:i:s" see http://php.net/manual/fr/function.date.php
     */
    public function sendDateFormater($sFormat = "d/m/Y H:i:s") {
        return $this->tSendDate->format($sFormat);
    }

}


?>