<?php

/**
 * \file Categories.class.inc
 * \class Categories
 *
 * \author Yoann Perollet <yoann.perollet@veremes.com>.
 *
 * 	\brief This file contains the Modes php class
 *
 * This class defines Rest Api to Vitis Categories
 * 
 */
require_once 'Gtf.class.inc';
require_once dirname($_SERVER['SCRIPT_FILENAME']) . '/class/vitis_lib/Connection.class.inc';
require_once 'Category.class.inc';
require_once dirname($_SERVER['SCRIPT_FILENAME']) . '/class/vmlib/BdDataAccess.inc';

class Categories extends GTF {
    /**
     * @SWG\Definition(
     *   definition="/categories",
     *   allOf={
     *     @SWG\Schema(ref="#/definitions/categories")
     *   }
     * )
     * * @SWG\Tag(
     *   name="Categories",
     *   description="Operations about Category"
     * )
     */

    /**
     * construct
     * @param type $aPath url of the request
     * @param type $aValues parameters of the request
     * @param type $properties properties
     * @param type $bShortcut false to reinit variables
     * @param type $oConnection connection object
     */
    function __construct($aPath, $aValues, $properties, $bShortcut = false, $oConnection = false) {
        parent::__construct($aPath, $aValues, $properties, $bShortcut, $oConnection);
        $this->aSelectedFields = Array('category_id', 'name', 'description');
    }

    /**
     * @SWG\Get(path="/categories",
     *   tags={"Categories"},
     *   summary="Get Category",
     *   description="Request to get Category",
     *   operationId="GET",
     *   produces={"application/xml", "application/json", "application/x-vm-json"},
     *   @SWG\Parameter(
     *     name="token",
     *     in="query",
     *     description="category token",
     *     required=true,
     *     type="string"
     *   ),
     *   @SWG\Parameter(
     *     name="order_by",
     *     in="query",
     *     description="list of ordering fields",
     *     required=false,
     *     type="string"
     *   ),
     * @SWG\Parameter(
     *     name="sort_order",
     *     in="query",
     *     description="sort order",
     *     required=false,
     *     type="string"
     *   ),
     *   @SWG\Parameter(
     *     name="limit",
     *     in="query",
     *     description="number of element",
     *     required=false,
     *     type="integer",
     *     format="int32"
     *   ),
     *   @SWG\Parameter(
     *     name="offset",
     *     in="query",
     *     description="index of first element",
     *     required=false,
     *     type="string",
     *     format="int32"
     *   ),
     *   @SWG\Parameter(
     *     name="attributs",
     *     in="query",
     *     description="list of attributs",
     *     required=false,
     *     type="string"
     *   ),
     *   @SWG\Parameter(
     *     name="filter",
     *     in="query",
     *     description="filter results",
     *     required=false,
     *     type="string"
     *   ),
     *   @SWG\Parameter(
     *     name="distinct",
     *     in="query",
     *     description="delete duplicates",
     *     required=false,
     *     type="boolean"
     *   ),
     *   @SWG\Response(
     *         response=200,
     *         description="Poprerties Response",
     *         @SWG\Schema(ref="#/definitions/categories")
     *     )
     *  )
     */

    /**
     * get Categories
     * @return  Categories
     */
    function GET() {
        $aReturn = $this->genericGet($this->aProperties['schema_gtf'], "category", "category_id");
        return $aReturn['sMessage'];
    }

    /**
     * @SWG\Post(path="/categories",
     *   tags={"Categories"},
     *   summary="Add Category",
     *   description="Request to add category",
     *   operationId="POST",
     *   produces={"application/xml", "application/json"},
     *   @SWG\Parameter(
     *     name="token",
     *     in="formData",
     *     description="category token",
     *     required=true,
     *     type="string"
     *   ),
     *   @SWG\Parameter(
     *     name="name",
     *     in="formData",
     *     description="name of the category",
     *     required=true,
     *     type="integer",
     *     format="int32"
     *   ),
     *   @SWG\Parameter(
     *     name="description",
     *     in="formData",
     *     description="description of the category",
     *     required=false,
     *     type="string"
     *   ),
     *   @SWG\Parameter(
     *     name="workspaces",
     *     in="formData",
     *     description="workspaces of the category",
     *     required=false,
     *     type="string"
     *   ),
     *   @SWG\Response(
     *         response=200,
     *         description="Poprerties Response",
     *         @SWG\Schema(ref="#/definitions/categories")
     *     )
     *  )
     */

    /**
     * insert category
     * @return id of the category created
     */
    function POST() {
        $aReturn = $this->genericPost($this->aProperties['schema_gtf'], 'category', $this->aProperties['schema_gtf'] . '.seq_common', 'category_id');
        if ($aReturn['sStatus'] == 1) {
            $aXmlRacineAttribute['status'] = 1;
            $sMessage = $this->asDocument('', 'vitis', $this->aValues['sEncoding'], True, $aXmlRacineAttribute, $this->aValues['sSourceEncoding'], $this->aValues['output']);
            $oCategory = new Categorie($this->aPath, $this->aValues, $this->aProperties, $this->oConnection);
            $oCategory->GET();
            //
            $this->insertTraitementCategory();
        } else {
            $sMessage = $aReturn['sMessage'];
        }
        return $sMessage;
    }

    /**
     * @SWG\Put(path="/categories/{category_id}",
     *   tags={"Categories"},
     *   summary="Update Category",
     *   description="Request to update Category",
     *   operationId="PUT",
     *   produces={"application/xml", "application/json"},
     *   @SWG\Parameter(
     *     name="token",
     *     in="query",
     *     description="category token",
     *     required=true,
     *     type="string"
     *   ),
     * * @SWG\Parameter(
     *     name="category_id",
     *     in="path",
     *     description="id of the category",
     *     required=true,
     *     type="integer",
     *     format = "int32"
     *   ),
     *   @SWG\Parameter(
     *     name="name",
     *     in="query",
     *     description="name of the category",
     *     required=true,
     *     type="integer",
     *     format="int32"
     *   ),
     *   @SWG\Parameter(
     *     name="description",
     *     in="query",
     *     description="description of the category",
     *     required=false,
     *     type="string"
     *   ),
     *   @SWG\Parameter(
     *     name="workspaces",
     *     in="query",
     *     description="workspaces of the category",
     *     required=false,
     *     type="string"
     *   ),
     *   @SWG\Response(
     *         response=200,
     *         description="Poprerties Response",
     *         @SWG\Schema(ref="#/definitions/categories")
     *     )
     *  )
     */

    /**
     * update category
     * @return id of the category if ok error object if ko
     */
    function PUT() {
        $aReturn = $this->genericPut($this->aProperties['schema_gtf'], 'category', 'category_id');
        if ($aReturn['sStatus'] == 1) {
            $aXmlRacineAttribute['status'] = 1;
            $sMessage = $this->asDocument('', 'vitis', $this->aValues['sEncoding'], True, $aXmlRacineAttribute, $this->aValues['sSourceEncoding'], $this->aValues['output']);
            $oCategory = new Categorie($this->aPath, $this->aValues, $this->aProperties, $this->oConnection);
            $oCategory->GET();
            //
            $this->insertTraitementCategory();
        } else {
            $sMessage = $aReturn['sMessage'];
        }
        return $sMessage;
    }

    /**
     * @SWG\Delete(path="/categories/",
     *   tags={"Categories"},
     *   summary="delete Category",
     *   description="Request to delete Category",
     *   operationId="DELETE",
     *   produces={"application/xml", "application/json"},
     *   @SWG\Parameter(
     *     name="token",
     *     in="query",
     *     description="category token",
     *     required=true,
     *     type="string"
     *   ),
     * * @SWG\Parameter(
     *     name="idList",
     *     in="query",
     *     description="id of the categories",
     *     required=true,
     *     type="string"
     *   ),
     * @SWG\Response(
     *         response=200,
     *         description="Poprerties Response",
     *         @SWG\Schema(ref="#/definitions/categories")
     *     )
     *  )
     */
    /**
     * @SWG\Delete(path="/categories/{category_id}",
     *   tags={"Categories"},
     *   summary="delete Category",
     *   description="Request to delete Category",
     *   operationId="DELETE",
     *   produces={"application/xml", "application/json"},
     *   @SWG\Parameter(
     *     name="token",
     *     in="query",
     *     description="category token",
     *     required=true,
     *     type="string"
     *   ),
     * * @SWG\Parameter(
     *     name="category_id",
     *     in="path",
     *     description="id of the category",
     *     required=true,
     *     type="integer",
     *     format = "int32"
     *   ),
     * @SWG\Response(
     *         response=200,
     *         description="Poprerties Response",
     *         @SWG\Schema(ref="#/definitions/categories")
     *     )
     *  )
     */

    /**
     * delete category
     * @return id of category deleted or error object if a category is not deleted
     */
    function DELETE() {
        $aReturn = $this->genericDelete($this->aProperties['schema_gtf'], 'category', 'category_id');
        return $aReturn['sMessage'];
    }

    /*
     * *	Traitements liés à la catégorie.
     */

    Public function insertTraitementCategory() {
        // Le paramètre "workspaces" est passé dans la requête ?
        if ($this->aValues['workspaces'] !== NULL) {
            // Projets FME à rattacher à la catégorie ?
            if (!empty($this->aValues['workspaces'])) {
                $aWorkspaces = explode('|', $this->aValues['workspaces']);
                foreach ($aWorkspaces as $iWorkspaceId) {
                    $aValuesList = array("category_id" => $this->aValues["my_vitis_id"]);
                    $this->oConnection->oBd->update($this->aProperties['schema_gtf'], 'workspace', $aValuesList, "workspace_id", $iWorkspaceId);
                }
            }
        }
    }

}

?>