<?php

/**
 * \file FmeEngines.class.inc
 * \class FmeEngines
 *
 * \author Yoann Perollet <yoann.perollet@veremes.com>.
 *
 * 	\brief This file contains the Modes php class
 *
 * This class defines Rest Api to Vitis FmeEngines
 * 
 */
require_once 'Gtf.class.inc';
require_once dirname($_SERVER['SCRIPT_FILENAME']) . '/class/vitis_lib/Connection.class.inc';
require_once 'FmeEngine.class.inc';
require_once dirname($_SERVER['SCRIPT_FILENAME']) . '/class/vmlib/BdDataAccess.inc';
require_once 'gtf_lib/FmeCloud.class.inc';
require_once 'gtf_lib/FmeServer.class.inc';

class FmeEngines extends GTF {
    /**
     * @SWG\Definition(
     *   definition="/fmeengines",
     *   allOf={
     *     @SWG\Schema(ref="#/definitions/fmeengines")
     *   }
     * )
     * * @SWG\Tag(
     *   name="FmeEngines",
     *   description="Operations about Fme Engines"
     * )
     */

    /**
     * construct
     * @param type $aPath url of the request
     * @param type $aValues parameters of the request
     * @param type $properties properties
     * @param type $bShortcut false to reinit variables
     * @param type $oConnection connection object
     */
    function __construct($aPath, $aValues, $properties, $bShortcut = false, $oConnection = false) {
        parent::__construct($aPath, $aValues, $properties, $bShortcut, $oConnection);
        $this->aSelectedFields = Array("fme_engine_id", "name", "server_id", "local_path", "server_name", "fme_engine_type_id", "server_url", "login", "password", "s3_access_key_id", "s3_secret_access_key", "fme_cloud_api_token", "fme_engine_type_label", "s3_region", "iam_access_key_id", "iam_secret_access_key", 'fme_server_instance_name', 'fme_server_s3_bucket', "useexternaldbconnection");
    }

    /**
     * @SWG\Get(path="/fmeengines",
     *   tags={"FmeEngines"},
     *   summary="Get Fme Engines",
     *   description="Request to get Fme Engines",
     *   operationId="GET",
     *   produces={"application/xml", "application/json", "application/x-vm-json"},
     *   @SWG\Parameter(
     *     name="token",
     *     in="query",
     *     description="fmeengine token",
     *     required=true,
     *     type="string"
     *   ),
     * @SWG\Parameter(
     *     name="order_by",
     *     in="query",
     *     description="list of ordering fields",
     *     required=false,
     *     type="string"
     *   ),
     * @SWG\Parameter(
     *     name="sort_order",
     *     in="query",
     *     description="sort order",
     *     required=false,
     *     type="string"
     *   ),
     * @SWG\Parameter(
     *     name="limit",
     *     in="query",
     *     description="number of element",
     *     required=false,
     *     type="integer",
     *     format="int32"
     *   ),
     * @SWG\Parameter(
     *     name="offset",
     *     in="query",
     *     description="index of first element",
     *     required=false,
     *     type="string",
     *   ),
     * @SWG\Parameter(
     *     name="attributs",
     *     in="query",
     *     description="list of attributs",
     *     required=false,
     *     type="string"
     *   ),
     * @SWG\Parameter(
     *     name="filter",
     *     in="query",
     *     description="filter results",
     *     required=false,
     *     type="string"
     *   ),
     * @SWG\Parameter(
     *     name="distinct",
     *     in="query",
     *     description="delete duplicates",
     *     required=false,
     *     type="boolean"
     *   ),
     *   @SWG\Response(
     *         response=200,
     *         description="Poprerties Response",
     *         @SWG\Schema(ref="#/definitions/fmeengines")
     *     )
     *  )
     */
    /**
     * @SWG\Get(path="/fmeengines/{fme_engine_id}/Test", 
     *   tags={"FmeEngines"},
     *   summary="Get FmeEngine test",
     *   description="Request to get FmeEnginetest by id",
     *   operationId="GET",
     *   produces={"application/xml", "application/json", "application/x-vm-json"},
     *   @SWG\Parameter(
     *     name="token",
     *     in="query",
     *     description="user token",
     *     required=true,
     *     type="string"
     *   ),
     * @SWG\Parameter(
     *     name="fme_engine_id",
     *     in="path",
     *     description="user id",
     *     required=true,
     *     type="integer",
     *   format="int32"
     *   ),
     * @SWG\Parameter(
     *     name="attributs",
     *     in="query",
     *     description="list of attributs",
     *     required=false,
     *     type="string"
     *   ),
     *   @SWG\Response(
     *         response=200,
     *         description="Poprerties Response",
     *         @SWG\Schema(ref="#/definitions/fmeengines")
     *     )
     *  )
     */

    /**
     * get FmeEngines
     * @return  FmeEngines
     */
    function GET() {
        // Instance FME Server sur Fme Cloud.
        if (!empty($this->aPath[3]) && $this->aPath[3] == 'instances') {
            if (empty($this->aPath[5])) {
                $sMessage = $this->getFmeServerInstance($this->aPath[2], $this->aPath[4], $oFmeCloud);
                return $sMessage;
            }
            else {
                if ($this->aPath[5] == 'info')
                    $sMessage = $this->getFmeServerInstanceInfo($this->aPath[2], $this->aPath[4]);
                return $sMessage;
            }
        }
        else {
            $aReturn = $this->genericGet($this->aProperties['schema_gtf'], "v_fme_engine", "fme_engine_id");
            if (!empty($this->aPath[3]) && $this->aPath[3] == 'Test') {
                if ($aReturn['sStatus'] == 1) {
                    // Test de validité de la licence FME.
                    $this->testFmeLicence();
                    $aXmlRacineAttribute['status'] = 1;
                    $sMessage = $this->asDocument('', 'vitis', $this->aValues['sEncoding'], True, $aXmlRacineAttribute, $this->aValues['sSourceEncoding'], $this->aValues['output']);
                    return $sMessage;
                }
            } else
                return $aReturn['sMessage'];
        }
    }

    /**
     * @SWG\Post(path="/fmeengines",
     *   tags={"FmeEngines"},
     *   summary="Add FmeEngine",
     *   description="Request to add fmeengine",
     *   operationId="POST",
     *   produces={"application/xml", "application/json", "application/x-vm-json"},
     *   @SWG\Parameter(
     *     name="token",
     *     in="formData",
     *     description="fmeengine token",
     *     required=true,
     *     type="string"
     *   ),
     *   @SWG\Parameter(
     *     name="name",
     *     in="formData",
     *     description="Name of the fme engine",
     *     required=true,
     *     type="string"
     *   ),
     *   @SWG\Parameter(
     *     name="server_id",
     *     in="formData",
     *     description="Server of fme engine",
     *     required=true,
     *     type="integer",
     *     format="int32"
     *   ),
     *   @SWG\Parameter(
     *     name="local_path",
     *     in="formData",
     *     description="local_path of the fme engine",
     *     required=true,
     *     type="string"
     *   ),
     *   @SWG\Response(
     *         response=200,
     *         description="Poprerties Response",
     *         @SWG\Schema(ref="#/definitions/fmeengines")
     *     )
     *  )
     */

    /**
     * insert fmeengine
     * @return id of the fmeengine created
     */
    function POST() {
        $aReturn = $this->genericPost($this->aProperties['schema_gtf'], 'v_fme_engine', $this->aProperties['schema_gtf'] . '.seq_common', 'fme_engine_id');
        if ($aReturn['sStatus'] == 1) {
            $aXmlRacineAttribute['status'] = 1;
            $sMessage = $this->asDocument('', 'vitis', $this->aValues['sEncoding'], True, $aXmlRacineAttribute, $this->aValues['sSourceEncoding'], $this->aValues['output']);
            $oFmeEngine = new FmeEngine($this->aPath, $this->aValues, $this->aProperties, $this->oConnection);
            $oFmeEngine->GET();
        } else {
            $sMessage = $aReturn['sMessage'];
        }
        return $sMessage;
    }

    /**
     * @SWG\Put(path="/fmeengines/{fme_engine_id}",
     *   tags={"FmeEngines"},
     *   summary="Update FmeEngine",
     *   description="Request to update FmeEngine",
     *   operationId="PUT",
     *   produces={"application/xml", "application/json", "application/x-vm-json"},
     *   @SWG\Parameter(
     *     name="token",
     *     in="query",
     *     description="fmeengine token",
     *     required=true,
     *     type="string"
     *   ),
     * * @SWG\Parameter(
     *     name="fme_engine_id",
     *     in="path",
     *     description="id of the fme engine",
     *     required=true,
     *     type="integer",
     *     format = "int32"
     *   ),
     *   @SWG\Parameter(
     *     name="name",
     *     in="query",
     *     description="Name of the fme engine",
     *     required=true,
     *     type="string"
     *   ),
     *   @SWG\Parameter(
     *     name="server_id",
     *     in="query",
     *     description="Server of fme engine",
     *     required=true,
     *     type="integer",
     *     format="int32"
     *   ),
     *   @SWG\Parameter(
     *     name="local_path",
     *     in="query",
     *     description="local_path of the fme engine",
     *     required=true,
     *     type="string"
     *   ),
     *   @SWG\Response(
     *         response=200,
     *         description="Poprerties Response",
     *         @SWG\Schema(ref="#/definitions/fmeengines")
     *     )
     *  )
     */
    
    /**
     * @SWG\Put(path="/fmeengines/{fme_engine_id}/instances/{instance_name}/start",
     *   tags={"FmeEngines"},
     *   summary="Start Fme Server instance",
     *   description="Request to start an instance of Fme Server on Fme Cloud",
     *   operationId="PUT",
     *   produces={"application/xml", "application/json", "application/x-vm-json"},
     *   @SWG\Parameter(
     *     name="token",
     *     in="query",
     *     description="user token",
     *     required=true,
     *     type="string"
     *   ),
     * * @SWG\Parameter(
     *     name="fme_engine_id",
     *     in="path",
     *     description="id of the fme engine",
     *     required=true,
     *     type="integer",
     *     format = "int32"
     *   ),
     * * @SWG\Parameter(
     *     name="instance_name",
     *     in="path",
     *     description="name of the fme server instance",
     *     required=true,
     *     type="string"
     *   ),
     *   @SWG\Response(
     *         response=200,
     *         description="Poprerties Response",
     *         @SWG\Schema(ref="#/definitions/fmeengines")
     *     )
     *  )
     */

    /**
     * @SWG\Put(path="/fmeengines/{fme_engine_id}/instances/{instance_name}/pause",
     *   tags={"FmeEngines"},
     *   summary="Pause Fme Server instance",
     *   description="Request to pause an instance of Fme Server on Fme Cloud",
     *   operationId="PUT",
     *   produces={"application/xml", "application/json", "application/x-vm-json"},
     *   @SWG\Parameter(
     *     name="token",
     *     in="query",
     *     description="user token",
     *     required=true,
     *     type="string"
     *   ),
     * * @SWG\Parameter(
     *     name="fme_engine_id",
     *     in="path",
     *     description="id of the fme engine",
     *     required=true,
     *     type="integer",
     *     format = "int32"
     *   ),
     * * @SWG\Parameter(
     *     name="instance_name",
     *     in="path",
     *     description="name of the fme server instance",
     *     required=true,
     *     type="string"
     *   ),
     *   @SWG\Response(
     *         response=200,
     *         description="Poprerties Response",
     *         @SWG\Schema(ref="#/definitions/fmeengines")
     *     )
     *  )
     */
    
    /**
     * update fmeengine
     * @return id of the fmeengine if ok error object if ko
     */
    function PUT() {
        if (!empty($this->aPath[3])) {
            if ($this->aPath[3] == 'instances') {
                switch($this->aPath[5]) {
                    case 'start':
                        $sMessage = $this->startFmeServerInstance($this->aPath[2], $this->aPath[4]);
                        break;
                    case 'pause':
                        $sMessage = $this->pauseFmeServerInstance($this->aPath[2], $this->aPath[4]);
                        break;
                }
            }
        }
        else {
            $aReturn = $this->genericPut($this->aProperties['schema_gtf'], 'v_fme_engine', 'fme_engine_id');
            if ($aReturn['sStatus'] == 1) {
                $aXmlRacineAttribute['status'] = 1;
                $sMessage = $this->asDocument('', 'vitis', $this->aValues['sEncoding'], True, $aXmlRacineAttribute, $this->aValues['sSourceEncoding'], $this->aValues['output']);
                $oFmeEngine = new FmeEngine($this->aPath, $this->aValues, $this->aProperties, $this->oConnection);
                $oFmeEngine->GET();
            } else {
                $sMessage = $aReturn['sMessage'];
            }
        }
        return $sMessage;
    }

    /**
     * @SWG\Delete(path="/fmeengines/",
     *   tags={"FmeEngines"},
     *   summary="delete FmeEngine",
     *   description="Request to delete FmeEngine",
     *   operationId="DELETE",
     *   produces={"application/xml", "application/json", "application/x-vm-json"},
     *   @SWG\Parameter(
     *     name="token",
     *     in="query",
     *     description="fmeengine token",
     *     required=true,
     *     type="string"
     *   ),
     * * @SWG\Parameter(
     *     name="idList",
     *     in="query",
     *     description="id of the fmeengines",
     *     required=true,
     *     type="string"
     *   ),
     * @SWG\Response(
     *         response=200,
     *         description="Poprerties Response",
     *         @SWG\Schema(ref="#/definitions/fmeengines")
     *     )
     *  )
     */
    /**
     * @SWG\Delete(path="/fmeengines/{fme_engine_id}",
     *   tags={"FmeEngines"},
     *   summary="delete FmeEngine",
     *   description="Request to delete FmeEngine",
     *   operationId="DELETE",
     *   produces={"application/xml", "application/json", "application/x-vm-json"},
     *   @SWG\Parameter(
     *     name="token",
     *     in="query",
     *     description="fmeengine token",
     *     required=true,
     *     type="string"
     *   ),
     * * @SWG\Parameter(
     *     name="fme_engine_id",
     *     in="path",
     *     description="id of the fmeengine",
     *     required=true,
     *     type="integer",
     *     format = "int32"
     *   ),
     * @SWG\Response(
     *         response=200,
     *         description="Poprerties Response",
     *         @SWG\Schema(ref="#/definitions/fmeengines")
     *     )
     *  )
     */

    /**
     * delete fmeengine.
     * @return id of fmeengine deleted or error object if a fmeengine is not deleted
     */
    function DELETE() {
        $aReturn = $this->genericDelete($this->aProperties['schema_gtf'], 'v_fme_engine', 'fme_engine_id');
        return $aReturn['sMessage'];
    }

    /**
     * get fme licence parameters.
     * @return id of fmeengine deleted or error object if a fmeengine is not deleted
     */
    function getFmeLicence($fmepath) {
        // Si le fichier existe et la fonction exec ne renvoie pas d'erreur : remplissage du tableau
        if (file_exists($fmepath)) {
            // Le fichier existe
            $this->aObjects[0]->aFields['exists'] = true;

            // Exécute fme.exe
            $aOutput = array();
            $iReturnStatus = 1;
            
            // Variable d'environement "SystemDrive" obligatoire sinon erreur avec FME !
            if (strtolower(substr(PHP_OS, 0, 3)) == 'win')
                putenv('SystemDrive=' . substr($fmepath, 0, 2));
            
            // Paramètre "INFO" obligatoire pour FME 2016 sinon pas de ligne "Temporary License".
            exec('"' . $fmepath . '"' . ' INFO', $aOutput, $iReturnStatus);
            // Paramètre "INFO" non reconnu par FME 2014 (erreur).
            if ($iReturnStatus === 1)
                exec($fmepath, $aOutput, $iReturnStatus);

            // Le fichier fme.exe s'est exécuté normalement ?
            if ($iReturnStatus === 0) {
                // Est ce que la licence a expiré ?
                if (stripos(implode($aOutput), 'licence has expired') !== false)
                    $this->aObjects[0]->aFields['valid'] = false;
                else {
                    // La licence est valide	
                    $this->aObjects[0]->aFields['valid'] = true;

                    // Version du logiciel
                    $this->aObjects[0]->aFields['version'] = trim($aOutput[0]);

                    // Type de licence
                    $this->aObjects[0]->aFields['type'] = trim($aOutput[1]);

                    // Validité de la licence
                    $this->aObjects[0]->aFields['validity'] = trim($aOutput[2]);

                    // Host
                    $this->aObjects[0]->aFields['host'] = trim(substr($aOutput[3], stripos($aOutput[3], ':') + 1));
                }
            }
        }
    }
    
    /**
     * Test Fme licence.
     */
    function testFmeLicence() {
        // Tableau de retour contenant les infos de la licence
        $aFmeLicence = array('exists' => false, 'valid' => false, 'version' => '', 'type' => '', 'validity' => '', 'host' => '');
        $this->aObjects[0]->aFields = array_merge($this->aObjects[0]->aFields, $aFmeLicence);
        //
        if (empty($this->aObjects[0]->aFields['fme_engine_type_id']))
            $this->aObjects[0]->aFields['fme_engine_type_id'] = 'desktop';
        switch ($this->aObjects[0]->aFields['fme_engine_type_id']) {
            case 'desktop':
                $this->getFmeLicence($this->aObjects[0]->aFields["local_path"]);
            break;
            case 'fme_server':
                $oFmeServer = new FmeServer($this->aObjects[0]->aFields['server_url'], $this->aObjects[0]->aFields['login'], $this->aObjects[0]->aFields['password'], 'day', 1);
                $oFmeServerLicenceStatus = $oFmeServer->getLicenceStatus();
                if ($oFmeServerLicenceStatus === false) {
                    $this->aObjects[0]->aFields['exists'] = true;
                    $this->aObjects[0]->aFields['request_error'] = true;
                }
                else if (empty($oFmeServerLicenceStatus))
                    $this->aObjects[0]->aFields['exists'] = false;
                else {
                    $this->aObjects[0]->aFields['exists'] = true;
                    if ($oFmeServerLicenceStatus->isLicenseExpired === true)
                        $this->aObjects[0]->aFields['expired'] = true;
                    else if ($oFmeServerLicenceStatus->isLicensed === false)
                        $this->aObjects[0]->aFields['valid'] = false;
                    else {
                        // Info sur la licence de FME Server.
                        //$this->aObjects[0]->aFields['exists'] = true;
                        $this->aObjects[0]->aFields['valid'] = true;
                        $sExpiryDate = $oFmeServerLicenceStatus->expiryDate;
                        if (!empty($sExpiryDate))
                            $sExpiryDate = substr($sExpiryDate, 6, 2) . '/' . substr($sExpiryDate, 4, 2) . '/' . substr($sExpiryDate, 0, 4);
                        $this->aObjects[0]->aFields['validity'] = $sExpiryDate;
                        // Info sur FME Server.
                        $oFmeServerInfo = $oFmeServer->getInfo();
                        $this->aObjects[0]->aFields['version'] = $oFmeServerInfo->build;
                        $this->aObjects[0]->aFields['type'] = $oFmeServerInfo->version;
                        $this->aObjects[0]->aFields['host'] = parse_url($this->aObjects[0]->aFields['server_url'], PHP_URL_HOST);
                    }
                }
            break;
        }
    }

    /**
     * Get a Fme Server instance on Fme Cloud.
     * @param {number} $iFmeEngineId Id of a Fme Engine.
     * @param {string} $sFmeServerInstanceName Name of a Fme Server instance on Fme Cloud.
     */
    function getFmeServerInstance($iFmeEngineId, $sFmeServerInstanceName, &$oFmeCloud = null, &$aFmeEngine = array()) {
        require $this->sRessourcesFile;
        $iErrorId = 17;
        $sErrorMessage = '';
        // Paramètres du moteur Fme.
        $aParams = array();
        $aParams['sSchemaGtf'] = array('value' => $this->aProperties['schema_gtf'], 'type' => 'schema_name');
        $aParams['fme_engine_id'] = array('value' => $iFmeEngineId, 'type' => 'number');
        $oPDOresult = $this->oConnection->oBd->executeWithParams($aSql['getFmeEngine'], $aParams);
        if ($this->oConnection->oBd->enErreur()) {
            $iErrorId = 1;
            $sErrorMessage = $this->oConnection->oBd->getBDMessage();
        }
        else if ($this->oConnection->oBd->nombreLigne($oPDOresult) > 0) {
            $aFmeEngine = $this->oConnection->oBd->ligneSuivante($oPDOresult);
            $oFmeCloud = new FmeCloud($aFmeEngine['server_url'], $aFmeEngine['fme_cloud_api_token']);
            // Liste des occurrences de FME Server.
            $aFmeServerInstances = $oFmeCloud->serviceRequest('instances');
            if (!empty($aFmeServerInstances)) {
                foreach($aFmeServerInstances as $oInstance) {
                    if ($oInstance->name == $sFmeServerInstanceName)
                        $oFmeServerInstance = $oInstance;
                }
                if (empty($oFmeServerInstance))
                    $sErrorMessage = 'L\'instance FME Server "' . $sFmeServerInstanceName . '" n\'existe pas.';
                else {
                    $this->aFields['instance'] = $oFmeServerInstance;
                }
            }
            else
                $sErrorMessage ='Aucune instance FME Server n\'est disponible.';
        }
        else
            $sErrorMessage = 'Le moteur FME spécifiée n\'existe pas.';
        // Message de retour pour l'API Rest.
        if (empty($sErrorMessage)) {
            $aXmlRacineAttribute['status'] = 1;
            $sMessage = $this->asDocument('', 'vitis', $this->aValues['sEncoding'], True, $aXmlRacineAttribute, $this->aValues['sSourceEncoding'], $this->aValues['output']);
        }
        else {
            $oError = new VitisError($iErrorId, $sErrorMessage);
            $aXmlRacineAttribute['status'] = 0;
            $sMessage = $oError->asDocument('', 'vitis', $this->aValues['sEncoding'], True, $aXmlRacineAttribute, $this->aValues['sSourceEncoding'], $this->aValues['output']);
        }
        return $sMessage;
    }

    
    /**
     * Start a Fme Server instance on Fme Cloud.
     * @param {number} $iFmeEngineId Id of a Fme Engine.
     * @param {string} $sFmeServerInstanceName Name of a Fme Server instance on Fme Cloud.
     */
    function startFmeServerInstance($iFmeEngineId, $sFmeServerInstanceName) {
        $sMessage = $this->getFmeServerInstance($iFmeEngineId, $sFmeServerInstanceName, $oFmeCloud);
        $oReturn = json_decode($sMessage);
        if ($oReturn->status == 1) {
            // Démarrage de l'occurence si elle est en pause.
            if ($oReturn->instance->state == 'PAUSED') {
                if ($oFmeCloud->startInstance($oReturn->instance->id) === false)
                    $sErrorMessage = 'L\'instance Fme Server n\'a pu être démarrée.' ;
                else {
                    $oFmeServerInstance = $oFmeCloud->serviceRequest('instances/' . $oReturn->instance->id);
                    if ($oFmeServerInstance === false)
                        $sErrorMessage = 'Erreur pendant le démarrage de l\'instance Fme Server.';
                    else
                        $this->aFields = array('instance' => $oFmeServerInstance);
                }
            }
            else if ($oReturn->instance->state == 'RUNNING')
                $sErrorMessage = 'L\'instance Fme Server est déja en cours d\'exécution.';
            else if ($oReturn->instance->state == 'executing_action')
                $sErrorMessage = 'Une action est en cours d\'exéution sur L\'instance Fme Server.';
            else
                $sErrorMessage = $oReturn->instance->state;
            // Message de retour pour l'API Rest.
            if (empty($sErrorMessage)) {
                $aXmlRacineAttribute['status'] = 1;
                $sMessage = $this->asDocument('', 'vitis', $this->aValues['sEncoding'], True, $aXmlRacineAttribute, $this->aValues['sSourceEncoding'], $this->aValues['output']);
            }
            else {
                $oError = new VitisError(17, $sErrorMessage);
                $aXmlRacineAttribute['status'] = 0;
                $sMessage = $oError->asDocument('', 'vitis', $this->aValues['sEncoding'], True, $aXmlRacineAttribute, $this->aValues['sSourceEncoding'], $this->aValues['output']);
            }
        }
        return $sMessage;
    }
    
    /**
     * Pause a Fme Server instance on Fme Cloud.
     * @param {number} $iFmeEngineId Id of a Fme Engine.
     * @param {string} $sFmeServerInstanceName Name of a Fme Server instance on Fme Cloud.
     */
    function pauseFmeServerInstance($iFmeEngineId, $sFmeServerInstanceName) {
        $sMessage = $this->getFmeServerInstance($iFmeEngineId, $sFmeServerInstanceName, $oFmeCloud);
        $oReturn = json_decode($sMessage);
        if ($oReturn->status == 1) {
            // Mise en pause de l'occurence si elle est en cours d'éxecution.
            if ($oReturn->instance->state == 'RUNNING') {
                if ($oFmeCloud->pauseInstance($oReturn->instance->id) === false)
                    $sErrorMessage = 'L\'instance Fme Server n\'a pu être arrêtée.' ;
                else {
                    $oFmeServerInstance = $oFmeCloud->serviceRequest('instances/' . $oReturn->instance->id);
                    if ($oFmeServerInstance === false) 
                        $sErrorMessage = 'Erreur pendant l\'arrêt de l\'instance Fme Server.';
                    else
                        $this->aFields = array('instance' => $oFmeServerInstance);
                }
            }
            else if ($oReturn->instance->state == 'PAUSED')
                $sErrorMessage = 'L\'instance Fme Server est déja à l\'arrêt';
            else if ($oReturn->instance->state == 'executing_action')
                $sErrorMessage = 'Une action est en cours d\'exécution sur L\'instance Fme Server.';
            else
                $sErrorMessage = $oReturn->instance->state;
            // Message de retour pour l'API Rest.
            if (empty($sErrorMessage)) {
                $aXmlRacineAttribute['status'] = 1;
                $sMessage = $this->asDocument('', 'vitis', $this->aValues['sEncoding'], True, $aXmlRacineAttribute, $this->aValues['sSourceEncoding'], $this->aValues['output']);
            }
            else {
                $oError = new VitisError(17, $sErrorMessage);
                $aXmlRacineAttribute['status'] = 0;
                $sMessage = $oError->asDocument('', 'vitis', $this->aValues['sEncoding'], True, $aXmlRacineAttribute, $this->aValues['sSourceEncoding'], $this->aValues['output']);
            }
        }
        return $sMessage;
    }
    
    /**
     * Get Fme Server instance information on Fme Cloud.
     * @param {number} $iFmeEngineId Id of a Fme Engine.
     * @param {string} $sFmeServerInstanceName Name of a Fme Server instance on Fme Cloud.
     */
    function getFmeServerInstanceInfo($iFmeEngineId, $sFmeServerInstanceName) {
        $sMessage = $this->getFmeServerInstance($iFmeEngineId, $sFmeServerInstanceName, $oFmeCloud, $aFmeEngine);
        $oReturn = json_decode($sMessage);
        if ($oReturn->status == 1) {
            $oFmeServer = new FmeServer('https://' . $oReturn->instance->public_ip, $aFmeEngine['login'], $aFmeEngine['password'], 'day', 1);
            $oInstanceInfo = $oFmeServer->getInfo();
            if ($oInstanceInfo === false)
                $sErrorMessage = 'Erreur pendant la requête de demande d\'information sur l\'instance Fme Server';
            else {
                $oInstanceInfo->host = $oReturn->instance->instance_url;
                $this->aFields = array('info' => $oInstanceInfo);
            }
            // Message de retour pour l'API Rest.
            if (empty($sErrorMessage)) {
                $aXmlRacineAttribute['status'] = 1;
                $sMessage = $this->asDocument('', 'vitis', $this->aValues['sEncoding'], True, $aXmlRacineAttribute, $this->aValues['sSourceEncoding'], $this->aValues['output']);
            }
            else {
                $oError = new VitisError(17, $sErrorMessage);
                $aXmlRacineAttribute['status'] = 0;
                $sMessage = $oError->asDocument('', 'vitis', $this->aValues['sEncoding'], True, $aXmlRacineAttribute, $this->aValues['sSourceEncoding'], $this->aValues['output']);
            }
        }
        return $sMessage;
    }
}
?>