<?php

/**
 * \file GtfEngines.class.inc
 * \class GtfEngines
 *
 * \author Yoann Perollet <yoann.perollet@veremes.com>.
 *
 * 	\brief This file contains the Modes php class
 *
 * This class defines Rest Api to Vitis GtfEngines
 * 
 */
require_once 'Gtf.class.inc';
require_once dirname($_SERVER['SCRIPT_FILENAME']) . '/class/vitis_lib/Connection.class.inc';
require_once 'GtfEngine.class.inc';
require_once dirname($_SERVER['SCRIPT_FILENAME']) . '/class/vmlib/BdDataAccess.inc';

class GtfEngines extends GTF {
    /**
     * @SWG\Definition(
     *   definition="/gtfengines",
     *   allOf={
     *     @SWG\Schema(ref="#/definitions/gtfengines")
     *   }
     * )
     * * @SWG\Tag(
     *   name="GtfEngines",
     *   description="Operations about Gtf Engines"
     * )
     */

    /**
     * construct
     * @param type $aPath url of the request
     * @param type $aValues parameters of the request
     * @param type $properties properties
     * @param type $bShortcut false to reinit variables
     * @param type $oConnection connection object
     */
    function __construct($aPath, $aValues, $properties, $bShortcut = false, $oConnection = false) {
        parent::__construct($aPath, $aValues, $properties, $bShortcut, $oConnection);
        $this->aSelectedFields = Array('gtf_engine_id', 'name', 'period', 'enabled', 'server_id', 'fme_engine_id', 'server_name', 'fme_name', 'tags');
    }

    /**
     * @SWG\Get(path="/gtfengines",
     *   tags={"GtfEngines"},
     *   summary="Get Gtf Engines",
     *   description="Request to get Gtf Engines",
     *   operationId="GET",
     *   produces={"application/xml", "application/json", "application/x-vm-json"},
     *   @SWG\Parameter(
     *     name="token",
     *     in="query",
     *     description="gtfengine token",
     *     required=true,
     *     type="string"
     *   ),
     * @SWG\Parameter(
     *     name="order_by",
     *     in="query",
     *     description="list of ordering fields",
     *     required=false,
     *     type="string"
     *   ),
     * @SWG\Parameter(
     *     name="sort_order",
     *     in="query",
     *     description="sort order",
     *     required=false,
     *     type="string"
     *   ),
     * @SWG\Parameter(
     *     name="limit",
     *     in="query",
     *     description="number of element",
     *     required=false,
     *     type="integer",
     *     format="int32"
     *   ),
     * @SWG\Parameter(
     *     name="offset",
     *     in="query",
     *     description="index of first element",
     *     required=false,
     *     type="string",
     *     format="int32"
     *   ),
     * @SWG\Parameter(
     *     name="attributs",
     *     in="query",
     *     description="list of attributs",
     *     required=false,
     *     type="string"
     *   ),
     * @SWG\Parameter(
     *     name="filter",
     *     in="query",
     *     description="filter results",
     *     required=false,
     *     type="string"
     *   ),
     * @SWG\Parameter(
     *     name="distinct",
     *     in="query",
     *     description="delete duplicates",
     *     required=false,
     *     type="boolean"
     *   ),
     *   @SWG\Response(
     *         response=200,
     *         description="Poprerties Response",
     *         @SWG\Schema(ref="#/definitions/gtfengines")
     *     )
     *  )
     */

    /**
     * get GtfEngines
     * @return  GtfEngines
     */
    function GET() {
        $aReturn = $this->genericGet($this->aProperties['schema_gtf'], "v_gtf_engine", "gtf_engine_id");
        return $aReturn['sMessage'];
    }

    /**
     * @SWG\Post(path="/gtfengines",
     *   tags={"GtfEngines"},
     *   summary="Add GtfEngine",
     *   description="Request to add gtfengine",
     *   operationId="POST",
     *   produces={"application/xml", "application/json"},
     *   @SWG\Parameter(
     *     name="token",
     *     in="formData",
     *     description="gtfengine token",
     *     required=true,
     *     type="string"
     *   ),
     *   @SWG\Parameter(
     *     name="name",
     *     in="formData",
     *     description="Name of the gtf engine",
     *     required=true,
     *     type="string"
     *   ),
     *   @SWG\Parameter(
     *     name="period",
     *     in="formData",
     *     description="period (minutes) of the gtf engine",
     *     required=false,
     *     type="string"
     *   ),
     *   @SWG\Parameter(
     *     name="enabled",
     *     in="formData",
     *     description="gtf engine enabled ?",
     *     required=false,
     *     type="boolean"
     *   ),
     *   @SWG\Parameter(
     *     name="server_id",
     *     in="formData",
     *     description="Server of gtf engine",
     *     required=false,
     *     type="integer",
     *     format="int32"
     *   ),
     *   @SWG\Parameter(
     *     name="fme_engine_id",
     *     in="formData",
     *     description="id of fme engine",
     *     required=true,
     *     type="integer",
     *     format="int32"
     *   ),
     *   @SWG\Parameter(
     *     name="tags",
     *     in="formData",
     *     description="tags of the gtf engine",
     *     required=false,
     *     type="string"
     *   ),
     *   @SWG\Response(
     *         response=200,
     *         description="Poprerties Response",
     *         @SWG\Schema(ref="#/definitions/gtfengines")
     *     )
     *  )
     */

    /**
     * insert gtfengine
     * @return id of the gtfengine created
     */
    function POST() {
        if (!isset($this->aValues['tags'])) {
            $this->aValues['tags'] = 'default';
        }
        require $this->sRessourcesFile;
        $aParams['sSchemaGtf'] = array('value' => $this->aProperties['schema_gtf'], 'type' => 'schema_name');
        $oPDOresult = $this->oConnection->oBd->executeWithParams($aSql['getLastGTFEngineID'], $aParams);
        while ($oId = $this->oConnection->oBd->objetSuivant($oPDOresult))
            $this->aValues['gtf_engine_id'] = $oId->id + 1;
        $oPDOresult = $this->oConnection->oBd->fermeResultat();
        $aReturn = $this->genericPost($this->aProperties['schema_gtf'], 'v_gtf_engine', '', 'gtf_engine_id');
        if ($aReturn['sStatus'] == 1) {
            $aXmlRacineAttribute['status'] = 1;
            $sMessage = $this->asDocument('', 'vitis', $this->aValues['sEncoding'], True, $aXmlRacineAttribute, $this->aValues['sSourceEncoding'], $this->aValues['output']);
            $oGtfEngine = new GtfEngine($this->aPath, $this->aValues, $this->aProperties, $this->oConnection);
            $oGtfEngine->GET();
            //
            $this->insertGtfEngineTags();
        } else {
            $sMessage = $aReturn['sMessage'];
        }
        return $sMessage;
    }

    /**
     * @SWG\Put(path="/gtfengines/{gtf_engine_id}",
     *   tags={"GtfEngines"},
     *   summary="Update GtfEngine",
     *   description="Request to update GtfEngine",
     *   operationId="PUT",
     *   produces={"application/xml", "application/json"},
     *   @SWG\Parameter(
     *     name="token",
     *     in="query",
     *     description="gtfengine token",
     *     required=true,
     *     type="string"
     *   ),
     * * @SWG\Parameter(
     *     name="gtf_engine_id",
     *     in="path",
     *     description="id of the gtfengine",
     *     required=true,
     *     type="integer",
     *     format = "int32"
     *   ),
     *   @SWG\Parameter(
     *     name="name",
     *     in="query",
     *     description="Name of the gtf engine",
     *     required=true,
     *     type="string"
     *   ),
     *   @SWG\Parameter(
     *     name="period",
     *     in="query",
     *     description="period (minutes) of the gtf engine",
     *     required=false,
     *     type="string"
     *   ),
     *   @SWG\Parameter(
     *     name="enabled",
     *     in="query",
     *     description="gtf engine enabled ?",
     *     required=false,
     *     type="boolean"
     *   ),
     *   @SWG\Parameter(
     *     name="server_id",
     *     in="query",
     *     description="Server of gtf engine",
     *     required=false,
     *     type="integer",
     *     format="int32"
     *   ),
     *   @SWG\Parameter(
     *     name="fme_engine_id",
     *     in="query",
     *     description="id of fme engine",
     *     required=true,
     *     type="integer",
     *     format="int32"
     *   ),
     *   @SWG\Parameter(
     *     name="tags",
     *     in="query",
     *     description="tags of the gtf engine",
     *     required=false,
     *     type="string"
     *   ),
     *   @SWG\Response(
     *         response=200,
     *         description="Poprerties Response",
     *         @SWG\Schema(ref="#/definitions/gtfengines")
     *     )
     *  )
     */

    /**
     * update gtfengine
     * @return id of the gtfengine if ok error object if ko
     */
    function PUT() {
        if (!isset($this->aValues['tags'])) {
            $this->aValues['tags'] = 'default';
        }
        $aReturn = $this->genericPut($this->aProperties['schema_gtf'], 'v_gtf_engine', 'gtf_engine_id');
        if ($aReturn['sStatus'] == 1) {
            $aXmlRacineAttribute['status'] = 1;
            $sMessage = $this->asDocument('', 'vitis', $this->aValues['sEncoding'], True, $aXmlRacineAttribute, $this->aValues['sSourceEncoding'], $this->aValues['output']);
            $oGtfEngine = new GtfEngine($this->aPath, $this->aValues, $this->aProperties, $this->oConnection);
            $oGtfEngine->GET();
            //
            $this->insertGtfEngineTags();
        } else {
            $sMessage = $aReturn['sMessage'];
        }
        return $sMessage;
    }

    /**
     * @SWG\Delete(path="/gtfengines/",
     *   tags={"GtfEngines"},
     *   summary="delete GtfEngine",
     *   description="Request to delete GtfEngine",
     *   operationId="DELETE",
     *   produces={"application/xml", "application/json"},
     *   @SWG\Parameter(
     *     name="token",
     *     in="query",
     *     description="gtfengine token",
     *     required=true,
     *     type="string"
     *   ),
     * * @SWG\Parameter(
     *     name="idList",
     *     in="query",
     *     description="id of the gtfengines",
     *     required=true,
     *     type="string"
     *   ),
     * @SWG\Response(
     *         response=200,
     *         description="Poprerties Response",
     *         @SWG\Schema(ref="#/definitions/gtfengines")
     *     )
     *  )
     */
    /**
     * @SWG\Delete(path="/gtfengines/{gtf_engine_id}",
     *   tags={"GtfEngines"},
     *   summary="delete GtfEngine",
     *   description="Request to delete GtfEngine",
     *   operationId="DELETE",
     *   produces={"application/xml", "application/json"},
     *   @SWG\Parameter(
     *     name="token",
     *     in="query",
     *     description="gtfengine token",
     *     required=true,
     *     type="string"
     *   ),
     * * @SWG\Parameter(
     *     name="gtf_engine_id",
     *     in="path",
     *     description="id of the gtfengine",
     *     required=true,
     *     type="integer",
     *     format = "int32"
     *   ),
     * @SWG\Response(
     *         response=200,
     *         description="Poprerties Response",
     *         @SWG\Schema(ref="#/definitions/gtfengines")
     *     )
     *  )
     */

    /**
     * delete gtfengine
     * @return id of gtfengine deleted or error object if a gtfengine is not deleted
     */
    function DELETE() {
        $aId = explode('|', $this->aValues['idList']);
        $aServeur = array();
        foreach ($aId as $id) {
            $this->aValues['my_vitis_id'] = $id;
            $oGtfEngine = new GtfEngine($this->aPath, $this->aValues, $this->aProperties, $this->oConnection);
            $oGtfEngine->GET();
            array_push($aServeur, $oGtfEngine->aFields['server_id']);
        }
        unset($this->aValues['my_vitis_id']);
        $aReturn = $this->genericDelete($this->aProperties['schema_gtf'], 'v_gtf_engine', 'gtf_engine_id');
        foreach ($aServeur as $serveur_id) {
            require_once("ServerClass.class.inc");
            $oCrontab = new ServerClass($oGtfEngine->aFields['server_id'], $this->oConnection->oBd, $this->aProperties['vas_home'], null, $this->aProperties);
            $oCrontab->deleteCrontabValuesEngine();
            $oCrontab->AddCrontabValuesEngines();
            $oCrontab->AddPropertiesFME();
        }
        return $aReturn['sMessage'];
    }

    /*
     * *	Insère les mots clefs liés au moteur GTF.
     */

    function insertGtfEngineTags() {
        require $this->sRessourcesFile;
        require_once("ServerClass.class.inc");
        $oCrontab = new ServerClass($this->aValues['server_id'], $this->oConnection->oBd, $this->aProperties['vas_home'], null, $this->aProperties);
        $oCrontab->deleteCrontabValuesEngine();
        $oCrontab->AddCrontabValuesEngines();
        $oCrontab->AddPropertiesFME();

        // Récupération des tags
        $aParams['sSchemaGtf'] = array('value' => $this->aProperties['schema_gtf'], 'type' => 'schema_name');
        $oPDOresult = $this->oConnection->oBd->executeWithParams($aSql['getTags'], $aParams);
        $aTags = array("default");
        while ($oTags = $this->oConnection->oBd->objetSuivant($oPDOresult)) {
            foreach (explode(",", trim($oTags->tags, ",")) as $sTag) {
                if ($sTag != "")
                    $aTags[] = $sTag;
            }
        }
        $aTags = array_unique($aTags);

        // truncate sur la table rt_tag
        $aParams['sSchemaGtf'] = array('value' => $this->aProperties['schema_gtf'], 'type' => 'schema_name');
        $this->oConnection->oBd->executeWithParams($aSql['truncateTags'], $aParams);

        // insertion des tags dans rt_tag
        $sSql = $aSql['insertTag'];
        foreach ($aTags as $sTag) {
            $sSql = $aSql['insertTag'];
            $aParams['sSchemaGtf'] = array('value' => $this->aProperties['schema_gtf'], 'type' => 'schema_name');
            $aParams['sTag'] = array('value' => $sTag, 'type' => 'string');
            $this->oConnection->oBd->executeWithParams($sSql, $aParams);
        }
    }

}

?>