<?php

/**
 * \file Orders.class.inc
 * \class Orders
 *
 * \author Yoann Perollet <yoann.perollet@veremes.com>.
 *
 * 	\brief This file contains the Orders php class
 *
 * This class defines Rest Api to Gtf orders
 * 
 */
require_once 'Gtf.class.inc';
require_once dirname($_SERVER['SCRIPT_FILENAME']) . '/class/vitis_lib/Connection.class.inc';
require_once 'Order.class.inc';
require_once dirname($_SERVER['SCRIPT_FILENAME']) . '/class/vmlib/BdDataAccess.inc';
require_once 'gtf_lib/FmeServer.class.inc';
require_once 'gtf_lib/FmeCloud.class.inc';

class Orders extends GTF {
    /**
     * @SWG\Definition(
     *   definition="/orders",
     *   allOf={
     *     @SWG\Schema(ref="#/definitions/orders")
     *   }
     * )
     * * @SWG\Tag(
     *   name="Orders",
     *   description="Operations about Orders"
     * )
     */

    /**
     * construct
     * @param type $aPath url of the request
     * @param type $aValues parameters of the request
     * @param type $properties properties
     * @param type $bShortcut false to reinit variables
     * @param type $oConnection connection object
     */
    function __construct($aPath, $aValues, $properties, $bShortcut = false, $oConnection = false) {
        parent::__construct($aPath, $aValues, $properties, $bShortcut, $oConnection);
        $this->aSelectedFields = Array("order_id", "priority_id", "workspace_id", "workspace_key", "label_name", "order_status_id", "user_id", "login", "period_id", "order_period_libelle", "wk_params", "result_url", $this->getDateSelectedFields('order_date', 'yeartosecond'), "log_url", $this->getDateSelectedFields('execution_date', 'yeartosecond'), "attempt", "email_notifications", "email_option_id", "deleted", "gtf_engine_id", "length_sec", "email_option_label", "order_status_label", "concat(gtf_engine_id, ' - ', name) as name", $this->getDateSelectedFields('minexecdate', 'yeartosecond'));
    }

    /**
     * @SWG\Get(path="/orders",
     *   tags={"Orders"},
     *   summary="Get Orders",
     *   description="Request to get Orders",
     *   operationId="GET",
     *   produces={"application/xml", "application/json"},
     *   @SWG\Parameter(
     *     name="token",
     *     in="query",
     *     description="user token",
     *     required=true,
     *     type="string"
     *   ),
     * @SWG\Parameter(
     *     name="order_by",
     *     in="query",
     *     description="list of ordering fields",
     *     required=false,
     *     type="string"
     *   ),
     * @SWG\Parameter(
     *     name="sort_order",
     *     in="query",
     *     description="sort order",
     *     required=false,
     *     type="string"
     *   ),
     * @SWG\Parameter(
     *     name="limit",
     *     in="query",
     *     description="number of element",
     *     required=false,
     *     type="integer",
     *     format="int32"
     *   ),
     * @SWG\Parameter(
     *     name="offset",
     *     in="query",
     *     description="index of first element",
     *     required=false,
     *     type="string",
     *     format="int32"
     *   ),
     * @SWG\Parameter(
     *     name="attributs",
     *     in="query",
     *     description="list of attributs",
     *     required=false,
     *     type="string"
     *   ),
     * @SWG\Parameter(
     *     name="filter",
     *     in="query",
     *     description="filter results",
     *     required=false,
     *     type="string"
     *   ),
     * @SWG\Parameter(
     *     name="distinct",
     *     in="query",
     *     description="delete duplicates",
     *     required=false,
     *     type="boolean"
     *   ),
     *   @SWG\Response(
     *         response=200,
     *         description="Poprerties Response",
     *         @SWG\Schema(ref="#/definitions/orders")
     *     )
     *  )
     */

    /**
     * @SWG\Get(path="/orders/running",
     *   tags={"Orders"},
     *   summary="Get running orders",
     *   description="Request to get all the running orders",
     *   operationId="GET",
     *   produces={"application/xml", "application/json"},
     *   @SWG\Parameter(
     *     name="token",
     *     in="query",
     *     description="user token",
     *     required=true,
     *     type="string"
     *   ),
     *   @SWG\Response(
     *         response=200,
     *         description="Properties Response",
     *         @SWG\Schema(ref="#/definitions/orders")
     *     )
     *  )
     */
    
    /**
     * get Orders
     * @return  Orders
     */
    function GET() {
        if (!empty($this->aPath[2]) && $this->aPath[2] == "running")
            $sMessage = $this->getRunningOrders();
        else {
            if (in_array('gtf_admin', $this->oConnection->aPrivileges))
                $aReturn = $this->genericGet($this->aProperties['schema_gtf'], "v_order_fast", "order_id");
            else {
                $oError = new VitisError(12, "You don't have the rights to get orders");
                $aXmlRacineAttribute['status'] = 0;
                $aReturn = array();
                $aReturn['sMessage'] = $oError->asDocument('', 'vitis', $this->aValues['sEncoding'], True, $aXmlRacineAttribute, $this->aValues['sSourceEncoding'], $this->aValues['output']);
            }
            $sMessage = $aReturn['sMessage'];
        }
        return $sMessage;
    }

    /**
     * @SWG\Put(path="/orders/gtfengine",
     *   tags={"Orders"},
     *   summary="Update gtf engine of orders",
     *   description="Request to update the gtf engine of orders",
     *   operationId="PUT",
     *   produces={"application/xml", "application/json"},
     *   @SWG\Parameter(
     *     name="token",
     *     in="query",
     *     description="category token",
     *     required=true,
     *     type="string"
     *   ),
     *   @SWG\Parameter(
     *     name="gtf_engine_id",
     *     in="query",
     *     description="id of the gtf engine",
     *     required=true,
     *     type="integer",
     *     format = "int32"
     *   ),
     *   @SWG\Parameter(
     *     name="idList",
     *     in="query",
     *     description="List of orders id",
     *     required=true,
     *     type="string",
     *   ),
     *   @SWG\Response(
     *         response=200,
     *         description="Poprerties Response",
     *         @SWG\Schema(ref="#/definitions/orders")
     *     )
     *  )
     */
    /**
     * @SWG\Put(path="/orders/reset",
     *   tags={"Orders"},
     *   summary="Reset orders",
     *   description="Request to reset the orders",
     *   operationId="PUT",
     *   produces={"application/xml", "application/json"},
     *   @SWG\Parameter(
     *     name="token",
     *     in="query",
     *     description="category token",
     *     required=true,
     *     type="string"
     *   ),
     *   @SWG\Parameter(
     *     name="idList",
     *     in="query",
     *     description="List of orders id",
     *     required=true,
     *     type="string",
     *   ),
     *   @SWG\Response(
     *         response=200,
     *         description="Poprerties Response",
     *         @SWG\Schema(ref="#/definitions/orders")
     *     )
     *  )
     */
    /**
     * @SWG\Put(path="/orders/activate",
     *   tags={"Orders"},
     *   summary="Activate orders",
     *   description="Request to activate orders",
     *   operationId="PUT",
     *   produces={"application/xml", "application/json"},
     *   @SWG\Parameter(
     *     name="token",
     *     in="query",
     *     description="user token",
     *     required=true,
     *     type="string"
     *   ),
     *   @SWG\Parameter(
     *     name="idList",
     *     in="query",
     *     description="List of orders id",
     *     required=true,
     *     type="string",
     *   ),
     *   @SWG\Response(
     *         response=200,
     *         description="Poprerties Response",
     *         @SWG\Schema(ref="#/definitions/orders")
     *     )
     *  )
     */
    /**
     * @SWG\Put(path="/orders/desactivate",
     *   tags={"Orders"},
     *   summary="Desactivate orders",
     *   description="Request to desactivate orders",
     *   operationId="PUT",
     *   produces={"application/xml", "application/json"},
     *   @SWG\Parameter(
     *     name="token",
     *     in="query",
     *     description="user token",
     *     required=true,
     *     type="string"
     *   ),
     *   @SWG\Parameter(
     *     name="idList",
     *     in="query",
     *     description="List of orders id",
     *     required=true,
     *     type="string",
     *   ),
     *   @SWG\Response(
     *         response=200,
     *         description="Poprerties Response",
     *         @SWG\Schema(ref="#/definitions/orders")
     *     )
     *  )
     */

    /**
     * @SWG\Put(path="/orders/{order_id}/stop",
     *   tags={"Orders"},
     *   summary="Stop order",
     *   description="Request to stop an order",
     *   operationId="PUT",
     *   produces={"application/xml", "application/json"},
     *   @SWG\Parameter(
     *     name="token",
     *     in="query",
     *     description="user token",
     *     required=true,
     *     type="string"
     *   ),
     *   @SWG\Parameter(
     *     name="order_id",
     *     in="path",
     *     description="Order id",
     *     required=true,
     *     type="integer",
     *     format = "int32"
     *   ),
     *   @SWG\Response(
     *         response=200,
     *         description="Poprerties Response",
     *         @SWG\Schema(ref="#/definitions/orders")
     *     )
     *  )
     */
    
    /**
     * update order
     */
    function PUT() {
        if (!empty($this->aPath[2])) {
            if (!empty($this->aPath[3])) {
                // Arrêt d'une demande en cours.
                if ($this->aPath[3] == 'stop')
                    $aReturn = $this->stopFmeOrder($this->aValues["my_vitis_id"]);
            }
            else {
                switch ($this->aPath[2]) {
                    case 'gtfengine':
                        $aReturn = $this->setOrderGtfEngine();
                        break;
                    case 'reset':
                        $aReturn = $this->resetOrder();
                        break;
                    case 'activate':
                        $aReturn = $this->activateOrder();
                        break;
                    case 'desactivate':
                        $aReturn = $this->desactivateOrder();
                        break;
                }
            }
            $aXmlRacineAttribute['status'] = $aReturn['status'];
            if ($aReturn['status'] == 1)
                $sMessage = $this->asDocument('', 'vitis', $this->aValues['sEncoding'], True, $aXmlRacineAttribute, $this->aValues['sSourceEncoding'], $this->aValues['output']);
            else {
                $oError = new VitisError($aReturn['error_code'], $aReturn['message']);
                $sMessage = $oError->asDocument('', 'vitis', $this->aValues['sEncoding'], True, $aXmlRacineAttribute, $this->aValues['sSourceEncoding'], $this->aValues['output']);
            }
            return $sMessage;
        }
    }

    /**
     * activateOrder
     */
    function activateOrder() {
        require $this->sRessourcesFile;
        $aReturn = array('status' => 1, 'message' => '');
        $aParams['sSchemaGtf'] = array('value' => $this->aProperties['schema_gtf'], 'type' => 'schema_name');
        $aParams['idList'] = array('value' => $this->aValues['idList'], 'type' => 'group');
        $oPDOresult = $this->oConnection->oBd->executeWithParams($aSql['activateOrder'], $aParams);
        if ($this->oConnection->oBd->enErreur())
            $aReturn = array('status' => 0, 'message' => $this->oConnection->oBd->getBDMessage(), 'error_code' => 1);
        return $aReturn;
    }

    /**
     * desactivateOrder
     */
    function desactivateOrder() {
        require $this->sRessourcesFile;
        $aReturn = array('status' => 1, 'message' => '');
        $aParams['sSchemaGtf'] = array('value' => $this->aProperties['schema_gtf'], 'type' => 'schema_name');
        $aParams['idList'] = array('value' => $this->aValues['idList'], 'type' => 'group');
        $oPDOresult = $this->oConnection->oBd->executeWithParams($aSql['desactivateOrder'], $aParams);
        if ($this->oConnection->oBd->enErreur())
            $aReturn = array('status' => 0, 'message' => $this->oConnection->oBd->getBDMessage(), 'error_code' => 1);
        return $aReturn;
    }

    /**
     * @SWG\Delete(path="/orders/",
     *   tags={"Orders"},
     *   summary="delete Order",
     *   description="Request to delete Order",
     *   operationId="DELETE",
     *   produces={"application/xml", "application/json"},
     *   @SWG\Parameter(
     *     name="token",
     *     in="query",
     *     description="order token",
     *     required=true,
     *     type="string"
     *   ),
     * * @SWG\Parameter(
     *     name="idList",
     *     in="query",
     *     description="id of the orders",
     *     required=true,
     *     type="string"
     *   ),
     * @SWG\Response(
     *         response=200,
     *         description="Poprerties Response",
     *         @SWG\Schema(ref="#/definitions/orders")
     *     )
     *  )
     */
    /**
     * @SWG\Delete(path="/orders/{order_id}",
     *   tags={"Orders"},
     *   summary="delete Order",
     *   description="Request to delete Order",
     *   operationId="DELETE",
     *   produces={"application/xml", "application/json"},
     *   @SWG\Parameter(
     *     name="token",
     *     in="query",
     *     description="order token",
     *     required=true,
     *     type="string"
     *   ),
     * * @SWG\Parameter(
     *     name="order_id",
     *     in="path",
     *     description="id of the order",
     *     required=true,
     *     type="integer",
     *     format = "int32"
     *   ),
     * @SWG\Response(
     *         response=200,
     *         description="Poprerties Response",
     *         @SWG\Schema(ref="#/definitions/orders")
     *     )
     *  )
     */
    /**
     * @SWG\Delete(path="/orders/clean/",
     *   tags={"Orders"},
     *   summary="clean Orders",
     *   description="Request to clean the orders",
     *   operationId="DELETE",
     *   produces={"application/xml", "application/json"},
     *   @SWG\Parameter(
     *     name="token",
     *     in="query",
     *     description="order token",
     *     required=true,
     *     type="string"
     *   ),
     * * @SWG\Parameter(
     *     name="idList",
     *     in="query",
     *     description="id of the orders",
     *     required=false,
     *     type="string"
     *   ),
     * @SWG\Response(
     *         response=200,
     *         description="Poprerties Response",
     *         @SWG\Schema(ref="#/definitions/orders")
     *     )
     *  )
     */

    /**
     * delete order
     * @return id of order deleted or error object if a order is not deleted
     */
    function DELETE() {
        if (!in_array('gtf_admin', $this->oConnection->aPrivileges)) {
            $oError = new VitisError(12, "You don't have the rights to delete orders");
            $aXmlRacineAttribute['status'] = 0;
            $aReturn['sMessage'] = $oError->asDocument('', 'vitis', $this->aValues['sEncoding'], True, $aXmlRacineAttribute, $this->aValues['sSourceEncoding'], $this->aValues['output']);
            $sMessage = $aReturn['sMessage'];
        }
        else if (!empty($this->aPath[2]) && $this->aPath[2] == "clean") {
            $aReturn = $this->cleanOrders();
            $aXmlRacineAttribute['status'] = $aReturn['status'];
            if ($aReturn['status'] == 1)
                $sMessage = $this->asDocument('', 'vitis', $this->aValues['sEncoding'], True, $aXmlRacineAttribute, $this->aValues['sSourceEncoding'], $this->aValues['output']);
            else {
                $oError = new VitisError($aReturn['error_code'], $aReturn['message']);
                $sMessage = $oError->asDocument('', 'vitis', $this->aValues['sEncoding'], True, $aXmlRacineAttribute, $this->aValues['sSourceEncoding'], $this->aValues['output']);
            }
        } else {
            // Stoppe les demandes en cours (Fme Server -> supprime le traitement et le répertoire des données).
            $this->deleteFmeOrders();
            //
            $aReturn = $this->genericDelete($this->aProperties['schema_gtf'], 'order', 'order_id');
            $sMessage = $aReturn['sMessage'];
        }
        return $sMessage;
    }

    /**
     * cleanOrders
     */
    function cleanOrders() {
        require $this->sRessourcesFile;
        $aReturn = array('status' => 1, 'message' => '');
        // Liste des demandes.
        if (!empty($this->aValues['idList'])) {
            $sSql = $aSql['getOrders'];
            $aParams['idList'] = array('value' => $this->aValues['idList'], 'type' => 'group');
        } else
            $sSql = $aSql['getAllOrders'];
        $aParams['sSchemaGtf'] = array('value' => $this->aProperties['schema_gtf'], 'type' => 'schema_name');
        $oPDOresult = $this->oConnection->oBd->executeWithParams($sSql, $aParams);
        if ($this->oConnection->oBd->enErreur())
            $aReturn = array('status' => 0, 'message' => $this->oConnection->oBd->getBDMessage(), 'error_code' => 1);
        else {
            while ($aDemande = $this->oConnection->oBd->ligneSuivante($oPDOresult)) {
                $aFolder = explode("/", $aDemande['log_url']);
                $sFolder = $this->aProperties['dir_export'] . "/gtf/" . $aFolder[0];
                clearDir($sFolder);
            }
        }
        // Supprime les demandes dont la colonne "deleted" = true.
        $this->oConnection->oBd->delete($this->aProperties['schema_gtf'], 'order', 'deleted', 'true', 'bool');
        return $aReturn;
    }

    /**
     * setOrderGtfEngine
     */
    function setOrderGtfEngine() {
        require $this->sRessourcesFile;
        $aReturn = array('status' => 1, 'message' => '');
        $aParams['sSchemaGtf'] = array('value' => $this->aProperties['schema_gtf'], 'type' => 'schema_name');
        $aParams['gtf_engine_id'] = array('value' => $this->aValues['gtf_engine_id'], 'type' => 'number');
        $aParams['idList'] = array('value' => $this->aValues['idList'], 'type' => 'group');
        $oPDOresult = $this->oConnection->oBd->executeWithParams($aSql['setOrderGtfEngine'], $aParams);
        if ($this->oConnection->oBd->enErreur())
            $aReturn = array('status' => 0, 'message' => $this->oConnection->oBd->getBDMessage(), 'error_code' => 1);
        return $aReturn;
    }

    /**
     * resetOrder
     */
    function resetOrder() {
        require $this->sRessourcesFile;
        // Supprime l'ancien résultat.
        $aParams['sSchemaGtf'] = array('value' => $this->aProperties['schema_gtf'], 'type' => 'schema_name');
        $aParams['idList'] = array('value' => $this->aValues['idList'], 'type' => 'group');
        $oPDOresult = $this->oConnection->oBd->executeWithParams($aSql['getOrders'], $aParams);
        if ($this->oConnection->oBd->enErreur())
            $aReturn = array('status' => 0, 'message' => $this->oConnection->oBd->getBDMessage(), 'error_code' => 1);
        else {
            while ($aDemande = $this->oConnection->oBd->ligneSuivante($oPDOresult)) {
                $aFolder = explode("/", $aDemande['log_url']);
                $sFolder = $this->aProperties['dir_export'] . "/gtf/" . $aFolder[0];
                clearDir($sFolder);
            }
        }
        // Réinitialise la demande.
        $aOrderId = explode('|', $this->aValues['idList']);
        foreach ($aOrderId as $iOrderId) {
            // Vérifie si le moteur déja assigné est actif.
            $aParams = array();
            $aParams['sSchemaGtf'] = array('value' => $this->aProperties['schema_gtf'], 'type' => 'schema_name');
            $aParams['order_id'] = array('value' => $iOrderId, 'type' => 'number');
            $oPDOresult = $this->oConnection->oBd->executeWithParams($aSql['getOrderEngineStatus'], $aParams);
            if ($this->oConnection->oBd->enErreur()) {
                $oError = new VitisError(1, $this->oConnection->oBd->getBDMessage());
                $aXmlRacineAttribute['status'] = 0;
                $sMessage = $oError->asDocument('', 'vitis', $this->aValues['sEncoding'], True, $aXmlRacineAttribute, $this->aValues['sSourceEncoding'], $this->aValues['output']);
            }
            else if ($this->oConnection->oBd->nombreLigne($oPDOresult) > 0) {
                $aGtfEngine = $this->oConnection->oBd->ligneSuivante($oPDOresult);
                // Si le moteur déja assigné est inactif, il faut en sélectionner un autre.
                if ($aGtfEngine['enabled'] === false) {
                    // Récupère l'id du projet de la demande.
                    $aParams = array();
                    $aParams['sSchemaGtf'] = array('value' => $this->aProperties['schema_gtf'], 'type' => 'schema_name');
                    $aParams['order_id'] = array('value' => $iOrderId, 'type' => 'number');
                    $oPDOresult = $this->oConnection->oBd->executeWithParams($aSql['getOrderWorkspace'], $aParams);
                    if ($this->oConnection->oBd->enErreur()) {
                        $oError = new VitisError(1, $this->oConnection->oBd->getBDMessage());
                        $aXmlRacineAttribute['status'] = 0;
                        $sMessage = $oError->asDocument('', 'vitis', $this->aValues['sEncoding'], True, $aXmlRacineAttribute, $this->aValues['sSourceEncoding'], $this->aValues['output']);
                    }
                    else if ($this->oConnection->oBd->nombreLigne($oPDOresult) > 0) {
                        $aWorkspace = $this->oConnection->oBd->ligneSuivante($oPDOresult);
                        $this->getIdEngine($iOrderId, 'order', $aWorkspace['workspace_id']);
                        // Met à jour le n° du moteur.
                        $aParams = array();
                        $aParams['sSchemaGtf'] = array('value' => $this->aProperties['schema_gtf'], 'type' => 'schema_name');
                        $aParams['order_id'] = array('value' => $iOrderId, 'type' => 'number');
                        $aParams['gtf_engine_id'] = array('value' => $this->aValues['gtf_engine_id'], 'type' => 'number');
                        $oPDOresult = $this->oConnection->oBd->executeWithParams($aSql['setUserOrderEngineId'], $aParams);
                        if ($this->oConnection->oBd->enErreur()) {
                            $oError = new VitisError(1, $this->oConnection->oBd->getBDMessage());
                            $aXmlRacineAttribute['status'] = 0;
                            $sMessage = $oError->asDocument('', 'vitis', $this->aValues['sEncoding'], True, $aXmlRacineAttribute, $this->aValues['sSourceEncoding'], $this->aValues['output']);
                        }
                    }
                }
            }
            // Réinitialise la demande.
            $aParams = array();
            $aParams['sSchemaGtf'] = array('value' => $this->aProperties['schema_gtf'], 'type' => 'schema_name');
            $aParams['order_id'] = array('value' => $iOrderId, 'type' => 'number');
            $oPDOresult = $this->oConnection->oBd->executeWithParams($aSql['resetOrder'], $aParams);
            if ($this->oConnection->oBd->enErreur()) {
                $aReturn = array('status' => 0, 'message' => $this->oConnection->oBd->getBDMessage(), 'error_code' => 1);
                return $aReturn;
            }
        }
        $aReturn = array('status' => 1, 'message' => '');
        return $aReturn;
    }

    /*
     * function getIdEngine
     * \brief function that assigns each request processing an engine number. This engine number will be defined by the algorithm associated to the treatment.
     * \param $iUserOrderId Identifier of the request
     * \param $sTable Name of the table.
     * \param $iWorkspaceId Identifier of the treatment.
     */
    function getIdEngine($iUserOrderId, $sTable, $iWorkspaceId) {
        require $this->sRessourcesFile;
        // Récupération de l'algorithme
        $aParams['sSchemaGtf'] = array('value' => $this->aProperties['schema_gtf'], 'type' => 'schema_name');
        $aParams['iWorkspaceId'] = array('value' => $iWorkspaceId, 'type' => 'number');
        $oPDOresult = $this->oConnection->oBd->executeWithParams($aSql['getEnginesId'], $aParams);
        $this->oConnection->oBd->cleanMessage();
        $aEnginesId = array();
        while ($oEnginesId = $this->oConnection->oBd->objetSuivant($oPDOresult))
            $aEnginesId[] = $oEnginesId->gtf_engine_id;
        $this->oConnection->oBd->fermeResultat();

        if (count($aEnginesId) == 0) {
            // Le tag correspondant n'a pas été trouvé, nous allons cherché la liste des moteur ayant comme tag |default|
            $aParams['sSchemaGtf'] = array('value' => $this->aProperties['schema_gtf'], 'type' => 'schema_name');
            $oPDOresult = $this->oConnection->oBd->executeWithParams($aSql['getEnginesDefaultTag'], $aParams);
            $this->oConnection->oBd->cleanMessage();
            $aEnginesId = array();
            while ($oEnginesId = $this->oConnection->oBd->objetSuivant($oPDOresult))
                $aEnginesId[] = $oEnginesId->gtf_engine_id;
            $this->oConnection->oBd->fermeResultat();
            if (count($aEnginesId) > 0)
                writeToErrorLog(str_replace('[iUserOrderId]', $iUserOrderId, ERROR_0006));
        }
        // Si la requête est en erreur ou Si aucun moteur n'a été trouvé.
        if ($this->oConnection->oBd->erreurRencontree || count($aEnginesId) == 0) {
            $aGtfEngineList = explode(",", $this->aProperties["engines_list"]);
            // Aucun tag ne correspond
            $iEngine = $aGtfEngineList[0];
            if ($iEngine != "")
                writeToErrorLog(str_replace('[aGtfEngineList[0]]', $iEngine, str_replace('[iUserOrderId]', $iUserOrderId, ERROR_0007)));
        }else {
            $iEngine = $this->returnEngineId($iUserOrderId, $aEnginesId);
        }

        if ($iEngine == "") {
            writeToErrorLog(str_replace('[iUserOrderId]', $iUserOrderId, ERROR_0008));
            $this->aValues['gtf_engine_id'] = 1;
        } else {
            $this->aValues['gtf_engine_id'] = $iEngine;
        }
    }

    /*
     * function returnEngineId
     * \param $oAlgoIndex Object of the algorithm.
     * \param $aGtfEngineList Array that corresponds to the list of available engines numbers.
     * \return Returns the engine number.
     */

    function returnEngineId($iUserOrderId, $aEnginesId) {
        $iEngine = "";
        if (count($aEnginesId) > 0) {
            $sRowCount = $iUserOrderId % count($aEnginesId);
            $iEngine = $aEnginesId[$sRowCount];
        }
        return $iEngine;
    }
    
    /**
     * function getRunningOrders
     */
    function getRunningOrders() {
        require $this->sRessourcesFile;
        // Liste des demandes.
        $aParams = array();
        $aParams['sSchemaGtf'] = array('value' => $this->aProperties['schema_gtf'], 'type' => 'schema_name');
        $oPDOresult = $this->oConnection->oBd->executeWithParams($aSql['getRunningOrders'], $aParams);
        if ($this->oConnection->oBd->enErreur()) {
            $oError = new VitisError(1, $this->oConnection->oBd->getBDMessage());
            $aXmlRacineAttribute['status'] = 0;
            $sMessage = $oError->asDocument('', 'vitis', $this->aValues['sEncoding'], True, $aXmlRacineAttribute, $this->aValues['sSourceEncoding'], $this->aValues['output']);
        }
        else {
            $aOrders = array();
            while ($aOrder = $this->oConnection->oBd->ligneSuivante($oPDOresult))
                $aOrders['orders'][] = $aOrder['order_id'];
            $this->aFields = $aOrders;
            $aXmlRacineAttribute['status'] = 1;
            $sMessage = $this->asDocument('', 'vitis', $this->aValues['sEncoding'], True, $aXmlRacineAttribute, $this->aValues['sSourceEncoding'], $this->aValues['output']);
        }
        return $sMessage;
    }
    
    /**
     * function deleteFmeOrders
     */
    function deleteFmeOrders() {
        require $this->sRessourcesFile;
        // Infos des demandes à supprimer.
        $aParams['sSchemaGtf'] = array('value' => $this->aProperties['schema_gtf'], 'type' => 'schema_name');
        $aParams['idList'] = array('value' => $this->aValues['idList'], 'type' => 'group');
        $oPDOresult = $this->oConnection->oBd->executeWithParams($aSql['getOrdersToDelete'], $aParams);
        if (!$this->oConnection->oBd->enErreur()) {
            while ($aDemande = $this->oConnection->oBd->ligneSuivante($oPDOresult)) {
                $aRunningJob = array();
                if ($aDemande['order_status_id'] == 5) {
                    // Paramètres de la demande si c'est un traitement asynchrone sur Fme server.
                    if ($aDemande['fme_engine_type_id'] != "desktop") {
                        $aRunningJobs = $this->getFmeServerRunningJobs($aDemande['gtf_engine_id']);
                        $aRunningJob = $this->getFmeServerRunningJob($aDemande['order_id'], $aRunningJobs);
                    }
                    // Stoppe la demande en cours.
                    $this->stopFmeOrder($aDemande['order_id']);
                }
                // Supprime le traitement sur Fme Server.
                if ($aDemande['fme_engine_type_id'] != "desktop")
                    $this->deleteFmeServerOrder($aDemande['order_id'], $aDemande['gtf_engine_id'], $aRunningJob);
            }
        }
    }
    
    /**
     * function deleteFmeServerOrder
     * \param $iOrderId Id of the Gtf order.
     * \param $iGtfEngineId Id of the Gtf engine.
     * \param $aRunningJob A running job on Fme Server.
     */
    function deleteFmeServerOrder($iOrderId, $iGtfEngineId, $aRunningJob = array()) {
        $sEngineLogFilePath = $this->aProperties['vas_home'] . '/log/engines/' . date($this->aProperties['log_period']) . '/engine_' . $iGtfEngineId . '/engine.log';
        if (!empty($aRunningJob)) {
            // Demande en cours -> informations de connexion sur Fme Server dans la liste des traitements asynchrones.
            $sFmeServerUrl = $aRunningJob['fme_server_url'];
            $sLogin = $aRunningJob['login'];
            $sPassword = $aRunningJob['password'];
            $sResource = $aRunningJob['resource'];
            $sPath = $aRunningJob['path'];
            $iJobId = $aRunningJob['job_id'];
        }
        else {
            // Demande en attente ou terminée -> informations de connexion sur Fme Server dans la base.
            require $this->sRessourcesFile;
            $aParams = array();
            $aParams['sSchemaGtf'] = array('value' => $this->aProperties['schema_gtf'], 'type' => 'schema_name');
            $aParams['order_id'] = array('value' => $iOrderId, 'type' => 'number');
            $oPDOresult = $this->oConnection->oBd->executeWithParams($aSql['getFmeServerOrder'], $aParams);
            if (!$this->oConnection->oBd->enErreur()) {
                $aOrder = $this->oConnection->oBd->ligneSuivante($oPDOresult);
                require $aOrder['engines_home'] . '/' . 'string.inc';
                // Si Fme Cloud -> l'instance de Fme Server doit être démarrée.
                if ($aOrder['fme_engine_type_id'] == 'fme_cloud') {
                    // Ip de l'instance Fme Server.
                    $oFmeCloud = new FmeCloud($aOrder['server_url'], $aOrder['fme_cloud_api_token']);
                    $oFmeCloud->sLogFilePath = $sEngineLogFilePath;
                    // Adresse Ip publique de l'instance de FME Server.
                    $aFmeServerInstances = $oFmeCloud->serviceRequest('instances');
                    if (!empty($aFmeServerInstances)) {
                        foreach($aFmeServerInstances as $oInstance) {
                            if ($oInstance->name == $aOrder['fme_server_instance_name'])
                                $oFmeServerInstance = $oInstance;
                        }
                        if (empty($oFmeServerInstance)) {
                            // Aucune instace active.
                            writeToLog(INFO_FME_CLOUD_INSTANCE_NOT_FOUND, $sEngineLogFilePath);
                        }
                        else {
                            // Statut de l'instance.
                            writeToLog(INFO_FME_CLOUD_INSTANCE_STATUS . $oFmeServerInstance->state, $sEngineLogFilePath);
                            $sFmeServerUrl = 'https://' . $oFmeServerInstance->public_ip;
                            // Pas de suppression possible si l'instance n'est pas démarrée.
                            if ($oFmeServerInstance->state != "RUNNING")
                                return false;
                        }
                    }
                }
                $sLogin = $aOrder['login'];
                $sPassword = $aOrder['password'];
                $sResource = 'FME_SHAREDRESOURCE_TEMP';
                if (!empty($aOrder['log_url']))
                    $sPath = dirname($this->aProperties["gtf_instance_id"] . '/' . pathinfo($aOrder['log_url'], PATHINFO_DIRNAME));
                $iJobId = $aOrder['pid'];
            }
        }
        // Suppression du traitement et du répertoire sur Fme Server.
        if (!empty($sFmeServerUrl) && !empty($sLogin) && !empty($sPassword)) {
            $oFmeServer = new FmeServer($sFmeServerUrl, $sLogin, $sPassword);
            $oFmeServer->sLogFilePath = $sEngineLogFilePath;
            if (!empty($iJobId))
                if ($oFmeServer->deleteJob($iJobId) === false)
                    return false;
            // Supprime le répertoire des ressources de la demande sur Fme Server.
            if (!empty($sResource) && !empty($sPath))
                if ($oFmeServer->serviceRequest('resources/connections/' . $sResource . '/filesys/' . $sPath, 'delete') === false)
                    return false;
        }
    }
    
    /**
     * function stopFmeOrder
     * \param $iOrderId Id of the Gtf order.
     */
    function stopFmeOrder($iOrderId) {
        require $this->sRessourcesFile;
        // Informations de la demande à stopper.
        $aParams = array();
        $aParams['sSchemaGtf'] = array('value' => $this->aProperties['schema_gtf'], 'type' => 'schema_name');
        $aParams['order_id'] = array('value' => $iOrderId, 'type' => 'number');
        $oPDOresult = $this->oConnection->oBd->executeWithParams($aSql['getOrderFmeEngineType'], $aParams);
        if (!$this->oConnection->oBd->enErreur()) {
            $aOrder = $this->oConnection->oBd->ligneSuivante($oPDOresult);
            if ($aOrder['fme_engine_type_id'] == 'desktop') {
                if ($this->stopFmeDesktopOrderProcess($iOrderId) === false)
                    $sErrorMessage = ERROR_0037;
            }
            else {
                if ($this->stopFmeServerOrder($iOrderId) === false)
                    $sErrorMessage = ERROR_0038;
            }
            // Message de retour pour l'API Rest.
            if (empty($sErrorMessage)) {
                $aReturn = array('status' => 1, 'message' => '');
                // Mise à jour du statut de la demande (stoppé).
                $aParams = array();
                $aParams['sSchemaGtf'] = array('value' => $this->aProperties['schema_gtf'], 'type' => 'schema_name');
                $aParams['order_id'] = array('value' => $iOrderId, 'type' => 'number');
                $aParams['order_status_id'] = array('value' => 7, 'type' => 'number');
                $this->oConnection->oBd->executeWithParams($aSql['updateOrderStatus'], $aParams);
            }
            else
                $aReturn = array('status' => 0, 'message' => $sErrorMessage, 'error_code' => 17);
            return $aReturn;
        }
    }
    
    /**
     * function stopFmeDesktopOrderProcess
     * \param $iOrderId Id of the Gtf order.
     */
    function stopFmeDesktopOrderProcess($iOrderId) {
        require $this->sRessourcesFile;
        $aParams = array();
        $aParams['sSchemaGtf'] = array('value' => $this->aProperties['schema_gtf'], 'type' => 'schema_name');
        $aParams['order_id'] = array('value' => $iOrderId, 'type' => 'number');
        $oPDOresult = $this->oConnection->oBd->executeWithParams($aSql['getGtfEngineHomeFromOrderId'], $aParams);
        if (!$this->oConnection->oBd->enErreur()) {
            $aGtfEngineHome = $this->oConnection->oBd->ligneSuivante($oPDOresult);
            $sPidFilePath = $aGtfEngineHome['engines_home'] . '/' . 'pid_' . $aGtfEngineHome['gtf_engine_id'] . '.txt';
            if (file_exists($sPidFilePath)) {
                $iPid = file_get_contents($sPidFilePath);
                if (is_numeric($iPid)) {
                    if (strtolower(substr(PHP_OS, 0, 3)) == 'win') {
                        // Supprime le processus "engine.exe".
                        $this->killProcess($iPid);
                        // Tue tous les processus liés à une demande et un traitement sur Fme Desktop.
                        $aProcessName = array('cmd.exe', 'php.exe', 'cmd.exe', 'fme.exe');
                        foreach ($aProcessName as $sProcessName) {
                            $iPid = $this->getChildProcess($iPid, $sProcessName);
                            if ($iPid !== false)
                                $this->killProcess($iPid);
                        }
                    }
                    else {
                        // Supprime tous les processus enfants.
                        $aResult = array();
                        $sResultCommande = exec('pkill -9 -P ' . $iPid, $aResult, $iResult);
                        // Supprime le processus du moteur.
                        $this->killProcess($iPid);
                    }
                }
                // Suppression du fichier contenant le pid du processus "engine.exe".
                unlink($sPidFilePath);
            }
        }
    }    
    
    /**
     * function killProcess
     * \param $iPid Id of the process.
     */
    function killProcess($iPid) {
        $aResult = array();
        if (strtolower(substr(PHP_OS, 0, 3)) == 'win')
            $sResultCommande = exec('taskkill /PID ' . $iPid . ' /F', $aResult, $iResult);
        else
            $sResultCommande = exec('kill ' . $iPid, $aResult, $iResult);
        // Log si erreur.
        if ($iResult !== 0)
            writeToErrorLog(ERROR_0035 . ' (pid = ' . $iPid . ').');
    }
    
    /**
     * function getChildProcess
     * \param $iParentPid Id of the parent process.
     * \param $sProcessName Name of the process.
     */
    function getChildProcess($iParentPid, $sProcessName) {
        $aResult = array();
        if (strtolower(substr(PHP_OS, 0, 3)) == 'win')
            $sResultCommande = exec('wmic process where "ParentProcessId=' . $iParentPid . ' AND Name=\'' . $sProcessName . '\'" get ProcessId', $aResult, $iResult);
        //else
            //$sResultCommande = exec('' . $iParentPid, $aResult, $iResult);
            //ps -o ppid=
        // Log si erreur.
        if ($iResult !== 0)
            writeToErrorLog(ERROR_0036 . ' (pid = ' . $iPid . ').');
        else if ($iResult == 0 && !empty($aResult) && is_numeric($aResult[1]))
            return $aResult[1];
        else
            return false;
    }
    
    /**
     * function stopFmeServerOrder
     * \param $iOrderId Id of the Gtf order.
     */
    function stopFmeServerOrder($iOrderId) {
        require $this->sRessourcesFile;
        // Récupère le répertoire des moteurs de GTF.
        $aParams = array();
        $aParams['sSchemaGtf'] = array('value' => $this->aProperties['schema_gtf'], 'type' => 'schema_name');
        $aParams['order_id'] = array('value' => $iOrderId, 'type' => 'number');
        $oPDOresult = $this->oConnection->oBd->executeWithParams($aSql['getGtfEngineHomeFromOrderId'], $aParams);
        if (!$this->oConnection->oBd->enErreur()) {
            $aGtfEngineHome = $this->oConnection->oBd->ligneSuivante($oPDOresult);
            // Stoppe le traitement sur Fme Server.
            $sAsyncOrdersFilePath = $aGtfEngineHome['engines_home'] . '/' . 'gtf_engine' . $aGtfEngineHome['gtf_engine_id'] . '_jobid.txt';
            if (file_exists($sAsyncOrdersFilePath)) {
                $aRunningJobs = unserialize(file_get_contents($sAsyncOrdersFilePath));
                $aRunningJob = $this->getFmeServerRunningJob($iOrderId, $aRunningJobs);
                if ($aRunningJob !== false && !empty($aRunningJob['job_id'])) {
                    $sEngineLogFilePath = $this->aProperties['vas_home'] . '/log/engines/' . date($this->aProperties['log_period']) . '/engine_' . $aGtfEngineHome['gtf_engine_id'] . '/engine.log';
                    $oFmeServer = new FmeServer($aRunningJob['fme_server_url'], $aRunningJob['login'], $aRunningJob['password'], 'day', 1);
                    $oFmeServer->sLogFilePath = $sEngineLogFilePath;
                    $oFmeServer->stopJob($aRunningJob['job_id']);
                    // Supprime le traitement dans la liste des demandes asynchrones.
                    $iJobIndex = null;
                    foreach ($aRunningJobs as $iIndex => $aRunningJob) {
                        if ($aRunningJob['order_id'] == $iOrderId) {
                            $iJobIndex = $iIndex;
                            break;
                        }
                    }
                    if ($iJobIndex !== null) {
                        // Sauve la liste des demandes asynchrones en cours.
                        $aRunningJob = array_splice($aRunningJobs, $iJobIndex, 1)[0];
                        if (empty($aRunningJobs))
                            unlink($sAsyncOrdersFilePath);
                        else
                            file_put_contents($sAsyncOrdersFilePath, serialize($aRunningJobs));
                        // Log.
                        require $aGtfEngineHome['engines_home'] . '/' . 'string.inc';
                        writeToLog(INFO_FME_SERVER_ASYNCHRONOUS_JOB_REMOVED_FROM_LIST . " (demande $iOrderId).", $sEngineLogFilePath);
                    }
                }
            }
        }
    }
    
    /**
     * Retourne les infos d'un traitement asynchrones en cours d'exécution.
     * @param {number} $iOrderId I de la demande en cours.
     * @param {array} $aRunningJobs Tableau contenant la liste des traitements asynchrones en cours d'exécution.
     * return array
     */
    function getFmeServerRunningJob($iOrderId, $aRunningJobs) {
        foreach ($aRunningJobs as $aRunningJob) {
            if ($aRunningJob['order_id'] == $iOrderId)
                return $aRunningJob;
        }
        return false;
    }
    
    /**
     * Retourne la liste des traitements asynchrones en cours d'exécution d'un moteur GTF.
     * @param {number} $iGtfEngineId Id du moteur GTF.
     * return array
     */
    function getFmeServerRunningJobs($iGtfEngineId) {
        require $this->sRessourcesFile;
        // Récupère le répertoire des moteurs de GTF.
        $aRunningJobs = array();
        $aParams = array();
        $aParams['sSchemaGtf'] = array('value' => $this->aProperties['schema_gtf'], 'type' => 'schema_name');
        $aParams['iGtfEngineId'] = array('value' => $iGtfEngineId, 'type' => 'number');
        $oPDOresult = $this->oConnection->oBd->executeWithParams($aSql['getGtfEngineHomeFromEngineId'], $aParams);
        if (!$this->oConnection->oBd->enErreur()) {
            $aGtfEngineHome = $this->oConnection->oBd->ligneSuivante($oPDOresult);
            $sAsyncOrdersFilePath = $aGtfEngineHome['engines_home'] . '/' . 'gtf_engine' . $aGtfEngineHome['gtf_engine_id'] . '_jobid.txt';
            if (file_exists($sAsyncOrdersFilePath))
                $aRunningJobs = unserialize(file_get_contents($sAsyncOrdersFilePath));
        }
        return $aRunningJobs;
    }
}
?>