<?php

/**
 * \file Servers.class.inc
 * \class Servers
 *
 * \author Yoann Perollet <yoann.perollet@veremes.com>.
 *
 * 	\brief This file contains the Modes php class
 *
 * This class defines Rest Api to Vitis Servers
 * 
 */
require_once 'Gtf.class.inc';
require_once dirname($_SERVER['SCRIPT_FILENAME']) . '/class/vitis_lib/Connection.class.inc';
require_once 'Server.class.inc';
require_once dirname($_SERVER['SCRIPT_FILENAME']) . '/class/vmlib/BdDataAccess.inc';

class Servers extends GTF {
    /**
     * @SWG\Definition(
     *   definition="/servers",
     *   allOf={
     *     @SWG\Schema(ref="#/definitions/servers")
     *   }
     * )
     * * @SWG\Tag(
     *   name="Servers",
     *   description="Operations about Server"
     * )
     */

    /**
     * construct
     * @param type $aPath url of the request
     * @param type $aValues parameters of the request
     * @param type $properties properties
     * @param type $bShortcut false to reinit variables
     * @param type $oConnection connection object
     */
    function __construct($aPath, $aValues, $properties, $bShortcut = false, $oConnection = false) {
        parent::__construct($aPath, $aValues, $properties, $bShortcut, $oConnection);
        $this->aSelectedFields = Array('server_id', 'hostname', 'scheduler_network_home', 'engines_home');
    }

    /**
     * @SWG\Get(path="/servers",
     *   tags={"Servers"},
     *   summary="Get Server",
     *   description="Request to get Server",
     *   operationId="GET",
     *   produces={"application/xml", "application/json", "application/x-vm-json"},
     *   @SWG\Parameter(
     *     name="token",
     *     in="query",
     *     description="server token",
     *     required=true,
     *     type="string"
     *   ),
     * @SWG\Parameter(
     *     name="order_by",
     *     in="query",
     *     description="list of ordering fields",
     *     required=false,
     *     type="string"
     *   ),
     * @SWG\Parameter(
     *     name="sort_order",
     *     in="query",
     *     description="sort order",
     *     required=false,
     *     type="string"
     *   ),
     * @SWG\Parameter(
     *     name="limit",
     *     in="query",
     *     description="number of element",
     *     required=false,
     *     type="integer",
     *     format="int32"
     *   ),
     * @SWG\Parameter(
     *     name="offset",
     *     in="query",
     *     description="index of first element",
     *     required=false,
     *     type="string",
     *     format="int32"
     *   ),
     * @SWG\Parameter(
     *     name="attributs",
     *     in="query",
     *     description="list of attributs",
     *     required=false,
     *     type="string"
     *   ),
     * @SWG\Parameter(
     *     name="filter",
     *     in="query",
     *     description="filter results",
     *     required=false,
     *     type="string"
     *   ),
     * @SWG\Parameter(
     *     name="distinct",
     *     in="query",
     *     description="delete duplicates",
     *     required=false,
     *     type="boolean"
     *   ),
     *   @SWG\Response(
     *         response=200,
     *         description="Poprerties Response",
     *         @SWG\Schema(ref="#/definitions/servers")
     *     )
     *  )
     */

    /**
     * get Servers
     * @return  Servers
     */
    function GET() {
        $aReturn = $this->genericGet($this->aProperties['schema_gtf'], "server", "server_id");
        return $aReturn['sMessage'];
    }

    /**
     * @SWG\Post(path="/servers",
     *   tags={"Servers"},
     *   summary="Add Server",
     *   description="Request to add server",
     *   operationId="POST",
     *   produces={"application/xml", "application/json"},
     *   @SWG\Parameter(
     *     name="token",
     *     in="formData",
     *     description="server token",
     *     required=true,
     *     type="string"
     *   ),
     *   @SWG\Parameter(
     *     name="hostname",
     *     in="formData",
     *     description="hostname of server",
     *     required=true,
     *     type="integer",
     *     format="int32"
     *   ),
     *   @SWG\Parameter(
     *     name="scheduler_network_home",
     *     in="formData",
     *     description="scheduler network home of the server",
     *     required=true,
     *     type="string"
     *   ),
     *   @SWG\Parameter(
     *     name="engines_home",
     *     in="formData",
     *     description="engines home of the server",
     *     required=true,
     *     type="string"
     *   ),
     *   @SWG\Response(
     *         response=200,
     *         description="Poprerties Response",
     *         @SWG\Schema(ref="#/definitions/servers")
     *     )
     *  )
     */

    /**
     * insert server
     * @return id of the server created
     */
    function POST() {
        $aReturn = $this->genericPost($this->aProperties['schema_gtf'], 'server', $this->aProperties['schema_gtf'] . '.seq_common', 'server_id');
        if ($aReturn['sStatus'] == 1) {
            $aXmlRacineAttribute['status'] = 1;
            $sMessage = $this->asDocument('', 'vitis', $this->aValues['sEncoding'], True, $aXmlRacineAttribute, $this->aValues['sSourceEncoding'], $this->aValues['output']);
            $oServer = new Server($this->aPath, $this->aValues, $this->aProperties, $this->oConnection);
            $oServer->GET();
        } else {
            $sMessage = $aReturn['sMessage'];
        }
        return $sMessage;
    }

    /**
     * @SWG\Put(path="/servers/{server_id}",
     *   tags={"Servers"},
     *   summary="Update Server",
     *   description="Request to update Server",
     *   operationId="PUT",
     *   produces={"application/xml", "application/json"},
     *   @SWG\Parameter(
     *     name="token",
     *     in="query",
     *     description="server token",
     *     required=true,
     *     type="string"
     *   ),
     * * @SWG\Parameter(
     *     name="server_id",
     *     in="path",
     *     description="id of the server",
     *     required=true,
     *     type="integer",
     *     format = "int32"
     *   ),
     *   @SWG\Parameter(
     *     name="hostname",
     *     in="query",
     *     description="hostname of server",
     *     required=true,
     *     type="integer",
     *     format="int32"
     *   ),
     *   @SWG\Parameter(
     *     name="scheduler_network_home",
     *     in="query",
     *     description="scheduler network home of the server",
     *     required=true,
     *     type="string"
     *   ),
     *   @SWG\Parameter(
     *     name="engines_home",
     *     in="query",
     *     description="engines home of the server",
     *     required=true,
     *     type="string"
     *   ),
     *   @SWG\Response(
     *         response=200,
     *         description="Poprerties Response",
     *         @SWG\Schema(ref="#/definitions/servers")
     *     )
     *  )
     */

    /**
     * update server
     * @return id of the server if ok error object if ko
     */
    function PUT() {
        $aReturn = $this->genericPut($this->aProperties['schema_gtf'], 'server', 'server_id');
        if ($aReturn['sStatus'] == 1) {
            $aXmlRacineAttribute['status'] = 1;
            $sMessage = $this->asDocument('', 'vitis', $this->aValues['sEncoding'], True, $aXmlRacineAttribute, $this->aValues['sSourceEncoding'], $this->aValues['output']);
            $oServer = new Server($this->aPath, $this->aValues, $this->aProperties, $this->oConnection);
            $oServer->GET();
            //
            $this->insertCrontab();
        } else {
            $sMessage = $aReturn['sMessage'];
        }
        return $sMessage;
    }

    /**
     * @SWG\Delete(path="/servers/",
     *   tags={"Servers"},
     *   summary="delete Server",
     *   description="Request to delete Server",
     *   operationId="DELETE",
     *   produces={"application/xml", "application/json"},
     *   @SWG\Parameter(
     *     name="token",
     *     in="query",
     *     description="server token",
     *     required=true,
     *     type="string"
     *   ),
     * * @SWG\Parameter(
     *     name="idList",
     *     in="query",
     *     description="id of the servers",
     *     required=true,
     *     type="string"
     *   ),
     * @SWG\Response(
     *         response=200,
     *         description="Poprerties Response",
     *         @SWG\Schema(ref="#/definitions/servers")
     *     )
     *  )
     */
    /**
     * @SWG\Delete(path="/servers/{server_id}",
     *   tags={"Servers"},
     *   summary="delete Server",
     *   description="Request to delete Server",
     *   operationId="DELETE",
     *   produces={"application/xml", "application/json"},
     *   @SWG\Parameter(
     *     name="token",
     *     in="query",
     *     description="server token",
     *     required=true,
     *     type="string"
     *   ),
     * * @SWG\Parameter(
     *     name="server_id",
     *     in="path",
     *     description="id of the server",
     *     required=true,
     *     type="integer",
     *     format = "int32"
     *   ),
     * @SWG\Response(
     *         response=200,
     *         description="Poprerties Response",
     *         @SWG\Schema(ref="#/definitions/servers")
     *     )
     *  )
     */

    /**
     * delete server
     * @return id of server deleted or error object if a server is not deleted
     */
    function DELETE() {
        $aReturn = $this->genericDelete($this->aProperties['schema_gtf'], 'server', 'server_id');
        return $aReturn['sMessage'];
    }

    /*
     * *	
     */

    Public function insertCrontab() {
        require_once("ServerClass.class.inc");
        if (isset($this->aValues['old_hostname']))
            $sOldHostname = $this->aValues['old_hostname'];
        else
            $sOldHostname = null;
        $oCrontab = new ServerClass($this->aValues["my_vitis_id"], $this->oConnection->oBd, $this->aProperties['vas_home'], $sOldHostname, $this->aProperties);
        $oCrontab->deleteCrontabValuesEngine();
        $oCrontab->deleteCrontabValuesSubscription();
        $oCrontab->AddCrontabValuesEngines();
        if ($this->aValues["hostname"] == "localhost")
            $oCrontab->AddCrontabValuesSubscription();
        $oCrontab->AddPropertiesFME();
    }

}

?>