<?php

/**
 * \file Subscriptions.class.inc
 * \class Subscriptions
 *
 * \author Yoann Perollet <yoann.perollet@veremes.com>.
 *
 * 	\brief This file contains the Subscriptions php class
 *
 * This class defines Rest Api to Gtf subscriptions
 * 
 */
require_once 'Gtf.class.inc';
require_once dirname($_SERVER['SCRIPT_FILENAME']) . '/class/vitis_lib/Connection.class.inc';
require_once 'Subscription.class.inc';
require_once dirname($_SERVER['SCRIPT_FILENAME']) . '/class/vmlib/BdDataAccess.inc';

class Subscriptions extends GTF {
    /**
     * @SWG\Definition(
     *   definition="/subscriptions",
     *   allOf={
     *     @SWG\Schema(ref="#/definitions/subscriptions")
     *   }
     * )
     * * @SWG\Tag(
     *   name="Subscriptions",
     *   description="Operations about Subscriptions"
     * )
     */

    /**
     * construct
     * @param type $aPath url of the request
     * @param type $aValues parameters of the request
     * @param type $properties properties
     * @param type $bShortcut false to reinit variables
     * @param type $oConnection connection object
     */
    function __construct($aPath, $aValues, $properties, $bShortcut = false, $oConnection = false) {
        parent::__construct($aPath, $aValues, $properties, $bShortcut, $oConnection);
        $this->aSelectedFields = Array('order_id', 'priority_id', 'workspace_id', 'workspace_key', 'enabled', 'label_name', 'order_status_id', 'user_id', 'login', 'period_id', 'order_period_libelle', 'wk_params', 'result_url', $this->getDateSelectedFields('order_date', 'yeartosecond'), 'log_url', $this->getDateSelectedFields('execution_date', 'yeartosecond'), 'attempt', 'email_notifications', 'email_option_id', 'deleted', 'gtf_engine_id', 'notification');
    }

    /**
     * @SWG\Get(path="/subscriptions",
     *   tags={"Subscriptions"},
     *   summary="Get Subscriptions",
     *   description="Request to get Subscriptions",
     *   operationId="GET",
     *   produces={"application/xml", "application/json"},
     *   @SWG\Parameter(
     *     name="token",
     *     in="query",
     *     description="user token",
     *     required=true,
     *     type="string"
     *   ),
     * @SWG\Parameter(
     *     name="order_by",
     *     in="query",
     *     description="list of subscriptioning fields",
     *     required=false,
     *     type="string"
     *   ),
     * @SWG\Parameter(
     *     name="sort_order",
     *     in="query",
     *     description="sort order",
     *     required=false,
     *     type="string"
     *   ),
     * @SWG\Parameter(
     *     name="limit",
     *     in="query",
     *     description="number of element",
     *     required=false,
     *     type="integer",
     *     format="int32"
     *   ),
     * @SWG\Parameter(
     *     name="offset",
     *     in="query",
     *     description="index of first element",
     *     required=false,
     *     type="string",
     *     format="int32"
     *   ),
     * @SWG\Parameter(
     *     name="attributs",
     *     in="query",
     *     description="list of attributs",
     *     required=false,
     *     type="string"
     *   ),
     * @SWG\Parameter(
     *     name="filter",
     *     in="query",
     *     description="filter results",
     *     required=false,
     *     type="string"
     *   ),
     * @SWG\Parameter(
     *     name="distinct",
     *     in="query",
     *     description="delete duplicates",
     *     required=false,
     *     type="boolean"
     *   ),
     *   @SWG\Response(
     *         response=200,
     *         description="Poprerties Response",
     *         @SWG\Schema(ref="#/definitions/subscriptions")
     *     )
     *  )
     */

    /**
     * get Subscriptions
     * @return  Subscriptions
     */
    function GET() {
        $aReturn = $this->genericGet($this->aProperties['schema_gtf'], "v_subscription", "order_id");
        return $aReturn['sMessage'];
    }

    /*
     * function getIdEngine
     * \brief function that assigns each request processing an engine number. This engine number will be defined by the algorithm associated to the treatment.
     * \param $iSubscriptionId Identifier of the request
     * \param $sTable Name of the table.
     * \param $iWorkspaceId Identifier of the treatment.
     */

    function getIdEngine($iSubscriptionId, $sTable, $iWorkspaceId) {
        require $this->sRessourcesFile;
        // Récupération de l'algorithme
        $aParams['sSchemaGtf'] = array('value' => $this->aProperties['schema_gtf'], 'type' => 'schema_name');
        $aParams['iWorkspaceId'] = array('value' => $iWorkspaceId, 'type' => 'number');
        $oPDOresult = $this->oConnection->oBd->executeWithParams($aSql['getEnginesId'], $aParams);
        $this->oConnection->oBd->cleanMessage();
        $aEnginesId = array();
        while ($oEnginesId = $this->oConnection->oBd->objetSuivant($oPDOresult))
            $aEnginesId[] = $oEnginesId->gtf_engine_id;
        $this->oConnection->oBd->fermeResultat();

        if (count($aEnginesId) == 0) {
            // Le tag correspondant n'a pas été trouvé, nous allons cherché la liste des moteur ayant comme tag |default|
            $aParams = array();
            $aParams['sSchemaGtf'] = array('value' => $this->aProperties['schema_gtf'], 'type' => 'schema_name');
            $oPDOresult = $this->oConnection->oBd->executeWithParams($aSql['getEnginesDefaultTag'], $aParams);
            $this->oConnection->oBd->cleanMessage();
            $aEnginesId = array();
            while ($oEnginesId = $this->oConnection->oBd->objetSuivant($oPDOresult))
                $aEnginesId[] = $oEnginesId->gtf_engine_id;
            $this->oConnection->oBd->fermeResultat();
            if (count($aEnginesId) > 0)
                writeToErrorLog(str_replace('[iSubscriptionId]', $iSubscriptionId, ERROR_0006));
        }
        // Si la requête est en erreur ou Si aucun moteur n'a été trouvé.
        if ($this->oConnection->oBd->erreurRencontree || count($aEnginesId) == 0) {
            $aGtfEngineList = explode(",", $this->aProperties["engines_list"]);
            // Aucun tag ne correspond
            $iEngine = $aGtfEngineList[0];
            if ($iEngine != "")
                writeToErrorLog(str_replace('[aGtfEngineList[0]]', $iEngine, str_replace('[iSubscriptionId]', $iSubscriptionId, ERROR_0007)));
        }else {
            $iEngine = $this->returnEngineId($iSubscriptionId, $aEnginesId);
        }

        if ($iEngine == "") {
            writeToErrorLog(str_replace('[iSubscriptionId]', $iSubscriptionId, ERROR_0008));
            $this->aValues['gtf_engine_id'] = 1;
        } else {
            $this->aValues['gtf_engine_id'] = $iEngine;
        }
    }

    /*
     * function returnEngineId
     * \param $oAlgoIndex Object of the algorithm.
     * \param $aGtfEngineList Array that corresponds to the list of available engines numbers.
     * \return Returns the engine number.
     */

    function returnEngineId($iSubscriptionId, $aEnginesId) {
        $iEngine = "";
        if (count($aEnginesId) > 0) {
            $sRowCount = $iSubscriptionId % count($aEnginesId);
            $iEngine = $aEnginesId[$sRowCount];
        }
        return $iEngine;
    }

    /**
     * @SWG\Delete(path="/subscriptions/",
     *   tags={"Subscriptions"},
     *   summary="delete Subscription",
     *   description="Request to delete Subscription",
     *   operationId="DELETE",
     *   produces={"application/xml", "application/json"},
     *   @SWG\Parameter(
     *     name="token",
     *     in="query",
     *     description="user token",
     *     required=true,
     *     type="string"
     *   ),
     * * @SWG\Parameter(
     *     name="idList",
     *     in="query",
     *     description="id of the subscriptions",
     *     required=true,
     *     type="string"
     *   ),
     * @SWG\Response(
     *         response=200,
     *         description="Poprerties Response",
     *         @SWG\Schema(ref="#/definitions/subscriptions")
     *     )
     *  )
     */
    /**
     * @SWG\Delete(path="/subscriptions/{order_id}",
     *   tags={"Subscriptions"},
     *   summary="delete Subscription",
     *   description="Request to delete Subscription",
     *   operationId="DELETE",
     *   produces={"application/xml", "application/json"},
     *   @SWG\Parameter(
     *     name="token",
     *     in="query",
     *     description="user token",
     *     required=true,
     *     type="string"
     *   ),
     * * @SWG\Parameter(
     *     name="order_id",
     *     in="path",
     *     description="id of the order",
     *     required=true,
     *     type="integer",
     *     format = "int32"
     *   ),
     * @SWG\Response(
     *         response=200,
     *         description="Poprerties Response",
     *         @SWG\Schema(ref="#/definitions/subscriptions")
     *     )
     *  )
     */

    /**
     * delete subscription
     * @return id of subscription deleted or error object if a subscription is not deleted
     */
    function DELETE() {
        $aReturn = $this->genericDelete($this->aProperties['schema_gtf'], 'v_subscription', 'order_id');
        return $aReturn['sMessage'];
    }

}

?>