<?php

/**
 * \file UserOrders.class.inc
 * \class UserOrders
 *
 * \author Yoann Perollet <yoann.perollet@veremes.com>.
 *
 * 	\brief This file contains the UserOrders php class
 *
 * This class defines Rest Api to Gtf userorders
 * 
 */
require_once 'Gtf.class.inc';
require_once dirname($_SERVER['SCRIPT_FILENAME']) . '/class/vitis_lib/Connection.class.inc';
require_once 'UserOrder.class.inc';
require_once dirname($_SERVER['SCRIPT_FILENAME']) . '/class/vmlib/BdDataAccess.inc';
require_once dirname($_SERVER['SCRIPT_FILENAME']) . '/ws/vitis/PrivateToken.class.inc';

class UserOrders extends GTF {
    /**
     * @SWG\Definition(
     *   definition="/userorders",
     *   allOf={
     *     @SWG\Schema(ref="#/definitions/userorders")
     *   }
     * )
     * * @SWG\Tag(
     *   name="UserOrders",
     *   description="Operations about UserOrders"
     * )
     */

    /**
     * construct
     * @param type $aPath url of the request
     * @param type $aValues parameters of the request
     * @param type $properties properties
     * @param type $bShortcut false to reinit variables
     * @param type $oConnection connection object
     */
    function __construct($aPath, $aValues, $properties, $bShortcut = false, $oConnection = false) {
        parent::__construct($aPath, $aValues, $properties, $bShortcut, $oConnection);
        $this->aSelectedFields = Array("order_id", "priority_id", "workspace_key", "workspace_id", "name", "order_status_id", "user_id", "wk_params", "result_url", $this->getDateSelectedFields('order_date', 'yeartosecond'), "log_url", $this->getDateSelectedFields('execution_date', 'yeartosecond'), "attempt", "email_notifications", "email_option_id", "deleted", "gtf_engine_id", "length_sec", "workspace_name", "email_option_label", "order_status_label", $this->getDateSelectedFields('minexecdate', 'yeartosecond'));
    }

    /**
     * @SWG\Get(path="/userorders",
     *   tags={"UserOrders"},
     *   summary="Get UserOrders",
     *   description="Request to get UserOrders",
     *   operationId="GET",
     *   produces={"application/xml", "application/json",  "application/x-vm-json"},
     *   @SWG\Parameter(
     *     name="token",
     *     in="query",
     *     description="user token",
     *     required=true,
     *     type="string"
     *   ),
     * @SWG\Parameter(
     *     name="order_by",
     *     in="query",
     *     description="list of ordering fields",
     *     required=false,
     *     type="string"
     *   ),
     * @SWG\Parameter(
     *     name="limit",
     *     in="query",
     *     description="number of element",
     *     required=false,
     *     type="integer",
     *     format="int32"
     *   ),
     * @SWG\Parameter(
     *     name="offset",
     *     in="query",
     *     description="index of first element",
     *     required=false,
     *     type="string",
     *     format="int32"
     *   ),
     * @SWG\Parameter(
     *     name="attributs",
     *     in="query",
     *     description="list of attributs",
     *     required=false,
     *     type="string"
     *   ),
     * @SWG\Parameter(
     *     name="filter",
     *     in="query",
     *     description="filter results",
     *     required=false,
     *     type="string"
     *   ),
     * @SWG\Parameter(
     *     name="distinct",
     *     in="query",
     *     description="delete duplicates",
     *     required=false,
     *     type="boolean"
     *   ),
     *   @SWG\Response(
     *         response=200,
     *         description="Poprerties Response",
     *         @SWG\Schema(ref="#/definitions/userorders")
     *     )
     *  )
     */

    /**
     * get UserOrders
     * @return  UserOrders
     */
    function GET() {
        $aReturn = $this->genericGet($this->aProperties['schema_gtf'], "v_user_order", "order_id");
        return $aReturn['sMessage'];
    }

    /**
     * @SWG\Post(path="/userorders",
     *   tags={"UserOrders"},
     *   summary="Add UserOrder",
     *   description="Request to add userorder",
     *   operationId="POST",
     *   produces={"application/xml", "application/json",  "application/x-vm-json"},
     *   @SWG\Parameter(
     *     name="token",
     *     in="formData",
     *     description="user token",
     *     required=true,
     *     type="string"
     *   ),
     *   @SWG\Parameter(
     *     name="order_date",
     *     in="formData",
     *     description="userorder date",
     *     required=true,
     *     type="string"
     *   ),
     *   @SWG\Parameter(
     *     name="workspace_id",
     *     in="formData",
     *     description="workspace id",
     *     required=true,
     *     type="integer",
     *     format="int32"
     *   ),
     *   @SWG\Parameter(
     *     name="priority_id",
     *     in="formData",
     *     description="priority id",
     *     required=true,
     *     type="integer",
     *     format="int32"
     *   ),
     *   @SWG\Parameter(
     *     name="email_option_id",
     *     in="formData",
     *     description="email option id",
     *     required=true,
     *     type="integer",
     *     format="int32"
     *   ),
     *   @SWG\Parameter(
     *     name="wk_params",
     *     in="formData",
     *     description="workspace parameters",
     *     required=true,
     *     type="string"
     *   ),
     *   @SWG\Response(
     *         response=200,
     *         description="Poprerties Response",
     *         @SWG\Schema(ref="#/definitions/userorders")
     *     ),
     * 
     *  )
     */

    /**
     * insert user's order
     * @return id of the user's userorder created
     */
    function POST() {
        $this->aValues['order_date'] = gmdate('Y-m-d H:i:s');
        $this->aValues['priority_id'] = 1;
        if (!empty($this->aValues['minexecdate']))
            $this->aValues['minexecdate'] = gmdate('Y-m-d H:i:00', date_timestamp_get(date_create_from_format('d/m/Y H:i', $this->aValues['minexecdate'])));
        else
            $this->aValues['minexecdate'] = null;
        if (!empty($_SESSION['ses_user_id']))
            $this->aValues['user_id'] = $_SESSION['ses_user_id'];
        if (!empty($this->aValues['email_notifications']))
            $this->aValues['email_notifications'] = str_replace(chr(13) . chr(10), ";", $this->aValues['email_notifications']);
        // Si token public -> id du compte public.
        if ($this->aValues['token'] == $this->aProperties['public_token']) {
            $aPrivateTokenValues = $this->aValues;
            $aPrivateTokenValues['user'] = $_SESSION['ses_Login'];
            $aPrivateTokenValues['password'] = $this->oConnection->oBd->mdp;
            $oPrivateToken = new PrivateToken($this->aPath, $aPrivateTokenValues, $this->aProperties, $this->oConnection);
            $oPrivateToken->POST();
            $this->aValues['user_id'] = $oPrivateToken->aFields['user_id'];
            // Teste si le compte 'public' a un des groupes qui est lié au traitement.
            require $this->sRessourcesFile;
            $aParams['sSchemaGtf'] = array('value' => $this->aProperties['schema_gtf'], 'type' => 'schema_name');
            $aParams['workspace_id'] = array('value' => $this->aValues['workspace_id'], 'type' => 'number');
            $oPDOresult = $this->oConnection->oBd->executeWithParams($aSql['checkUserWorkspace'], $aParams);
            if ($this->oConnection->oBd->enErreur()) {
                $oError = new VitisError(1, $this->oConnection->oBd->getBDMessage());
                $aXmlRacineAttribute['status'] = 0;
                $sMessage = $oError->asDocument('', 'vitis', $this->aValues['sEncoding'], True, $aXmlRacineAttribute, $this->aValues['sSourceEncoding'], $this->aValues['output']);
                return $sMessage;
            } else {
                if ($this->oConnection->oBd->nombreLigne($oPDOresult) == 0) {
                    $oError = new VitisError(1, "ERROR_WORKSPACE_NOT_LINKED_USER_GROUPS_WIDGET");
                    $aXmlRacineAttribute['status'] = 0;
                    $sMessage = $oError->asDocument('', 'vitis', $this->aValues['sEncoding'], True, $aXmlRacineAttribute, $this->aValues['sSourceEncoding'], $this->aValues['output']);
                    return $sMessage;
                }
            }
        }
        // Captcha ?
        if (!empty($this->aValues['captcha_response'])) {
            $sResponse = file_get_contents("https://www.google.com/recaptcha/api/siteverify?secret=" . $this->aProperties['private_captcha_key'] . "&response=" . $this->aValues['captcha_response']);
            $oResponse = json_decode($sResponse, true);
            if (empty($oResponse) || !$oResponse['success']) {
                $oError = new VitisError(1, "ERROR_CAPTCHA_VALIDATION_WIDGET");
                $aXmlRacineAttribute['status'] = 0;
                $sMessage = $oError->asDocument('', 'vitis', $this->aValues['sEncoding'], True, $aXmlRacineAttribute, $this->aValues['sSourceEncoding'], $this->aValues['output']);
                return $sMessage;
            }
        }
        $aReturn = $this->genericPost($this->aProperties['schema_gtf'], 'v_user_order', $this->aProperties['schema_gtf'] . '.seq_order', 'order_id');
        if ($aReturn['sStatus'] == 1) {
            $aXmlRacineAttribute['status'] = 1;
            $sMessage = $this->asDocument('', 'vitis', $this->aValues['sEncoding'], True, $aXmlRacineAttribute, $this->aValues['sSourceEncoding'], $this->aValues['output']);
            $oUserOrder = new UserOrder($this->aPath, $this->aValues, $this->aProperties, $this->oConnection);
            $oUserOrder->GET();
            // Fichiers à uploader ?
            if (!empty($_FILES)) {
                foreach ($_FILES as $sName => $aFile) {

                    $sDirName = $this->aProperties['upload_dir'] . "/" . $this->aValues['upload_directory'];
                    if (!is_dir($sDirName))
                        mkdir($sDirName, 0777, true);
                    uploadFile($sName, "", $sDirName . "/" . $aFile['name'], $aFile['size'] + 1);
                }
            }
            // Met à jour le n° du moteur.
            $this->getIdEngine($this->aValues["my_vitis_id"], 'userorder', $this->aValues['workspace_id']);
            require $this->sRessourcesFile;
            $aParams = array();
            $aParams['sSchemaGtf'] = array('value' => $this->aProperties['schema_gtf'], 'type' => 'schema_name');
            $aParams['order_id'] = array('value' => $this->aValues['my_vitis_id'], 'type' => 'number');
            $aParams['gtf_engine_id'] = array('value' => $this->aValues['gtf_engine_id'], 'type' => 'number');
            $oPDOresult = $this->oConnection->oBd->executeWithParams($aSql['setUserOrderEngineId'], $aParams);
            if ($this->oConnection->oBd->enErreur()) {
                $oError = new VitisError(1, $this->oConnection->oBd->getBDMessage());
                $aXmlRacineAttribute['status'] = 0;
                $sMessage = $oError->asDocument('', 'vitis', $this->aValues['sEncoding'], True, $aXmlRacineAttribute, $this->aValues['sSourceEncoding'], $this->aValues['output']);
            }
        } else {
            $sMessage = $aReturn['sMessage'];
        }
        /*
          if ($aReturn['sStatus'] == 1){
          $oUserOrder = new UserOrder($this->aPath, $this->aValues, $this->aProperties, $this->oConnection);
          $oUserOrder->GET();
          $this->getIdEngine($this->aValues["my_vitis_id"], 'userorder', $this->aValues['workspace_id']);
          // Met à jour le n° du moteur.
          require $this->sRessourcesFile;
          $sSql = $aSql['setUserOrderEngineId'];
          $sSql = str_replace("[sSchemaGtf]", $this->aProperties['schema_gtf'], $sSql);
          $sSql = str_replace("[order_id]", $this->aValues["my_vitis_id"], $sSql);
          $sSql = str_replace("[gtf_engine_id]", $this->aValues['gtf_engine_id'], $sSql);
          $resultat = $this->oConnection->oBd->execute ($sSql);
          if ($this->oConnection->oBd->enErreur()) {
          $oError = new VitisError(1, $this->oConnection->oBd->getBDMessage());
          $aXmlRacineAttribute['status'] = 0;
          $sMessage = $oError->asDocument('', 'vitis', $this->aValues['sEncoding'], True, $aXmlRacineAttribute, $this->aValues['sSourceEncoding'], $this->aValues['output']);
          }
          else {
          $aXmlRacineAttribute['status'] = 1;
          $sMessage = $this->asDocument('', 'vitis', $this->aValues['sEncoding'], True, $aXmlRacineAttribute, $this->aValues['sSourceEncoding'], $this->aValues['output']);
          }
          }else{
          $sMessage = $aReturn['sMessage'];
          }
         */
        return $sMessage;
    }

    /*
     * function getIdEngine
     * \brief function that assigns each request processing an engine number. This engine number will be defined by the algorithm associated to the treatment.
     * \param $iUserOrderId Identifier of the request
     * \param $sTable Name of the table.
     * \param $iWorkspaceId Identifier of the treatment.
     */

    function getIdEngine($iUserOrderId, $sTable, $iWorkspaceId) {
        require $this->sRessourcesFile;
        // Récupération de l'algorithme
        $aParams['sSchemaGtf'] = array('value' => $this->aProperties['schema_gtf'], 'type' => 'schema_name');
        $aParams['iWorkspaceId'] = array('value' => $iWorkspaceId, 'type' => 'number');
        $oPDOresult = $this->oConnection->oBd->executeWithParams($aSql['getEnginesId'], $aParams);
        $this->oConnection->oBd->cleanMessage();
        $aEnginesId = array();
        while ($oEnginesId = $this->oConnection->oBd->objetSuivant($oPDOresult))
            $aEnginesId[] = $oEnginesId->gtf_engine_id;
        $this->oConnection->oBd->fermeResultat();

        if (count($aEnginesId) == 0) {
            // Le tag correspondant n'a pas été trouvé, nous allons cherché la liste des moteur ayant comme tag |default|
            $aParams['sSchemaGtf'] = array('value' => $this->aProperties['schema_gtf'], 'type' => 'schema_name');
            $oPDOresult = $this->oConnection->oBd->executeWithParams($aSql['getEnginesDefaultTag'], $aParams);
            $this->oConnection->oBd->cleanMessage();
            $aEnginesId = array();
            while ($oEnginesId = $this->oConnection->oBd->objetSuivant($oPDOresult))
                $aEnginesId[] = $oEnginesId->gtf_engine_id;
            $this->oConnection->oBd->fermeResultat();
            if (count($aEnginesId) > 0)
                writeToErrorLog(str_replace('[iUserOrderId]', $iUserOrderId, ERROR_0006));
        }
        // Si la requête est en erreur ou Si aucun moteur n'a été trouvé.
        if ($this->oConnection->oBd->erreurRencontree || count($aEnginesId) == 0) {
            $aGtfEngineList = explode(",", $this->aProperties["engines_list"]);
            // Aucun tag ne correspond
            $iEngine = $aGtfEngineList[0];
            if ($iEngine != "")
                writeToErrorLog(str_replace('[aGtfEngineList[0]]', $iEngine, str_replace('[iUserOrderId]', $iUserOrderId, ERROR_0007)));
        }else {
            $iEngine = $this->returnEngineId($iUserOrderId, $aEnginesId);
        }

        if ($iEngine == "") {
            writeToErrorLog(str_replace('[iUserOrderId]', $iUserOrderId, ERROR_0008));
            $this->aValues['gtf_engine_id'] = 1;
        } else {
            $this->aValues['gtf_engine_id'] = $iEngine;
        }
    }

    /*
     * function returnEngineId
     * \param $oAlgoIndex Object of the algorithm.
     * \param $aGtfEngineList Array that corresponds to the list of available engines numbers.
     * \return Returns the engine number.
     */

    function returnEngineId($iUserOrderId, $aEnginesId) {
        $iEngine = "";
        if (count($aEnginesId) > 0) {
            $sRowCount = $iUserOrderId % count($aEnginesId);
            $iEngine = $aEnginesId[$sRowCount];
        }
        return $iEngine;
    }

    /**
     * @SWG\Delete(path="/userorders/",
     *   tags={"UserOrders"},
     *   summary="delete UserOrder",
     *   description="Request to delete UserOrder",
     *   operationId="DELETE",
     *   produces={"application/xml", "application/json",  "application/x-vm-json"},
     *   @SWG\Parameter(
     *     name="token",
     *     in="query",
     *     description="userorder token",
     *     required=true,
     *     type="string"
     *   ),
     * * @SWG\Parameter(
     *     name="idList",
     *     in="query",
     *     description="id of the userorders",
     *     required=true,
     *     type="string"
     *   ),
     * @SWG\Response(
     *         response=200,
     *         description="Poprerties Response",
     *         @SWG\Schema(ref="#/definitions/userorders")
     *     )
     *  )
     */
    /**
     * @SWG\Delete(path="/userorders/{order_id}",
     *   tags={"UserOrders"},
     *   summary="delete UserOrder",
     *   description="Request to delete UserOrder",
     *   operationId="DELETE",
     *   produces={"application/xml", "application/json",  "application/x-vm-json"},
     *   @SWG\Parameter(
     *     name="token",
     *     in="query",
     *     description="userorder token",
     *     required=true,
     *     type="string"
     *   ),
     * * @SWG\Parameter(
     *     name="order_id",
     *     in="path",
     *     description="id of the userorder",
     *     required=true,
     *     type="integer",
     *     format = "int32"
     *   ),
     * @SWG\Response(
     *         response=200,
     *         description="Poprerties Response",
     *         @SWG\Schema(ref="#/definitions/userorders")
     *     )
     *  )
     */

    /**
     * delete userorder
     * @return id of userorder deleted or error object if a userorder is not deleted
     */
    function DELETE() {
        $aReturn = $this->genericDelete($this->aProperties['schema_gtf'], 'v_user_order', 'order_id');
        return $aReturn['sMessage'];
    }

}

?>
