<?php

/**
 * \file Workspaces.class.inc
 * \class Workspaces
 *
 * \author Yoann Perollet <yoann.perollet@veremes.com>.
 *
 * 	\brief This file contains the Workspaces php class
 *
 * This class defines Rest Api to Gtf workspaces
 * 
 */
require_once 'Gtf.class.inc';
require_once dirname($_SERVER['SCRIPT_FILENAME']) . '/class/vitis_lib/Connection.class.inc';
require_once 'Workspace.class.inc';
require_once dirname($_SERVER['SCRIPT_FILENAME']) . '/class/vmlib/BdDataAccess.inc';
require_once dirname($_SERVER['SCRIPT_FILENAME']) . '/class/vmlib/phpUtil.inc';
require_once dirname($_SERVER['SCRIPT_FILENAME']) . '/class/gtf_lib/GtfFmwParser.class.inc';

class Workspaces extends GTF {
    /**
     * @SWG\Definition(
     *   definition="/workspaces",
     *   allOf={
     *     @SWG\Schema(ref="#/definitions/workspaces")
     *   }
     * )
     * * @SWG\Tag(
     *   name="Workspaces",
     *   description="Operations about Workspaces"
     * )
     */

    /**
     * construct
     * @param type $aPath url of the request
     * @param type $aValues parameters of the request
     * @param type $properties properties
     * @param type $bShortcut false to reinit variables
     * @param type $oConnection connection object
     */
    function __construct($aPath, $aValues, $properties, $bShortcut = false, $oConnection = false) {
        parent::__construct($aPath, $aValues, $properties, $bShortcut, $oConnection);
        $this->aSelectedFields = Array('workspace_id', 'name', 'fmw_file', 'owner', $this->getDateSelectedFields('creation_date', 'yeartosecond'), 'description', 'subscription', 'category_id', 'success_action_id', 'failed_action_id', 'survey', 'form_type', 'legal_terms_conditions', $this->getDateSelectedFields('last_save_date', 'yeartosecond'), 'last_save_build', 'file_encoding', 'title', 'requirements', 'history', 'usage', 'key', 'email_template_id', 'locked', 'tag', 'login', 'category_name', 'nb_demande', 'email_template', 'dataencrypt', 'usepersonaldata');
    }

    /**
     * @SWG\Get(path="/workspaces",
     *   tags={"Workspaces"},
     *   summary="Get Workspaces",
     *   description="Request to get Workspaces",
     *   operationId="GET",
     *   produces={"application/xml", "application/json"},
     *   @SWG\Parameter(
     *     name="token",
     *     in="query",
     *     description="user token",
     *     required=true,
     *     type="string"
     *   ),
     * @SWG\Parameter(
     *     name="workspace_by",
     *     in="query",
     *     description="list of workspaceing fields",
     *     required=false,
     *     type="string"
     *   ),
     * @SWG\Parameter(
     *     name="limit",
     *     in="query",
     *     description="number of element",
     *     required=false,
     *     type="integer",
     *     format="int32"
     *   ),
     * @SWG\Parameter(
     *     name="offset",
     *     in="query",
     *     description="index of first element",
     *     required=false,
     *     type="string",
     *     format="int32"
     *   ),
     * @SWG\Parameter(
     *     name="attributs",
     *     in="query",
     *     description="list of attributs",
     *     required=false,
     *     type="string"
     *   ),
     * @SWG\Parameter(
     *     name="filter",
     *     in="query",
     *     description="filter results",
     *     required=false,
     *     type="string"
     *   ),
     * @SWG\Parameter(
     *     name="distinct",
     *     in="query",
     *     description="delete duplicates",
     *     required=false,
     *     type="boolean"
     *   ),
     *   @SWG\Response(
     *         response=200,
     *         description="Poprerties Response",
     *         @SWG\Schema(ref="#/definitions/workspaces")
     *     )
     *  )
     */
    /**
     * @SWG\Get(path="/workspaces/{workspace_id}/CheckWidgetFile",
     *   tags={"Workspaces"},
     *   summary="Check if the widget file exist",
     *   description="Request to check if the widget file exist",
     *   operationId="GET",
     *   produces={"application/xml", "application/json"},
     *   @SWG\Parameter(
     *     name="token",
     *     in="query",
     *     description="user token",
     *     required=true,
     *     type="string"
     *   ),
     *   @SWG\Parameter(
     *     name="workspace_id",
     *     in="path",
     *     description="workspace id",
     *     required=true,
     *     type="string"
     *   ),
     *   @SWG\Response(
     *         response=200,
     *         description="Properties Response",
     *         @SWG\Schema(ref="#/definitions/workspaces")
     *     )
     *  )
     */

    /**
     * get Workspaces
     * @return  Workspaces
     */
    function GET() {
        if (!empty($this->aPath[3])) {
            if ($this->aPath[3] == "File") {
                $this->getWorkspaceFile();
            } else {
                if ($this->aPath[3] == "Tree") {
                    $aReturn = $this->getWorkspaceDirectoryTree();
                } else if ($this->aPath[3] == "Metadata") {
                    $aReturn = $this->regenerateMetadata();
                } else if ($this->aPath[3] == "CheckWidgetFile") {
                    $aReturn = $this->checkWidgetFile();
                }
                $aXmlRacineAttribute['status'] = $aReturn['status'];
                if ($aReturn['status'] == 1) {
                    $sMessage = $this->asDocument('', 'vitis', $this->aValues['sEncoding'], True, $aXmlRacineAttribute, $this->aValues['sSourceEncoding'], $this->aValues['output']);
                } else {
                    $oError = new VitisError($aReturn['error_code'], $aReturn['message']);
                    $sMessage = $oError->asDocument('', 'vitis', $this->aValues['sEncoding'], True, $aXmlRacineAttribute, $this->aValues['sSourceEncoding'], $this->aValues['output']);
                }
                return $sMessage;
            }
        } else {
            $aReturn = $this->genericGet($this->aProperties['schema_gtf'], "v_workspace", "workspace_id");
            // Formatage de la colonne "form_type".
            if ($aReturn['sStatus'] == 1) {
                foreach ($this->aObjects as &$oWorkspace) {
                    $sFile = $this->aProperties['workspace_dir'] . "/" . $oWorkspace->aFields['workspace_id'] . '/form/Subform.json';
                    if (!file_exists($sFile)) {
                        $oWorkspace->aFields['form_type'] = '';
                    } else {
                        if ($oWorkspace->aFields['form_type'] != "custom") {
                            $oWorkspace->aFields['form_type'] = "default";
                        }
                    }
                }
                $aXmlRacineAttribute['status'] = 1;
                $sMessage = $this->asDocument('', 'vitis', $this->aValues['sEncoding'], True, $aXmlRacineAttribute, $this->aValues['sSourceEncoding'], $this->aValues['output']);
            } else {
                $sMessage = $aReturn['sMessage'];
            }
            return $sMessage;
        }
    }

    /**
     * @SWG\Post(path="/workspaces",
     *   tags={"Workspaces"},
     *   summary="Add Workspace",
     *   description="Request to add workspace",
     *   operationId="POST",
     *   produces={"application/xml", "application/json"},
     *   consumes= { "multipart/form-data"},
     *   @SWG\Parameter(
     *     name="token",
     *     in="formData",
     *     description="workspace token",
     *     required=true,
     *     type="string"
     *   ),
     *   @SWG\Parameter(
     *     name="name",
     *     in="formData",
     *     description="Name of the workspace",
     *     required=false,
     *     type="string"
     *   ),
     *   @SWG\Parameter(
     *     name="fmw_file",
     *     in="formData",
     *     description="Fmw file of the workspace",
     *     required=false,
     *     type="file"
     *   ),
     *   @SWG\Parameter(
     *     name="comp_file",
     *     in="formData",
     *     description="Other file (.zip) of the workspace",
     *     required=false,
     *     type="file"
     *   ),
     *   @SWG\Parameter(
     *     name="tag",
     *     in="formData",
     *     description="Tag of the workspace",
     *     required=false,
     *     type="string"
     *   ),
     *   @SWG\Parameter(
     *     name="email_template_id",
     *     in="formData",
     *     description="Email template of the workspace",
     *     required=false,
     *     type="integer",
     *     format="int32"
     *   ),
     *   @SWG\Parameter(
     *     name="subscription",
     *     in="formData",
     *     description="Subscription ?",
     *     required=false,
     *     default=false,
     *     type="boolean"
     *   ),
     *   @SWG\Parameter(
     *     name="survey",
     *     in="formData",
     *     description="Survey ?",
     *     required=false,
     *     default=false,
     *     type="boolean"
     *   ),
     *   @SWG\Parameter(
     *     name="locked",
     *     in="formData",
     *     description="Locked ?",
     *     required=false,
     *     default=false,
     *     type="boolean"
     *   ),
     *   @SWG\Response(
     *         response=200,
     *         description="Poprerties Response",
     *         @SWG\Schema(ref="#/definitions/workspaces")
     *     )
     *  )
     */
    /**
     * @SWG\Post(path="/workspaces/{workspace_id}/GenerateWidgetFile",
     *   tags={"Workspaces"},
     *   summary="Generate Widget file",
     *   description="Request to get a file from the workspace directory",
     *   operationId="POST",
     *   produces={"application/xml", "application/json"},
     *   @SWG\Parameter(
     *     name="token",
     *     in="formData",
     *     description="user token",
     *     required=true,
     *     type="string"
     *   ),
     *   @SWG\Parameter(
     *     name="workspace_id",
     *     in="path",
     *     description="workspace id",
     *     required=true,
     *     type="string"
     *   ),
     *   @SWG\Parameter(
     *     name="content_file",
     *     in="formData",
     *     description="File content of the widget",
     *     required=true,
     *     type="string"
     *   ),
     *   @SWG\Response(
     *         response=200,
     *         description="Poprerties Response",
     *         @SWG\Schema(ref="#/definitions/workspaces")
     *     )
     *  )
     */

    /**
     * insert workspace
     * @return id of the workspace created
     */
    function POST() {
        require $this->sRessourcesFile;
        // Importation de projets.
        if (!empty($this->aPath[3]) && $this->aPath[3] === "GenerateWidgetFile") {
            unset($this->aPath[3]);
            $oWorkspace = new Workspace($this->aPath, $this->aValues, $this->aProperties, $this->oConnection);
            $oWorkspace->GET();
            $sWorkspace_key = $oWorkspace->aFields["key"];
            // Emplacement des widgets
            $sWidgetDirectory = $this->aProperties["vas_home"] . '/ws_data/gtf/widget';
            $bError = False;
            if (!is_dir($sWidgetDirectory)) {
                if (!mkdir($sWidgetDirectory, 0777, true)) {
                    $bError = True;
                    $this->aFields['errorMessage'] = "Failed to create directory " . $sWidgetDirectory;
                }
            }
            if (!$bError) {
                $this->aFields['workspace_key'] = $sWorkspace_key;
                // Widget dans une page html.
                $sWidgetHtml = '<!DOCTYPE html>' . PHP_EOL . '<html>' . PHP_EOL . '<head>' . PHP_EOL . '<title>Widget</title>' . PHP_EOL . '<meta name="viewport" content="width=device-width, initial-scale=1.0">' . PHP_EOL . '<meta http-equiv="X-UA-Compatible" content="IE=edge">' . PHP_EOL . '</head>' . PHP_EOL . '<body>';
                $sWidgetHtml .= $this->aValues['widgetFileContent'];
                $sWidgetHtml .= '' . PHP_EOL . '</body>' . PHP_EOL . '</html>';
                file_put_contents($this->aProperties["vas_home"] . '/ws_data/gtf/widget/' . $sWorkspace_key . ".html", $sWidgetHtml);
            }
            $aXmlRacineAttribute['status'] = (int) !$bError;
            $sMessage = $this->asDocument('', 'vitis', $this->aValues['sEncoding'], True, $aXmlRacineAttribute, $this->aValues['sSourceEncoding'], $this->aValues['output']);
            return $sMessage;
        }
        if (!empty($this->aValues['mode']) && $this->aValues['mode'] == "import") {
            for ($i = 0; $i < count($_FILES['fmw_file']['name']); $i++) {
                $aParams = array();
                $aParams['sSchemaGtf'] = array('value' => $this->aProperties['schema_gtf'], 'type' => 'schema_name');
                $aParams['sFileName'] = array('value' => str_replace(".fmw", "", str_replace(".fme", "", $_FILES['fmw_file']['name'][$i])), 'type' => 'string');
                $oPDOresult = $this->oConnection->oBd->executeWithParams($aSql['getWorkspaceId'], $aParams);
                if ($aWorkspace = $this->oConnection->oBd->ligneSuivante($oPDOresult)) {
                    $this->aValues['name'] = str_replace(".fmw", "", str_replace(".fme", "", $_FILES['fmw_file']['name'][$i])) . gmdate("Ymd-His") . rand(0, 100);
                } else {
                    $this->aValues['name'] = str_replace(".fmw", "", str_replace(".fme", "", $_FILES['fmw_file']['name'][$i]));
                }

                // **** Récupération de user_id

                $this->aValues['subscription'] = 'FALSE';
                $this->aValues['fmw_file'] = $_FILES['fmw_file']['name'][$i];
                $this->aValues['tag'] = 'default';
                $this->aValues['survey'] = 'FALSE';
                $this->aValues['form_type'] = 'default';
                $this->aValues['creation_date'] = gmdate("Y-m-d H:i:s");
                // Id de l'utilisateur connecté.
                $aParams = array();
                $aParams['sSchemaFramework'] = array('value' => $this->aProperties["schema_framework"], 'type' => 'schema_name');
                $aParams['sLogin'] = array('value' => $_SESSION['ses_Login'], 'type' => 'string');
                $oPDOresult = $this->oConnection->oBd->executeWithParams($aSql['getOwnerId'], $aParams);
                $oOwner = $this->oConnection->oBd->objetSuivant($oPDOresult);
                $this->aValues['owner'] = $oOwner->user_id;
                $oPDOresult = $this->oConnection->oBd->fermeResultat();
                // Création de la clé du projet.
                $bExistKey = false;
                while ($bExistKey == false) {
                    // En majuscules et suppression du point.
                    $this->aValues['key'] = strtoupper(base_convert(rand(1000000000, 9999999999) . time(), 10, 36));
                    $aParams = array();
                    $aParams['sSchemaGtf'] = array('value' => $this->aProperties["schema_gtf"], 'type' => 'schema_name');
                    $aParams['sKey'] = array('value' => $this->aValues['key'], 'type' => 'string');
                    $oPDOresult = $this->oConnection->oBd->executeWithParams($aSql['getWorkspaceId2'], $aParams);
                    if (!$aWorkspace = $this->oConnection->oBd->ligneSuivante($oPDOresult))
                        $bExistKey = true;
                }

                $aReturn = $this->genericPost($this->aProperties['schema_gtf'], 'workspace', $this->aProperties['schema_gtf'] . '.seq_common', 'workspace_id');
                $oWorkspace = new Workspace($this->aPath, $this->aValues, $this->aProperties, $this->oConnection);
                $oWorkspace->GET();
                $sFilePathName = $this->aProperties['workspace_dir'] . "/" . $oWorkspace->aFields["workspace_id"] . "/fme/" . $_FILES['fmw_file']['name'][$i];
                $sDirName = $this->aProperties['workspace_dir'] . "/" . $oWorkspace->aFields["workspace_id"];
                // Si le répertoire existe déja : suppression.
                if (is_dir($sDirName))
                    clearDir($sDirName);
                //
                mkdir($sDirName, 0777, true);
                mkdir($sDirName . "/form", 0777, true);
                mkdir($sDirName . "/form/ressources", 0777, true);
                mkdir($sDirName . "/fme", 0777, true);

                if (!move_uploaded_file($_FILES['fmw_file']['tmp_name'][$i], utf8_decode($sFilePathName))) {
                    writeToErrorLog(ERROR_0031 . $_FILES['file']['name']);
                    $aReturn = array(sStatus => 1, sMessage => WORKSPACE_ERROR_COPYING_FILE);
                } else {
                    // Fichier .fmw original en .bak.
                    copy($sDirName . "/fme/" . utf8_decode($this->aValues['fmw_file']), $sDirName . "/fme/" . utf8_decode($this->aValues['fmw_file']) . '.bak');
                    //
                    fclose(fopen($sDirName . "/form/ressources/Subform.js", "w+"));
                    fclose(fopen($sDirName . "/form/ressources/Subform.css", "w+"));
                    $oFmwParser = new GtfFmwParser($sDirName . "/fme/" . $_FILES['fmw_file']['name'][$i]);
                    // Sauve le .fmw sans les visualizers.
                    $oFmwParser->save($sDirName . "/fme/" . utf8_decode($this->aValues['fmw_file']));
                    // Création des formulaires json.
                    $aJson = $oFmwParser->productJson($this->aProperties, $oWorkspace->aFields["name"]);
                    $fJson = fopen($sDirName . "/form/Subform.json", "w+");
                    if (fwrite($fJson, json_encode($aJson)) == FALSE) {
                        if ($fJson == FALSE) {
                            writeToErrorLog("Subform.json can't be opened : " . $sDirName . "/form/ressources/Subform.json" . "(Workspaces.class.inc : verify the property workspace_dir)");
                        } else {
                            writeToErrorLog("Subform.json can't be writed : " . $sDirName . "/form/ressources/Subform.json" . "(Workspaces.class.inc : verify the folder's rights)");
                        }
                    }
                    fclose($fJson);
                    copy($sDirName . "/form/Subform.json", $sDirName . "/form/WSubform.json");
                    copy($sDirName . "/form/Subform.json", $sDirName . "/form/DSubform.json");
                    $oFmwParser->updateMetadata($this->oConnection->oBd, $oWorkspace->aFields["workspace_id"]);
                    $this->aValues["errorkeys"] = "";
                }
                $sMessage = $aReturn['sMessage'];
            }
        } else {
            $this->aValues["fmw_file"] = $_FILES['fmw_file']['name'];
            $aParams = array();
            $aParams['sSchemaFramework'] = array('value' => $this->aProperties["schema_framework"], 'type' => 'schema_name');
            $aParams['sLogin'] = array('value' => $_SESSION['ses_Login'], 'type' => 'string');
            $oPDOresult = $this->oConnection->oBd->executeWithParams($aSql['getOwnerId'], $aParams);
            $oOwner = $this->oConnection->oBd->objetSuivant($oPDOresult);
            $this->aValues['owner'] = $oOwner->user_id;
            $oPDOresult = $this->oConnection->oBd->fermeResultat();
            // Le nom du projet n'est pas spécifié.
            if (empty($this->aValues["name"])) {
                $aParams = array();
                $aParams['sSchemaGtf'] = array('value' => $this->aProperties['schema_gtf'], 'type' => 'schema_name');
                $aParams['sFileName'] = array('value' => str_replace(".fmw", "", str_replace(".fme", "", $_FILES['fmw_file']['name'])), 'type' => 'string');
                $oPDOresult = $this->oConnection->oBd->executeWithParams($aSql['getWorkspaceId'], $aParams);
                if ($aWorkspace = $this->oConnection->oBd->ligneSuivante($oPDOresult)) {
                    $this->aValues['name'] = str_replace(".fmw", "", str_replace(".fme", "", $_FILES['fmw_file']['name'])) . gmdate("Ymd-His") . rand(0, 100);
                } else {
                    $this->aValues['name'] = str_replace(".fmw", "", str_replace(".fme", "", $_FILES['fmw_file']['name']));
                }
            }
            // Vérifie si un projet existe déja avec ce nom.
            $aParams = array();
            $aParams['sSchemaGtf'] = array('value' => $this->aProperties['schema_gtf'], 'type' => 'schema_name');
            $aParams['name'] = array('value' => $this->aValues['name'], 'type' => 'string');
            $oPDOresult = $this->oConnection->oBd->executeWithParams($aSql['checkWorkspaceName'], $aParams);
            if (!$this->oConnection->oBd->enErreur()) {
                if ($this->oConnection->oBd->nombreLigne($oPDOresult) > 0) {
                    $oError = new VitisError(1, "FORM_WORKSPACE_NAME_EXISTS_PUBLICATION_WORKSPACE");
                    $aXmlRacineAttribute['status'] = 0;
                    $sMessage = $oError->asDocument('', 'vitis', $this->aValues['sEncoding'], True, $aXmlRacineAttribute, $this->aValues['sSourceEncoding'], $this->aValues['output']);
                }          
                else {
                    $this->aValues['creation_date'] = gmdate('Y-m-d H:i:s');
                    // Création de la clé du projet.
                    $bExistKey = false;
                    while ($bExistKey == false) {
                        // En majuscules et suppression du point.
                        $this->aValues['key'] = strtoupper(base_convert(rand(1000000000, 9999999999) . time(), 10, 36));
                        $aParams = array();
                        $aParams['sSchemaGtf'] = array('value' => $this->aProperties["schema_gtf"], 'type' => 'schema_name');
                        $aParams['sKey'] = array('value' => $this->aValues['key'], 'type' => 'string');
                        $oPDOresult = $this->oConnection->oBd->executeWithParams($aSql['getWorkspaceId2'], $aParams);
                        if (!$aWorkspace = $this->oConnection->oBd->ligneSuivante($oPDOresult))
                            $bExistKey = true;
                    }
                    $aReturn = $this->genericPost($this->aProperties['schema_gtf'], 'workspace', $this->aProperties['schema_gtf'] . '.seq_common', 'workspace_id');
                    if ($aReturn['sStatus'] == 1) {
                        $aXmlRacineAttribute['status'] = 1;
                        $sMessage = $this->asDocument('', 'vitis', $this->aValues['sEncoding'], True, $aXmlRacineAttribute, $this->aValues['sSourceEncoding'], $this->aValues['output']);
                        $oWorkspace = new Workspace($this->aPath, $this->aValues, $this->aProperties, $this->oConnection);
                        $oWorkspace->GET();
                        // Upload de fichiers ?
                        if (!empty($_FILES)) {
                            $this->aValues["fmw_file"] = $_FILES['fmw_file']['name'];
                            $this->aValues["comp_file"] = '';
                            if (!empty($_FILES["comp_file"]))
                                $this->aValues["comp_file"] = $_FILES["comp_file"]["name"]; //si pas de zip la chaine est vide
                            $aExt = explode(".", $this->aValues["fmw_file"]);
                            if ($aExt[count($aExt) - 1] == "fmw") {
                                $this->aValues['creation_date'] = gmdate('Y-m-d H:i:s');
                                $this->aValues['owner'] = $_SESSION["ses_user_id"];
                                $this->aValues['form_type'] = 'custom';
                                if ($this->aValues["name"] == "") {
                                    $sSQL = "SELECT workspace_id FROM " . $this->aProperties['schema_gtf'] . ".workspace WHERE fmw_file = [fmw_file]";
                                    $aParams = array();
                                    $aParams['fmw_file'] = array('value' => $this->aValues["fmw_file"], 'type' => 'string');
                                    $oPDOresult = $this->oConnection->oBd->executeWithParams($aSql['getCategoryTraitement'], $aParams);
                                    for ($i = 0; $i < count($aExt) - 1; $i++) {
                                        $sFilename = $sFilename . $aExt[$i];
                                    }
                                    if (!$this->oConnection->oBd->enErreur() && $this->oConnection->oBd->nombreLigne($oPDOresult) > 0) {
                                        $this->aValues['name'] = $sFilename . gmdate("Ymd-His") . rand(0, 100);
                                    } else {
                                        $this->aValues['name'] = $sFilename;
                                    }
                                }
                                $sDirName = $this->aProperties['workspace_dir'] . "/" . $oWorkspace->aFields["workspace_id"];
                                // Si le répertoire existe déja : suppression.
                                if (is_dir($sDirName))
                                    clearDir($sDirName);
                                //
                                mkdir($sDirName, 0777, true);
                                mkdir($sDirName . "/form", 0777, true);
                                mkdir($sDirName . "/form/ressources", 0777, true);
                                mkdir($sDirName . "/fme", 0777, true);
                                uploadFile('fmw_file', "fmw", $sDirName . "/fme/" . $_FILES['fmw_file']['name'], $_FILES['fmw_file']['size'] + 1);
                                // Fichier .fmw original en .bak.
                                copy($sDirName . "/fme/" . utf8_decode($_FILES['fmw_file']['name']), $sDirName . "/fme/" . utf8_decode($_FILES['fmw_file']['name']) . '.bak');
                                //
                                fclose(fopen($sDirName . "/form/ressources/Subform.js", "w+"));
                                fclose(fopen($sDirName . "/form/ressources/Subform.css", "w+"));
                                if ($this->aValues["comp_file"] != "") {
                                    uploadFile('comp_file', "", $sDirName . "/fme/" . $_FILES['comp_file']['name'], $_FILES['comp_file']['size'] + 1);
                                    $ext = pathinfo($_FILES['comp_file']['name'], PATHINFO_EXTENSION);
                                    if ($ext == "zip") {
                                        unZip($sDirName . "/fme/" . utf8_decode($_FILES['comp_file']['name']), $sDirName . "/fme/");
                                    }
                                }
                                $oFmwParser = new GtfFmwParser($sDirName . "/fme/" . $_FILES['fmw_file']['name']);
                                // Sauve le .fmw sans les visualizers.
                                $oFmwParser->save($sDirName . "/fme/" . utf8_decode($_FILES['fmw_file']['name']));
                                // Création des formulaires json.
                                $aJson = $oFmwParser->productJson($this->aProperties, $oWorkspace->aFields["name"]);
                                $fJson = fopen($sDirName . "/form/Subform.json", "w+");
                                if (fwrite($fJson, json_encode($aJson)) == FALSE) {
                                    if ($fJson == FALSE) {
                                        writeToErrorLog("Subform.json can't be opened : " . $sDirName . "/form/ressources/Subform.json" . "(Workspaces.class.inc : verify the property workspace_dir)");
                                    } else {
                                        writeToErrorLog("Subform.json can't be writed : " . $sDirName . "/form/ressources/Subform.json" . "(Workspaces.class.inc : verify the folder's rights)");
                                    }
                                }
                                fclose($fJson);
                                copy($sDirName . "/form/Subform.json", $sDirName . "/form/WSubform.json");
                                copy($sDirName . "/form/Subform.json", $sDirName . "/form/DSubform.json");
                                $oFmwParser->updateMetadata($this->oConnection->oBd, $oWorkspace->aFields["workspace_id"]);
                                /*                         * ************************************************************** */
                            } else {
                                writeToErrorLog("the uploaded file is not a fmw file or no file uploaded (Workspaces.class.inc : user_id->" . $_SESSION["user_id"] . ")");
                                $sMessage = $this->asDocument('', 'vitis', $this->aValues['sEncoding'], True, $aXmlRacineAttribute, $this->aValues['sSourceEncoding'], $this->aValues['output']);
                            }
                        }
                    } else
                        $sMessage = $aReturn['sMessage'];
                }
            }
        };
        return $sMessage;
    }

    /**
     * @SWG\Put(path="/workspaces/{workspace_id}",
     *   tags={"Workspaces"},
     *   summary="Update Workspace",
     *   description="Request to update Workspace",
     *   operationId="PUT",
     *  consumes= { "multipart/form-data"},
     *   produces={"application/xml", "application/json"},
     *   @SWG\Parameter(
     *     name="token",
     *     in="query",
     *     description="workspace token",
     *     required=true,
     *     type="string"
     *   ),
     * * @SWG\Parameter(
     *     name="workspace_id",
     *     in="path",
     *     description="id of the workspace",
     *     required=true,
     *     type="string",
     *   ),
     *   @SWG\Parameter(
     *     name="name",
     *     in="query",
     *     description="Name of the workspace",
     *     required=false,
     *     type="string"
     *   ),
     *   @SWG\Parameter(
     *     name="fmw_file",
     *     in="formData",
     *     description="Fmw file of the workspace",
     *     required=false,
     *     type="file"
     *   ),
     *   @SWG\Parameter(
     *     name="comp_file",
     *     in="formData",
     *     description="Other file (.zip) of the workspace",
     *     required=false,
     *     type="file"
     *   ),
     *   @SWG\Parameter(
     *     name="json_file",
     *     in="formData",
     *     description="Wsubform.json",
     *     required=false,
     *     type="file"
     *   ),
     *  @SWG\Parameter(
     *     name="js_file",
     *     in="formData",
     *     description="Javascript file link with WSubform",
     *     required=false,
     *     type="file"
     *   ),
     *  @SWG\Parameter(
     *     name="css_file",
     *     in="formData",
     *     description="Css file link with WSubform",
     *     required=false,
     *     type="file"
     *   ),
     *   @SWG\Parameter(
     *     name="tag",
     *     in="query",
     *     description="Tag of the workspace",
     *     required=false,
     *     type="string"
     *   ),
     *   @SWG\Parameter(
     *     name="cmd",
     *     in="query",
     *     description="command send from the studio",
     *     required=false,
     *     type="string"
     *   ),
     *   @SWG\Parameter(
     *     name="action",
     *     in="query",
     *     description="command send from the studio",
     *     required=false,
     *     type="string"
     *   ),
     *   @SWG\Parameter(
     *     name="email_template_id",
     *     in="query",
     *     description="Email template of the workspace",
     *     required=false,
     *     type="integer",
     *     format="int32"
     *   ),
     *   @SWG\Parameter(
     *     name="subscription",
     *     in="query",
     *     description="Subscription ?",
     *     required=false,
     *     default=false,
     *     type="boolean"
     *   ),
     *   @SWG\Parameter(
     *     name="survey",
     *     in="query",
     *     description="Survey ?",
     *     required=false,
     *     default=false,
     *     type="boolean"
     *   ),
     *   @SWG\Parameter(
     *     name="locked",
     *     in="query",
     *     description="Locked ?",
     *     required=false,
     *     default=false,
     *     type="boolean"
     *   ),
     *   @SWG\Response(
     *         response=200,
     *         description="Poprerties Response",
     *         @SWG\Schema(ref="#/definitions/workspaces")
     *     )
     *  )
     */

    /**
     * update workspace
     * @return id of the workspace if ok error object if ko
     */
    function PUT() {
        if (!empty($this->aValues["action"]) && $this->aValues["action"] == "Metadata") {
            $aReturn = $this->reintegrateMetadata();

            $aXmlRacineAttribute['status'] = $aReturn['status'];
            $sMessage = $this->asDocument('', 'vitis', $this->aValues['sEncoding'], True, $aXmlRacineAttribute, $this->aValues['sSourceEncoding'], $this->aValues['output']);
        } else {
            $sDirName = $this->aProperties['workspace_dir'] . "/" . (string) $this->aValues["my_vitis_id"];


            $sDirName = $this->aProperties['workspace_dir'] . "/" . (string) $this->aValues["my_vitis_id"];

            $error_message = "";

            // Si le dossier {id} n'existe pas (Important pour gérer les erreurs)
            if (!is_dir($sDirName)) {
                $error_message .= $sDirName . " does not exist";
                writeToErrorLog("WARNING: " . $error_message);

                $aXmlRacineAttribute['status'] = 0;
                $this->aFields = array();
                $this->aFields['errorMessage'] = $error_message;
                $sMessage = $this->asDocument('', 'vitis', $this->aValues['sEncoding'], True, $aXmlRacineAttribute, $this->aValues['sSourceEncoding'], $this->aValues['output']);
                return $sMessage;
            }
            // Si le dossier {id}/fme n'existe pas (Important pour gérer les erreurs)
            if (!is_dir($sDirName . "/fme")) {
                $error_message .= $sDirName . "/fme does not exist";
                writeToErrorLog("WARNING: " . $error_message);

                $aXmlRacineAttribute['status'] = 0;
                $this->aFields = array();
                $this->aFields['errorMessage'] = $error_message;
                $sMessage = $this->asDocument('', 'vitis', $this->aValues['sEncoding'], True, $aXmlRacineAttribute, $this->aValues['sSourceEncoding'], $this->aValues['output']);
                return $sMessage;
            }

            // Si le dossier {id}/form n'existe pas il est crée
            if (!is_dir($sDirName . "/form")) {
                writeToErrorLog("WARNING: " . $sDirName . "/form dit not exist");
                @mkdir($sDirName . "/form");
            }

            if (!empty($this->aValues["cmd"])) {
                switch ($this->aValues["cmd"]) {

                    // DSubform: formulaire par défaut
                    // Subform: formulaire publié
                    // WSubform: formulaire personnalié
                    // Remplace le formulaire publié par celui par défaut
                    case "Default_Published":
                        // Si le fichier par défaut n'existe pas, il est généré
                        if (!file_exists($sDirname . "/form/DSubform.json")) {
                            $this->aValues["cmd"] = "Default_Reset";
                            $this->PUT();
                        }
                        @unlink($sDirname . "/form/Subform.json");
                        copy($sDirName . "/form/DSubform.json", $sDirName . "/form/Subform.json");
                        $this->aValues["form_type"] = 'default';
                        break;
                    // Regénère le formulaire par défaut en fonction du fmw
                    case "Default_Reset":
                        $oWorkspace = new Workspace($this->aPath, $this->aValues, $this->aProperties, $this->oConnection);
                        $oWorkspace->GET();

                        if (empty($oWorkspace->aFields["fmw_file"])) {
                            $error_message .= "Any FME file founded in" . $sDirName . "/fme/ ";
                            writeToErrorLog("WARNING: " . $error_message);

                            $aXmlRacineAttribute['status'] = 0;
                            $this->aFields = array();
                            $this->aFields['errorMessage'] = $error_message;
                            $sMessage = $this->asDocument('', 'vitis', $this->aValues['sEncoding'], True, $aXmlRacineAttribute, $this->aValues['sSourceEncoding'], $this->aValues['output']);
                            return $sMessage;
                        }

                        $oFmwParser = new GtfFmwParser($sDirName . "/fme/" . $oWorkspace->aFields["fmw_file"]);
                        $aJson = $oFmwParser->productJson($this->aProperties, $oWorkspace->aFields["name"]);
                        $fJson = fopen($sDirName . "/form/DSubform.json", "w");
                        if (fwrite($fJson, json_encode($aJson)) == FALSE) {
                            if ($fJson == FALSE) {
                                writeToErrorLog("Subform.json can't be opened : " . $sDirName . "/form/ressources/Subform.json" . "(Workspaces.class.inc : verify the property workspace_dir)");
                            } else {
                                writeToErrorLog("Subform.json can't be writed : " . $sDirName . "/form/ressources/Subform.json" . "(Workspaces.class.inc : verify the folder's rights)");
                            }
                        }
                        fclose($fJson);
                        //$oFmwParser->updateMetadata($this->oConnection->oBd, $oWorkspace->aFields["workspace_id"]);
                        break;
                    // Remplace le formulaire publié par celui personnalisé
                    case "Perso_Published" :
                        @unlink($sDirname . "/form/Subform.json");
                        copy($sDirName . "/form/WSubform.json", $sDirName . "/form/Subform.json");
                        $this->aValues["form_type"] = 'custom';
                        break;
                    // Remplace le fornulaire personnalisé par le formulaire par défaut
                    case "Perso_Reset":
                        // Si le fichier par défaut n'existe pas, il est généré
                        if (!file_exists($sDirname . "/form/DSubform.json")) {
                            $this->aValues["cmd"] = "Default_Reset";
                            $this->PUT();
                        }
                        @unlink($sDirname . "/form/WSubform.json");
                        copy($sDirName . "/form/DSubform.json", $sDirName . "/form/WSubform.json");
                        break;
                    // Sauvegarde le formulaire personnalisé
                    case "Perso_Save":
                        $pFile = fopen($sDirName . "/form/WSubform.json", "w");
                        if (fwrite($pFile, $this->aValues["Json"]) == FALSE) {
                            if ($fJson == FALSE) {
                                writeToErrorLog("Subform.json can't be opened : " . $sDirName . "/form/ressources/Subform.json" . "(Workspaces.class.inc : verify the property workspace_dir)");
                            } else {
                                writeToErrorLog("Subform.json can't be writed : " . $sDirName . "/form/ressources/Subform.json" . "(Workspaces.class.inc : verify the folder's rights)");
                            }
                        }
                        fclose($pFile);

                        if ($this->aValues["Js"] != "") {
                            $pFileJS = fopen($sDirName . "/form/ressources/Subform.js", "w+");
                            if (fwrite($pFileJS, $this->aValues["Js"]) == FALSE) {
                                if ($pFileJS == FALSE) {
                                    writeToErrorLog("Subform.js can't be opened : " . $sDirName . "/form/ressources/Subform.js" . "(Workspaces.class.inc : verify the property workspace_dir)");
                                } else {
                                    writeToErrorLog("Subform.js can't be writed : " . $sDirName . "/form/ressources/Subform.js" . "(Workspaces.class.inc : verify the folder's rights)");
                                }
                            }
                            fclose($pFileJS);
                        }

                        if ($this->aValues["Css"] != "") {
                            $pFileCSS = fopen($sDirName . "/form/ressources/Subform.css", "w+");
                            if (fwrite($pFileCSS, $this->aValues["Css"]) == FALSE) {
                                if ($pFileCSS == FALSE) {
                                    writeToErrorLog("Subform.css can't be opened : " . $sDirName . "/form/ressources/Subform.css" . "(Workspaces.class.inc : verify the property workspace_dir)");
                                } else {
                                    writeToErrorLog("Subform.css can't be writed : " . $sDirName . "/form/ressources/Subform.css" . "(Workspaces.class.inc : verify the folder's rights)");
                                }
                            }
                            fclose($pFileCSS);
                        }
                        break;
                    default : writeToErrorLog("ERROR  : Bad request from Client : " . $this->aValues["cmd"] . " form user ->" . $_SESSION["user_id"]);
                }
            }
            if (!empty($this->aValues["fmw_file_file"])) {
                $this->aValues["fmw_file"] = $this->aValues["fmw_file_name"];
            }
            if (!empty($this->aValues["comp_file_file"])) {
                $this->aValues["comp_file"] = $this->aValues["comp_file_name"];
                $fp = fopen($sDirName . "/fme/" . utf8_decode($this->aValues["comp_file_name"]), "w");
                fwrite($fp, $this->aValues["comp_file_file"]);
                fclose($fp);
                $ext = pathinfo($sDirName . "/fme/" . utf8_decode($this->aValues["comp_file_name"]), PATHINFO_EXTENSION);
                if ($ext == "zip") {
                    unZip($sDirName . "/fme/" . utf8_decode($this->aValues["comp_file_name"]), $sDirName . "/fme/");
                }
            }
            $aReturn = $this->genericPut($this->aProperties['schema_gtf'], 'workspace', 'workspace_id');
            if ($aReturn["sStatus"] == 1) {
                if (!empty($this->aValues["groups"])) {
                    require $this->sRessourcesFile;
                    $this->oConnection->oBd->delete($this->aProperties['schema_gtf'], 'workspace_group', 'workspace_id', $this->aValues["my_vitis_id"]);
                    // Traitements à rattacher au groupe ?
                    if (isset($this->aValues['groups'])) {
                        $aGroups = explode('|', $this->aValues['groups']);
                        foreach ($aGroups as $iGroupId) {
                            $aParams = array();
                            $aParams['sSchemaGtf'] = array('value' => $this->aProperties['schema_gtf'], 'type' => 'schema_name');
                            $aParams['workspace_id'] = array('value' => $this->aValues['my_vitis_id'], 'type' => 'number');
                            $aParams['group_id'] = array('value' => $iGroupId, 'type' => 'number');
                            $oPDOresult = $this->oConnection->oBd->executeWithParams($aSql['insertGroupWorkspaces'], $aParams);
                        }
                    }
                }
                if (!empty($this->aValues["fmw_file_file"])) {
                    if ($aReturn['sStatus'] == 1) {
                        $aXmlRacineAttribute['status'] = 1;
                        $sMessage = $this->asDocument('', 'vitis', $this->aValues['sEncoding'], True, $aXmlRacineAttribute, $this->aValues['sSourceEncoding'], $this->aValues['output']);
                        $oWorkspace = new Workspace($this->aPath, $this->aValues, $this->aProperties, $this->oConnection);
                        $oWorkspace->GET();
                        $aExt = explode(".", $this->aValues["fmw_file_name"]);
                        if ($aExt[count($aExt) - 1] == "fmw") {
                            // Sauve le nouveau fichier fmw.
                            $fp = fopen($sDirName . "/fme/" . utf8_decode($this->aValues["fmw_file_name"]), "w");
                            fwrite($fp, $this->aValues["fmw_file_file"]);
                            fclose($fp);
                            // Fichier .fmw original en .bak.
                            copy($sDirName . "/fme/" . utf8_decode($this->aValues['fmw_file_name']), $sDirName . "/fme/" . utf8_decode($this->aValues['fmw_file_name']) . '.bak');
                            // Création du nouveau formulaire.
                            $oFmwParser = new GtfFmwParser($sDirName . "/fme/" . $this->aValues["fmw_file_name"]);
                            // Sauve le .fmw sans les visualizers.
                            $oFmwParser->save($sDirName . "/fme/" . utf8_decode($this->aValues["fmw_file_name"]));
                            // Création des formulaires json.
                            $aJson = $oFmwParser->productJson($this->aProperties, $oWorkspace->aFields["name"]);
                            $fJson = fopen($sDirName . "/form/DSubform.json", "w");
                            if (fwrite($fJson, json_encode($aJson)) == FALSE) {
                                if ($fJson == FALSE) {
                                    writeToErrorLog("Subform.json can't be opened : " . $sDirName . "/form/ressources/Subform.json" . "(Workspaces.class.inc : verify the property workspace_dir)");
                                } else {
                                    writeToErrorLog("Subform.json can't be writed : " . $sDirName . "/form/ressources/Subform.json" . "(Workspaces.class.inc : verify the folder's rights)");
                                }
                            }
                            fclose($fJson);
                            if ($oWorkspace->aFields["form_type"] == 'default') {
                                copy($sDirName . "/form/DSubform.json", $sDirName . "/form/Subform.json");
                                copy($sDirName . "/form/DSubform.json", $sDirName . "/form/WSubform.json");
                            }
                            // Récupère les anciens paramètres.
                            $oFmwParser->getBdMetadata($this->oConnection->oBd, $this->aValues["my_vitis_id"]);
                            // Mise à jour des paramètres du nouveau fichier dans la base.
                            $oFmwParser->sSaveDate = gmdate('Y-m-d G:i:s');
                            $oFmwParser->updateMetadata($this->oConnection->oBd, $this->aValues["my_vitis_id"]);
                        }
                    } else {
                        $aXmlRacineAttribute['status'] = 1;
                        $sMessage = $aReturn['sMessage'];
                    }
                } else if (!empty($error_message)) {
                    $aXmlRacineAttribute['status'] = 0;
                    $this->aFields = array();
                    $this->aFields['errorMessage'] = $error_message;
                    $sMessage = $this->asDocument('', 'vitis', $this->aValues['sEncoding'], True, $aXmlRacineAttribute, $this->aValues['sSourceEncoding'], $this->aValues['output']);
                } else {
                    $aXmlRacineAttribute['status'] = 1;
                    $sMessage = $this->asDocument('', 'vitis', $this->aValues['sEncoding'], True, $aXmlRacineAttribute, $this->aValues['sSourceEncoding'], $this->aValues['output']);
                }
            } else
                $sMessage = $aReturn["sMessage"];
        }

        return $sMessage;
    }

    /**
     * @SWG\Delete(path="/workspaces/",
     *   tags={"Workspaces"},
     *   summary="Delete Workspace",
     *   description="Request to delete Workspace",
     *   operationId="DELETE",
     *   produces={"application/xml", "application/json"},
     *   @SWG\Parameter(
     *     name="token",
     *     in="query",
     *     description="workspace token",
     *     required=true,
     *     type="string"
     *   ),
     * * @SWG\Parameter(
     *     name="idList",
     *     in="query",
     *     description="id of the workspaces",
     *     required=true,
     *     type="string"
     *   ),
     * @SWG\Response(
     *         response=200,
     *         description="Poprerties Response",
     *         @SWG\Schema(ref="#/definitions/workspaces")
     *     )
     *  )
     */
    /**
     * @SWG\Delete(path="/workspaces/{workspace_id}",
     *   tags={"Workspaces"},
     *   summary="delete Workspace",
     *   description="Request to delete Workspace",
     *   operationId="DELETE",
     *   produces={"application/xml", "application/json"},
     *   @SWG\Parameter(
     *     name="token",
     *     in="query",
     *     description="workspace token",
     *     required=true,
     *     type="string"
     *   ),
     * * @SWG\Parameter(
     *     name="workspace_id",
     *     in="path",
     *     description="id of the workspace",
     *     required=true,
     *     type="string",
     *   ),
     * @SWG\Response(
     *         response=200,
     *         description="Poprerties Response",
     *         @SWG\Schema(ref="#/definitions/workspaces")
     *     )
     *  )
     */
    /**
     * @SWG\Delete(path="/workspaces/{workspace_id}/File",
     *   tags={"Workspaces"},
     *   summary="delete Workspace directory file",
     *   description="Request to delete a file of the Workspace directory",
     *   operationId="DELETE",
     *   produces={"application/xml", "application/json"},
     *   @SWG\Parameter(
     *     name="token",
     *     in="query",
     *     description="user token",
     *     required=true,
     *     type="string"
     *   ),
     *   @SWG\Parameter(
     *     name="workspace_id",
     *     in="path",
     *     description="workspace id",
     *     required=true,
     *     type="string"
     *   ),
     *   @SWG\Parameter(
     *     name="file_name",
     *     in="query",
     *     description="File name",
     *     required=true,
     *     type="string",
     *   ),
     * @SWG\Response(
     *         response=200,
     *         description="Poprerties Response",
     *         @SWG\Schema(ref="#/definitions/workspaces")
     *     )
     *  )
     */
    /**
     * @SWG\Delete(path="/workspaces/{workspace_id}/DeleteWidgetFile",
     *   tags={"Workspaces"},
     *   summary="delete Widget file",
     *   description="Request to delete the file of the Widget",
     *   operationId="DELETE",
     *   produces={"application/xml", "application/json"},
     *   @SWG\Parameter(
     *     name="token",
     *     in="query",
     *     description="user token",
     *     required=true,
     *     type="string"
     *   ),
     *   @SWG\Parameter(
     *     name="workspace_id",
     *     in="path",
     *     description="workspace id",
     *     required=true,
     *     type="string"
     *   ),
     * @SWG\Response(
     *         response=200,
     *         description="Poprerties Response",
     *         @SWG\Schema(ref="#/definitions/workspaces")
     *     )
     *  )
     */

    /**
     * delete workspace
     * @return id of workspace deleted or error object if a workspace is not deleted
     */
    function DELETE() {
        if (!empty($this->aPath[3])) {
            if ($this->aPath[3] == "File") {
                $aReturn = $this->deleteWorkspaceFile();
                $aXmlRacineAttribute['status'] = $aReturn['status'];
                if ($aReturn['status'] == 1) {
                    $sMessage = $this->asDocument('', 'vitis', $this->aValues['sEncoding'], True, $aXmlRacineAttribute, $this->aValues['sSourceEncoding'], $this->aValues['output']);
                } else {
                    $oError = new VitisError(1, 'LICENSE_FILE_INSTALL_ERROR_CONFIGURATION_CONFIGURATION_GTF_LICENSE');
                    $sMessage = $oError->asDocument('', 'vitis', $this->aValues['sEncoding'], True, $aXmlRacineAttribute, $this->aValues['sSourceEncoding'], $this->aValues['output']);
                }
            } elseif ($this->aPath[3] == "DeleteWidgetFile") {
                unset($this->aPath[3]);
                $oWorkspace = new Workspace($this->aPath, $this->aValues, $this->aProperties, $this->oConnection);
                $oWorkspace->GET();
                $sWorkspace_key = $oWorkspace->aFields["key"];
                // Emplacement des widgets
                $sWidgetFilePath = $this->aProperties["vas_home"] . '/ws_data/gtf/widget/' . $sWorkspace_key . ".html";
                $bError = False;
                if (is_file($sWidgetFilePath)) {
                    if (!unlink($sWidgetFilePath)) {
                        $bError = True;
                        $this->aFields['errorMessage'] = "Failed to remove the widget " . $sWidgetFilePath;
                    }
                }
                $aXmlRacineAttribute['status'] = (int) !$bError;
                $sMessage = $this->asDocument('', 'vitis', $this->aValues['sEncoding'], True, $aXmlRacineAttribute, $this->aValues['sSourceEncoding'], $this->aValues['output']);
                return $sMessage;
            }
        } else {
            $aReturn = $this->genericDelete($this->aProperties['schema_gtf'], 'workspace', $this->aValues['idList']);
            $sMessage = $aReturn['sMessage'];
        }
        return $sMessage;
    }

    /**
     * Get workspace directory tree.
     */
    function getWorkspaceDirectoryTree() {
        $aReturn = array('status' => 1, 'message' => '');
        $sWorkspaceDir = $this->aProperties['workspace_dir'] . '/' . $this->aValues["my_vitis_id"] . '/fme';
        $sTreeName = 'fme';
        if (is_dir($sWorkspaceDir)) {
            $aDirStructure = $this->getDirStructure($sWorkspaceDir, $sTreeName);
            $aTreeviewLogs["data"][] = $aDirStructure["data"];
            foreach ($aDirStructure["files"] as $aFiles) {
                $aTreeviewLogs["files"][] = $aFiles;
            }
            $this->aFields['tree'] = $aTreeviewLogs;
        } else
            writeToErrorLog('scandir(' . $sWorkspaceDir . '): failed to open dir');
        return $aReturn;
    }

    /*
     * Retourne l'arborescence d'un répertoire formaté pour "bootstrap-treeview".
     */

    function getDirStructure($sWorkspaceDir, $sDirName) {
        $aFilteredDir['data'] = array('text' => $sDirName, 'selectable' => false);
        $aDir = scandir($sWorkspaceDir);
        foreach ($aDir as $sName) {
            $sPath = $sWorkspaceDir . '/' . $sName;
            $sFileType = filetype($sPath);
            // Répertoire ?
            if ($sFileType == 'dir') {
                if ($sName != '.' && $sName != '..') {
                    if (is_dir($sPath)) {
                        $aDirStructure = $this->getDirStructure($sPath . '/', $sName);
                        if (!empty($aDirStructure['data']['nodes'])) {
                            // Structure du répertoire.
                            $aFilteredDir['data']['nodes'][] = $aDirStructure['data'];
                            // Liste des fichiers .log du répertoire.
                            foreach ($aDirStructure["files"] as $aFiles) {
                                $aFilteredDir["files"][] = $aFiles;
                            }
                        }
                    } else
                        writeToErrorLog('scandir(' . $sPath . '): failed to open dir');
                }
            }
            // fichier ?
            if ($sFileType == 'file') {
                // Taille + unité du fichier.
                //$iFileSize = filesize($sPath);
                $aFileInfos = stat($sPath);
                $iFileSizeLabel = $aFileInfos['size'];
                $aFileSizeUnity = array('o', 'Ko', 'Mo', 'Go', 'To');
                $i = 0;
                while ($iFileSizeLabel > 1024) {
                    $iFileSizeLabel /= 1024;
                    $i++;
                }
                // Sauve la structure du fichier pour le treeview.
                $sFileLabel = $sName . ' (' . round($iFileSizeLabel) . ' ' . $aFileSizeUnity[$i] . ')';
                $aFilteredDir['data']['nodes'][] = array(
                    'text' => utf8_encode($sFileLabel),
                    'icon' => 'glyphicon glyphicon-file',
                    'path' => utf8_encode($sPath),
                    'size' => $aFileInfos['size'],
                    'tags' => array(date('d/m/Y H:i', $aFileInfos['mtime']))
                );
                // Sauve le nom et la date du fichier.
                $aFilteredDir['files'][] = array(
                    'path' => utf8_encode($sPath),
                    'last_modif' => floor((time() - $aFileInfos['mtime']) / (24 * 3600))    // dernière modif. (Nb jours). 
                );
            }
        }
        return $aFilteredDir;
    }

    /**
     * Get workspace file.
     */
    function getWorkspaceFile() {
        $aReturn = array('status' => 1, 'message' => '');
        $sFilePath = $this->aProperties['workspace_dir'] . '/' . $this->aValues["my_vitis_id"] . '/fme/' . $this->aValues['file_name'];
        // Le fichier existe ?
        if (file_exists($sFilePath)) {
            header("Content-Type: application/octet-stream");
            header("Content-Transfer-Encoding: Binary");
            header("Content-disposition: attachment; filename=\"" . $this->aValues['file_name'] . "\"");
            header('Content-Length: ' . filesize($sFilePath));
            readfile($sFilePath);
        }
    }

    /**
     *  Check if Widget file exist
     */
    function checkWidgetFile() {
        $aReturn = array('status' => 1);
        unset($this->aPath[3]);
        $oWorkspace = new Workspace($this->aPath, $this->aValues, $this->aProperties, $this->oConnection);
        $oWorkspace->GET();
        $this->aFields['workspace_key'] = $oWorkspace->aFields["key"];
        $sWidgetFilePath = $this->aProperties["vas_home"] . '/ws_data/gtf/widget/' . $this->aFields['workspace_key'] . '.html';
        // Le fichier existe ?
        if (!file_exists($sWidgetFilePath)) {
            $aReturn = array('status' => 0, 'error_code' => 13, 'message' => 'The file ' . $this->aFields['workspace_key'] . '.html' . 'does not exist.');
            WriteToErrorLog("The file '$sWidgetFilePath' does not exist.");
        }
        return $aReturn;
    }

    /**
     * Delete workspace file.
     */
    function deleteWorkspaceFile() {
        $aReturn = array('status' => 1, 'message' => '');
        $sFilePath = $this->aProperties['workspace_dir'] . '/' . $this->aValues["my_vitis_id"] . '/fme/' . $this->aValues['file_name'];
        // Le fichier existe ?
        if (file_exists(utf8_decode($sFilePath))) {
            if (!unlink(utf8_decode($sFilePath)))
                $aReturn = array('status' => 0, 'message' => 'DELETE_FILE_ERROR_PUBLICATION_WORKSPACE_LOAD_PROJECT_DIRECTORY', 'error_code' => null);
        } else
            $aReturn = array('status' => 0, 'message' => 'FILE_NOT_FOUND_PUBLICATION_WORKSPACE_LOAD_PROJECT_DIRECTORY', 'error_code' => null);

        return $aReturn;
    }

    /**
     * Regenerate workspace metadata.
     */
    function regenerateMetadata() {
        require $this->sRessourcesFile;
        $aReturn = array('status' => 1, 'message' => '');
        $aFields = $this->getFields($this->aProperties['schema_gtf'], "workspace", "workspace_id");
        $sFmwFile = $this->aProperties['workspace_dir'] . '/' . $this->aValues["my_vitis_id"] . '/fme/' . $aFields['fmw_file'];
        $oObject = new GtfFmwParser($sFmwFile);
        // Valeurs des colonnes "last_save_date" et "last_save_build" originales.
        $aParams = array();
        $aParams['sSchemaGtf'] = array('value' => $this->aProperties['schema_gtf'], 'type' => 'schema_name');
        $aParams['workspace_id'] = array('value' => $this->aValues['my_vitis_id'], 'type' => 'number');
        $oPDOresult = $this->oConnection->oBd->executeWithParams($aSql['getWorkspaceLastSaveData'], $aParams);
        if (!$this->oConnection->oBd->enErreur()) {
            $aRow = $this->oConnection->oBd->ligneSuivante($oPDOresult);
            $oObject->sSaveDate = gmdate('Y-m-d G:i:s');
            $oObject->sSaveBuild = $aRow['last_save_build'];
        }
        // Mise à jour du projet en base.
        $oObject->updateMetadata($this->oConnection->oBd, $this->aValues["my_vitis_id"]);
        return $aReturn;
    }

    /**
     * Reintegrate workspace metadata.
     */
    function reintegrateMetadata() {
        $aReturn = array('status' => 1, 'message' => '');
        $aFields = $this->getFields($this->aProperties['schema_gtf'], "workspace", "workspace_id");
        $sFmwFile = $this->aProperties['workspace_dir'] . '/' . $this->aValues["my_vitis_id"] . '/fme/' . $aFields['fmw_file'];
        $oObject = new GtfFmwParser($sFmwFile);
        $oObject->getBdMetadata($this->oConnection->oBd, $this->aValues["my_vitis_id"]);
        $oObject->save($sFmwFile);
        return $aReturn;
    }

}

?>