/* global goog, ol, nsVitisComponent, vitisApp */

/**
 * @author: Anthony Borghi, Armand Bahi
 * @Description: Fichier contenant la classe nsVitisComponent.Map
 * Permet d'instancier un composant OpenLayers3
 */

/*********************************************************************************************************
 *  TODO LIST  (100%)
 *********************************************************************************************************/

'use strict';

goog.provide('nsVitisComponent.Map');

goog.require('nsVitisComponent');
goog.require('MapJSON');

goog.require('ol');
goog.require('ol.Map');
goog.require('ol.View');
goog.require('ol.proj');
goog.require('ol.proj.Projection');
goog.require('ol.Collection');

goog.require('ol.layer.Tile');
goog.require('ol.layer.Vector');
goog.require('ol.layer.Image');
goog.require('ol.layer.Group');

goog.require('ol.source.OSM');
goog.require('ol.source.BingMaps');
goog.require('ol.source.Vector');

goog.require('ol.format.WKT');

goog.require('ol.control');
goog.require('ol.control.MousePosition');
goog.require('ol.control.Control');
goog.require('ol.control.ScaleLine');

goog.require('ol.style.Style');
goog.require('ol.style.Fill');
goog.require('ol.style.Stroke');
goog.require('ol.style.Circle');

goog.require('ol.easing');

goog.require('ol.interaction.Interaction');
goog.require('ol.interaction.Snap');
goog.require('ol.interaction.Select');
goog.require('ol.interaction.Modify');
goog.require('ol.interaction.Draw');

goog.require('ol.events.condition');

goog.require('ol.Sphere');


var featureID = 0;

/**
 * Create a Componant Map on an element with options
 * @param {Object} opt_option
 * @param {Object} opt_option.target HTML Element where the componant will be
 * @param {Object} opt_option.options options to configure the componant
 * @param {string} opt_option.hiddenFieldId id of output's geometries field
 * @param {string} opt_option.oFormValues value of the output
 * @param {string} opt_option.sFormName
 * @constructor
 * @export
 */
nsVitisComponent.Map = function (opt_option) {

    var target = opt_option['target'];
    var options = opt_option['options'];
    this.hiddenFieldId = opt_option['hiddenFieldId'];
    this.oFormValues = opt_option['oFormValues'];
    this.sFormName = opt_option['sFormName'];
    var oProj = opt_option['oProj'];

    if (!goog.isDefAndNotNull(target)) {
        console.error('nsVitisComponent.Map: target not defined');
        return null;
    }
    if (!goog.isDefAndNotNull(options)) {
        console.error('nsVitisComponent.Map: options not defined');
        return null;
    }
    if (!goog.isDefAndNotNull(this.hiddenFieldId)) {
        console.error('nsVitisComponent.Map: hiddenFieldId not defined');
        return null;
    }
    if (!goog.isDefAndNotNull(this.oFormValues)) {
        console.error('nsVitisComponent.Map: oFormValues not defined');
        return null;
    }
    if (!goog.isDefAndNotNull(this.sFormName)) {
        console.error('nsVitisComponent.Map: sFormName not defined');
        return null;
    }
    if (!goog.isDefAndNotNull(oProj)) {
        console.error('nsVitisComponent.Map: oProj not defined');
        return null;
    }

    /**
     * @private
     */
    this.$propertiesSrvc_ = angular.element(vitisApp.appHtmlFormDrtv).injector().get(["propertiesSrvc"]);
    /**
     * @private
     */
    this.$translate_ = angular.element(vitisApp.appHtmlFormDrtv).injector().get(["$translate"]);
    /**
     * @private
     */
    this.$log_ = angular.element(vitisApp.appHtmlFormDrtv).injector().get(["$log"]);

    var this_ = this;

    this.$log_.info('nsVitisComponent.Map');

    this.appColor = $(".banner_line").css("background-color");
    if (goog.isDef(this.appColor)) {
        if (this.appColor.charAt(3) !== 'a') {
            this.appColor = this.appColor.split('(');
            this.appColor = this.appColor[1].split(')');
            this.appColor = "rgba(" + this.appColor[0] + ",0.58)";
        }
    }

    /********************** Translates **************************/

    var aTranslates = [
        'COMPONENT_MAP_LAYERSTREE',
        'COMPONENT_MAP_FULLSCREEN',
        'COMPONENT_MAP_ZOOM_EXTENT',
        'COMPONENT_MAP_DELETE_FEATURE',
        'COMPONENT_MAP_DELETE_ALL',
        'COMPONENT_MAP_MODIFY_FEATURE',
        'COMPONENT_MAP_DRAW_POINT',
        'COMPONENT_MAP_DRAW_LINE',
        'COMPONENT_MAP_DRAW_POLYGON',
        'COMPONENT_MAP_ZOOM_EXTENT',
        'COMPONENT_MAP_DIVIDE_SEGMENT',
        'COMPONENT_MAP_DIVIDE_SEGMENT_POINT_NOT_VALID',
        'COMPONENT_MAP_OVERSIZED_EXTENT'
    ];

    this.$translate_(aTranslates).then(function (translations) {
        this_.oTranslates = translations;
    });

    /*********************View**********************/
    this.addCustomProjections();

    var center, extent;

    if (goog.isDef(options["center"]["coord"]))
        center = options["center"]["coord"];
    if (goog.isDef(options["center"]["extent"]))
        extent = options["center"]["extent"];

    /**
     * Projection de la carte
     */
    this.proj_ = options["proj"];

    /**
     * Projection de la base de données
     */
    this.baseProj_ = goog.isDefAndNotNull(options["base_proj"]) ? options["base_proj"] : this.proj_;

    /**
     * Format de sortie (wkt/geojson)
     */
    this.outputFormat_ = goog.isDefAndNotNull(options["output_format"]) ? options["output_format"] : 'wkt';

    /**
     * WKT Format
     */
    this.wktFormat = new ol.format.WKT({
        splitCollection: true
    });

    /**
     * GeoJSON format
     */
    this.geoJSONFormat = new ol.format.GeoJSON({
        defaultDataProjection: this.baseProj_,
        featureProjection: this.proj_
    });

    if (options["center"]["coord"]) {
        this.View = new ol.View({
            center: center,
            zoom: 5,
            projection: ol.proj.get(options["proj"])
        });
    } else {
        this.View = new ol.View({
            center: [((extent[0] + extent[2]) / 2), ((extent[1] + extent[3]) / 2)],
            zoom: 5,
            projection: ol.proj.get(options["proj"])
        });
    }


    /****************Initialisation***********************/
    /**
     * contains all the used translates
     * @type object
     */
    this.oTranslates = {};
    /**
     * @type {Object}
     * @private
     */
    this.Target = target;
    /**
     * @type {number}
     * @private
     */
    this.Accuracy = options["coord_accuracy"];
    /**
     * @type {Array.<ol.control.Control>}
     * @private
     */
    this.Controls = [];
    /**
     * @type {Array.<ol.layer.Tile>}
     * @private
     */
    this.Layers = [];
    /**
     * @type {ol.Collection.<ol.Feature>}
     * @private
     */
    this.Features = new ol.Collection();
    /**
     * @type {ol.Collection.<ol.Feature>}
     * @private
     */
    this.DisplayFeatures = new ol.Collection();
    /**
     * @type {ol.interaction.Interaction}
     * @private
     */
    this.interaction = "none";
    /**
     * @type {Object}
     * @private
     */
    this.Config = {};
    /**
     * @type {Object}
     * @private
     */
    this.oProj_ = oProj;
    /**
     * @type {boolean}
     * @private
     */
    this.multiGeom = options["interactions"]["multi_geometry"];
    /**
     * The selected feature
     */
    this.selectedFeature = null;
    /**
     * Boolean saying if the layers tree is open
     */
    this['bLayersTreeOpen'] = options['bLayersTreeOpen'] === true ? options['bLayersTreeOpen'] : false;
    /**
     * Events called when selectedFeature changed
     */
    this.featureSelectedEvents = [];
    /**
     * Events called when Features changed
     */
    this.featuresChangedEvents = [];
    /**
     * Events called when a feature is added
     */
    this.featureAddedEvents = [];
    /**
     * Events called when a feature is added
     */
    this.featureAddedOnceEvents = [];

    /*********************Layers**********************/
    if (options["type"] === "OSM") {
        this.Layers.push(new ol.layer.Tile({
            source: new ol.source.OSM()
        }));
    } else if (options["type"] === "bing") {
        this.Layers.push(new ol.layer.Tile({
            source: new ol.source.BingMaps({
                culture: options["source"]["culture"],
                key: options["source"]["key"],
                imagerySet: options["source"]["style"]
            })
        }));
    }

    /*******************Current Projection Element***************************/
    this.element = document.createElement("DIV");
    $(this.element).addClass("ol-current-projection-studio ol-unselectable");

    /*******************Controls************************************/
    if (options["controls"]["MP"]) {
        this.Controls.push(new ol.control.MousePosition({
            coordinateFormat: function (coordinate) {
                //coordinate = ol.proj.transform(coordinate, "EPSG:3857", this_.proj_);
                return [coordinate[0].toPrecision(this_.Accuracy), coordinate[1].toPrecision(this_.Accuracy)];
            }
        }));
    }
    if (options["controls"]["SL"]) {
        this.Controls.push(new ol.control.ScaleLine());
    }
    if (options["controls"]["CP"]) {
        this.Controls.push(new ol.control.Control({
            element: this_.element,
            render: function () {
                var currentProj = "";
                for (var i = 0; i < this_.oProj_["projections"].length; i++) {
                    if (this_.proj_ === this_.oProj_["projections"][i]["code"]) {
                        currentProj = this_.oProj_["projections"][i]["name"];
                    }
                }
                if (currentProj !== '') {
                    $(".ol-current-projection-studio").html(currentProj);
                } else {
                    $(".ol-current-projection-studio").html(this_.proj_);
                }
            }
        }));
    }
    if (options["controls"]["ZO"]) {
        // Timeout pour attendre que les traductions soient faites
        setTimeout(function () {
            this_.ZoomGroup = this_.zoomControls(this_.Target);
        }, 500);
    }

    setTimeout(function () {
        this_.IntractionsGroup = this_.addInteractions(this_.Target, options["interactions"]);
    });

    if (Array.isArray(options["layers"])) {
        if (options["layers"].length > 0) {
            this.Layers.push(options["layers"]);
        }
    }

    /****************************************************Map*******************************/

    this.MapObject = new ol.Map({
        target: this_.Target,
        layers: this_.Layers,
        view: this_.View,
        controls: this_.Controls
    });

    if (goog.isDefAndNotNull(options["center"]["scale"])) {
        this.setScale(options["center"]["scale"]);
    }

    if (goog.isDefAndNotNull(options["center"]["extent"])) {
        this.setExtent(options["center"]["extent"]);
    }

    if (options["type"] === "vmap" && options["tree"]) {
        this['aTree'] = this.loadTree(options["tree"]);
    } else {
        this['aTree'] = [{
                'service': options["type"],
                'layers': this.Layers
            }];
    }

    /****************************************Extent***************************************/

    // Vérifie que l'étendue soit valable
    var projExtent = ol.proj.get(options["proj"]).getExtent();
    var viewExtent = this.View.calculateExtent(this.MapObject.getSize());

    if (this.isInsideProjExtent(viewExtent, projExtent)) {
        this.Extent = this.View.calculateExtent(this.MapObject.getSize());
    } else {
        console.error('map extent oversized');
        this.MapObject.getView().fit(projExtent);
    }


    //****************************************CSS******************************************/
    $(".ol-mouse-position").css("bottom", "8px");
    $(".ol-mouse-position").css("top", "auto");
    $(".ol-mouse-position").css("background", this.appColor);
    $(".ol-mouse-position").css("color", "#ffffff");
    $(".ol-mouse-position").css("border-radius", "4px");

    $(".ol-scale-line").css("background", this.appColor);

    $(".ol-current-projection-studio").css("background", this.appColor);

    /****************************************Feature Overlay********************************/

    options["draw_color"] = goog.isDef(options["draw_color"]) ? options["draw_color"] : "rgba(54,184,255,0.6)";
    options["contour_color"] = goog.isDef(options["contour_color"]) ? options["contour_color"] : "rgba(0,0,0,0.4)";
    options["contour_size"] = goog.isDef(options["contour_size"]) ? options["contour_size"] : 2;
    options["circle_radius"] = goog.isDef(options["circle_radius"]) ? options["circle_radius"] : 7;

    var style = new ol.style.Style({
        fill: new ol.style.Fill({
            color: options["draw_color"]
        }),
        stroke: new ol.style.Stroke({
            color: options["contour_color"],
            width: options["contour_size"]
        }),
        image: new ol.style.Circle({
            radius: options["circle_radius"],
            fill: new ol.style.Fill({
                color: options["draw_color"]
            }),
            stroke: new ol.style.Stroke({
                color: options["contour_color"],
                width: options["contour_size"]
            })
        })
    });

    // Couche vecteur contenant les géométries à éditer
    this.FeatureOverlay = new ol.layer.Vector({
        style: style,
        updateWhileAnimating: true,
        updateWhileInteracting: true,
        source: new ol.source.Vector({
            features: this_.Features
        })
    });

    // Couche vecteur contenant les ǵeométries uniquement visuelles
    // à ne pas inclure dans le formulaire
    this.FeatureDisplayOverlay = new ol.layer.Vector({
        style: style,
        updateWhileAnimating: true,
        updateWhileInteracting: true,
        source: new ol.source.Vector({
            features: this_.DisplayFeatures
        })
    });
    setTimeout(function () {
        this_.MapObject.addLayer(this_.FeatureOverlay);
        this_.MapObject.addLayer(this_.FeatureDisplayOverlay);
    }, 500);

    // Snaping
    // Doit être ajouté après les interractions de dessin
    this.snapInteraction_ = new ol.interaction.Snap({
        source: this_.FeatureOverlay.getSource()
    })

    var hiddenFeatures = $("#" + this.hiddenFieldId).val();

    // Lit les features écrites dans le champ hidden
    if (goog.isDef(hiddenFeatures)) {
        if (!goog.string.isEmpty(hiddenFeatures) && hiddenFeatures !== "[object Object]") {

            var aFeatures = this_.getFeaturesByString(hiddenFeatures);

            // Ajoute les features
            for (var i = 0; i < aFeatures.length; i++) {
                this.FeatureOverlay.getSource().addFeature(aFeatures[i]);
            }
            var featuresExtent = this.FeatureOverlay.getSource().getExtent();
            this.MapObject.getView().fit(featuresExtent);

            // En cas de simple point
            if (aFeatures.length === 1) {
                if (aFeatures[0].getGeometry().getType() === 'Point') {
                    this.MapObject.getView().setZoom(12);
                }
            }
        }
    }

    // Écrit les features dans le champ hidden
    this.Features.on("change", function () {
        setTimeout(function () {
            this_.saveFeatures();
            this_.MapObject.dispatchEvent('moveend');
        });
    });

    return this;
};

/**
 * Save the current features
 * @returns {undefined}
 */
nsVitisComponent.Map.prototype.saveFeatures = function () {
    this.$log_.info('nsVitisComponent.Map.saveFeatures');

    var sFeatures = this.getStringByFeatures(this.Features.getArray());

    if (goog.isDef(this.hiddenFieldId)) {
        document.getElementById(this.hiddenFieldId).setAttribute('value', sFeatures);
    }
    if (goog.isDef(this.sFormName) && goog.isDef(this.oFormValues)) {
        this.oFormValues[this.sFormName] = sFeatures;
    }

    // Lance les événements featuresChangedEvents
    for (var i = 0; i < this.featuresChangedEvents.length; i++) {
        if (goog.isFunction(this.featuresChangedEvents[i])) {
            this.featuresChangedEvents[i].call(this, this.Features.getArray());
        }
    }
};

/**
 * Parse the string (GeoJSON, WKT) and return the features
 * @param {string} sFeatures
 * @returns {Array}
 */
nsVitisComponent.Map.prototype.getFeaturesByString = function (sFeatures) {
    this.$log_.info('nsVitisComponent.Map.getFeaturesByString');

    var aFeatures = [];

    if (this.outputFormat_ === 'wkt' || this.outputFormat_ === 'ewkt') {
        if (ol.isEWKTGeom(sFeatures)) {
            aFeatures = ol.getFeaturesFromEWKT(sFeatures, this.proj_);
        } else {
            console.error('cannot read the geometry on format ewkt');
            try {
                aFeatures = this.wktFormat.readFeatures(sFeatures, {
                    dataProjection: this.baseProj_,
                    featureProjection: this.proj_
                });
            } catch (e) {
                console.error('cannot read the geometry on format wkt');
            }
        }
    } else if (this.outputFormat_ === 'geojson') {
        try {
            aFeatures = this.geoJSONFormat.readFeatures(sFeatures, {
                dataProjection: this.baseProj_,
                featureProjection: this.proj_
            });
            for (var i = 0; i < aFeatures.length; i++) {
                if (goog.isDefAndNotNull(aFeatures[i].get('style'))) {
                    aFeatures[i].setStyleByJSON(aFeatures[i].get('style'));
                }
            }
        } catch (e) {
            console.error('cannot read the geometry on format geojson');
        }
    }
    return aFeatures;
};

/**
 * Read the features passed and return a string
 * @param {array<object>} aFeatures
 * @returns {string}
 */
nsVitisComponent.Map.prototype.getStringByFeatures = function (aFeatures) {
    this.$log_.info('nsVitisComponent.Map.getStringByFeatures');

    var sFeatures = '';

    // Mémorise le style
    for (var i = 0; i < aFeatures.length; i++) {
        var olStyle = aFeatures[i].getStyleAsJSON();
        aFeatures[i].setProperties({
            style: goog.isDefAndNotNull(olStyle) ? olStyle : null
        });
    }

    if (this.outputFormat_ === 'ewkt') {
        sFeatures = ol.getEWKTFromFeatures(aFeatures, this.proj_);
    } else if (this.outputFormat_ === 'wkt') {
        sFeatures = ol.getWKTFromFeatures(aFeatures, this.proj_);
    } else if (this.outputFormat_ === 'geojson') {
        sFeatures = this.geoJSONFormat.writeFeatures(aFeatures, {
            dataProjection: this.baseProj_,
            featureProjection: this.proj_
        });
    }

    return sFeatures;
};

/**
 * Add projections EPSG:2154
 */
nsVitisComponent.Map.prototype.addCustomProjections = function () {

    var lambert93 = new ol.proj.Projection({
        code: 'EPSG:2154',
        // Cette extent est celle du niveau de zoom 0, on peut les trouver sur http://epsg.io/
        extent: [-378305.81, 6093283.21, 1212610.74, 7186901.68],
        units: 'm'
    });
    ol.proj.addProjection(lambert93);
    var WGSToLambert = this.WGSToLambert;
    var lambertToWGS = this.lambertToWGS;
    // Ajoute les fonctions de transformation de EPSG:4326 vers EPSG:2154
    ol.proj.addCoordinateTransforms('EPSG:4326', 'EPSG:2154', this.WGSToLambert, this.lambertToWGS);
    ol.proj.addCoordinateTransforms('EPSG:3857', 'EPSG:2154',
            function (coordinate3857) {
                var coordinate4326 = ol.proj.transform(coordinate3857, 'EPSG:3857', 'EPSG:4326');
                var coordinate2154 = ol.proj.transform(coordinate4326, 'EPSG:4326', 'EPSG:2154');
                return coordinate2154;
            },
            function (coordinate2154) {
                var coordinate4326 = ol.proj.transform(coordinate2154, 'EPSG:2154', 'EPSG:4326');
                var coordinate3857 = ol.proj.transform(coordinate4326, 'EPSG:4326', 'EPSG:3857');
                return coordinate3857;
            });
};

/**
 * Tansform WGS83 coordinates to Lambert93
 * @param {ol.coordinates} coordinates WGS coordinates to transform
 * @returns {ol.coordinates} Lambert93 coordinates
 */
nsVitisComponent.Map.prototype.WGSToLambert = function (coordinates) {

    var longitude = coordinates[0];
    var latitude = coordinates[1];
    var deg2rad = function deg2rad(angle) {
        return (angle / 180) * Math.PI;
    };
    //variables:
    var a = 6378137; //demi grand axe de l'ellipsoide (m)
    var e = 0.08181919106; //première excentricité de l'ellipsoide
    var lc = deg2rad(3);
    var phi0 = deg2rad(46.5); //latitude d'origine en radian
    var phi1 = deg2rad(44); //1er parallele automécoïque
    var phi2 = deg2rad(49); //2eme parallele automécoïque

    var x0 = 700000; //coordonnées à l'origine
    var y0 = 6600000; //coordonnées à l'origine

    var phi = deg2rad(latitude);
    var l = deg2rad(longitude);
    //calcul des grandes normales
    var gN1 = a / Math.sqrt(1 - e * e * Math.sin(phi1) * Math.sin(phi1));
    var gN2 = a / Math.sqrt(1 - e * e * Math.sin(phi2) * Math.sin(phi2));
    //calculs des latitudes isométriques
    var gl1 = Math.log(Math.tan(Math.PI / 4 + phi1 / 2) * Math.pow((1 - e * Math.sin(phi1)) / (1 + e * Math.sin(phi1)), e / 2));
    var gl2 = Math.log(Math.tan(Math.PI / 4 + phi2 / 2) * Math.pow((1 - e * Math.sin(phi2)) / (1 + e * Math.sin(phi2)), e / 2));
    var gl0 = Math.log(Math.tan(Math.PI / 4 + phi0 / 2) * Math.pow((1 - e * Math.sin(phi0)) / (1 + e * Math.sin(phi0)), e / 2));
    var gl = Math.log(Math.tan(Math.PI / 4 + phi / 2) * Math.pow((1 - e * Math.sin(phi)) / (1 + e * Math.sin(phi)), e / 2));
    //calcul de l'exposant de la projection
    var n = (Math.log((gN2 * Math.cos(phi2)) / (gN1 * Math.cos(phi1)))) / (gl1 - gl2);
    //calcul de la constante de projection
    var c = ((gN1 * Math.cos(phi1)) / n) * Math.exp(n * gl1);
    //calcul des coordonnées
    var ys = y0 + c * Math.exp(-1 * n * gl0);
    var x93 = x0 + c * Math.exp(-1 * n * gl) * Math.sin(n * (l - lc));
    var y93 = ys - c * Math.exp(-1 * n * gl) * Math.cos(n * (l - lc));
    return [x93, y93];
};

/**
 * Tansform Lambert93 coordinates to WGS83
 * @param {ol.coordinates} coordinates Lambert93 coordinates to transform
 * @returns {ol.coordinates} WGS84 coordinates
 */
nsVitisComponent.Map.prototype.lambertToWGS = function (coordinates) {

    Math.atanh = Math.atanh || function (x) {
        return Math.log((1 + x) / (1 - x)) / 2;
    };

    Math.tanh = Math.tanh || function (x) {
        if (x === Infinity) {
            return 1;
        } else if (x === -Infinity) {
            return -1;
        } else {
            return (Math.exp(x) - Math.exp(-x)) / (Math.exp(x) + Math.exp(-x));
        }
    };

    var x = coordinates[0].toFixed(10);
    var y = coordinates[1].toFixed(10);
    var b6 = 6378137.0000;
    var b7 = 298.257222101;
    var b8 = 1 / b7;
    var b9 = 2 * b8 - b8 * b8;
    var b10 = Math.sqrt(b9);
    var b13 = 3.000000000;
    var b14 = 700000.0000;
    var b15 = 12655612.0499;
    var b16 = 0.7256077650532670;
    var b17 = 11754255.426096;
    var delx = x - b14;
    var dely = y - b15;
    var gamma = Math.atan(-(delx) / dely);
    var r = Math.sqrt((delx * delx) + (dely * dely));
    var latiso = Math.log(b17 / r) / b16;
    var sinphiit0 = Math.tanh(latiso + b10 * Math.atanh(b10 * Math.sin(1)));
    var sinphiit1 = Math.tanh(latiso + b10 * Math.atanh(b10 * sinphiit0));
    var sinphiit2 = Math.tanh(latiso + b10 * Math.atanh(b10 * sinphiit1));
    var sinphiit3 = Math.tanh(latiso + b10 * Math.atanh(b10 * sinphiit2));
    var sinphiit4 = Math.tanh(latiso + b10 * Math.atanh(b10 * sinphiit3));
    var sinphiit5 = Math.tanh(latiso + b10 * Math.atanh(b10 * sinphiit4));
    var sinphiit6 = Math.tanh(latiso + b10 * Math.atanh(b10 * sinphiit5));
    var longrad = gamma / b16 + b13 / 180 * Math.PI;
    var latrad = Math.asin(sinphiit6);
    var lon = (longrad / Math.PI * 180);
    var lat = (latrad / Math.PI * 180);
    return ([lon, lat]);
};

/**
 * add Custom zoom Control
 * @param {Object} target target Element HTML where button will be
 * @returns {Object} Element who contain this control
 */
nsVitisComponent.Map.prototype.zoomControls = function (target) {

    var this_ = this;

    var buttonPlus = document.createElement("BUTTON");
    var buttonMinus = document.createElement("BUTTON");
    var buttonExtent = document.createElement("BUTTON");

    buttonPlus['innerHTML'] = "<span class='fa fa-search-plus fa-x1' aria-hidden='true'></span>";
    buttonMinus['innerHTML'] = "<span class='fa fa-search-minus fa-x1' aria-hidden='true'></span>";
    buttonExtent['innerHTML'] = "<span class='glyphicon glyphicon-home' aria-hidden='true'></span>";

    buttonPlus["className"] = "btn btn-success Map-btn map-ZoomIn";
    buttonMinus["className"] = "btn btn-success Map-btn map-ZoomOut";
    buttonExtent["className"] = "btn btn-success Map-btn map-ZoomToExtent";

    $(buttonPlus).prop("type", "button");
    $(buttonMinus).prop("type", "button");
    $(buttonExtent).prop("type", "button");

    $(buttonExtent).prop("title", this.oTranslates['COMPONENT_MAP_ZOOM_EXTENT']);

    $(buttonPlus).on("click", function () {
        this_.MapObject.getView().animate({
            zoom: this_.MapObject.getView().getZoom() + 1,
            duration: 250
        });
    });
    $(buttonMinus).on("click", function () {
        this_.MapObject.getView().animate({
            zoom: this_.MapObject.getView().getZoom() - 1,
            duration: 250
        });
    });
    $(buttonExtent).on("click", function () {
        this_.MapObject.getView().fit(this_.Extent);
    });

    var element = document.createElement("DIV");
    element["className"] = "btn-group-vertical btn-group-md form-map-Group-btn-zoom";

    element.appendChild(buttonPlus);
    element.appendChild(buttonMinus);
    element.appendChild(buttonExtent);

    target.appendChild(element);

    return element;
};

/**
 * Add all Interactions of the map
 * @param {Object} target target Element HTML where button will be
 * @param {Object} objInter who specify if an interaction is present or not
 * @returns {Object} Return the Element who contains all interaction's buttons
 */
nsVitisComponent.Map.prototype.addInteractions = function (target, objInter) {
    var arrButton = [];
    var this_ = this;

    /**
     * Full Screen
     */
    if (objInter["full_screen"]) {

        // Si il est possible d'afficher en fullScreen
        if (
                document['fullscreenEnabled'] ||
                document['webkitFullscreenEnabled'] ||
                document['mozFullScreenEnabled'] ||
                document['msFullscreenEnabled']
                ) {

            var button_full_screen = document.createElement("BUTTON");
            button_full_screen['innerHTML'] = "<span class='icon-enlarge' aria-hidden='true'></span>";
            button_full_screen["className"] = "btn btn-success Map-btn map-FullScreen";
            button_full_screen["title"] = this.oTranslates['COMPONENT_MAP_FULLSCREEN'];
            $(button_full_screen).prop("type", "button");
            $(button_full_screen).on("click", function () {

                var elem = this_.Target;

                // Si on est déjà en mode plein écran
                if (
                        document.fullscreenElement ||
                        document.webkitFullscreenElement ||
                        document.mozFullScreenElement ||
                        document.msFullscreenElement
                        ) {
                    if (document.exitFullscreen) {
                        document.exitFullscreen();
                    } else if (document.webkitExitFullscreen) {
                        document.webkitExitFullscreen();
                    } else if (document.mozCancelFullScreen) {
                        document.mozCancelFullScreen();
                    } else if (document.msExitFullscreen) {
                        document.msExitFullscreen();
                    }
                } else {
                    if (elem.requestFullscreen) {
                        elem.requestFullscreen();
                    } else if (elem.mozRequestFullScreen) {
                        elem.mozRequestFullScreen();
                    } else if (elem.webkitRequestFullscreen) {
                        elem.webkitRequestFullscreen();
                    }
                }

            });
            arrButton.push(button_full_screen);
        }
    }
    /**
     * Display the layers tree
     */
    if (objInter["layer_tree"]) {

        var button_layer_tree = document.createElement("BUTTON");
        button_layer_tree['innerHTML'] = "<span class='icon-layers2' aria-hidden='true'></span>";
        button_layer_tree["className"] = "btn btn-success Map-btn map-LayerTree";
        button_layer_tree["title"] = this.oTranslates['COMPONENT_MAP_LAYERSTREE'];
        $(button_layer_tree).prop("type", "button");
        $(button_layer_tree).on("click", function () {

            this_['bLayersTreeOpen'] = !this_['bLayersTreeOpen'];
            this_.MapObject.dispatchEvent('moveend');

        });
        arrButton.push(button_layer_tree);
    }
    /**
     * Select feature
     */
    if (objInter["select"]) {
        var button_select = document.createElement("BUTTON");
        button_select['innerHTML'] = "<span class='icon-hand-pointer' aria-hidden='true'></span>";
        button_select["className"] = "btn btn-success Map-btn map-Select";
        button_select["title"] = this.oTranslates['COMPONENT_MAP_SELECT'];
        $(button_select).prop("type", "button");
        $(button_select).on("click", function () {
            this_.activeInteraction('select');
        });
        arrButton.push(button_select);
    }
    /**
     * Remode All Features
     */
    if (objInter["RA"] || objInter["remove_all"]) {
        var button_ra = document.createElement("BUTTON");
        button_ra['innerHTML'] = "<span class='icon-trash' aria-hidden='true'></span>";
        button_ra["className"] = "btn btn-success Map-btn map-AllRemove";
        button_ra["title"] = this.oTranslates['COMPONENT_MAP_DELETE_ALL'];
        $(button_ra).prop("type", "button");
        $(button_ra).on("click", function () {
            if (this_.interaction !== "none") {
                this_.MapObject.removeInteraction(this_.interaction);
            }
            this_.FeatureOverlay.getSource().getFeaturesCollection().clear();
            this_.FeatureOverlay.getSource().changed();
            this_.Features.changed();
        });
        arrButton.push(button_ra);
    }
    /**
     * Remove Feature
     */
    if (objInter["RO"] || objInter["remove"]) {
        var button_ro = document.createElement("BUTTON");
        button_ro['innerHTML'] = "<span class='icon-eraser' aria-hidden='true'></span>";
        button_ro["className"] = "btn btn-success Map-btn map-Remove";
        button_ro["title"] = this.oTranslates['COMPONENT_MAP_DELETE_FEATURE'];
        $(button_ro).prop("type", "button");
        $(button_ro).on("click", function () {
            this_.activeInteraction('remove');
        });
        arrButton.push(button_ro);
    }
    /**
     * Modify Feature
     */
    if (objInter["ED"] || objInter["modify"]) {
        var button_ed = document.createElement("BUTTON");
        button_ed['innerHTML'] = "<span class='icon-edit' aria-hidden='true'></span>";
        button_ed["className"] = "btn btn-success Map-btn map-Modify";
        button_ed["title"] = this.oTranslates['COMPONENT_MAP_MODIFY_FEATURE'];
        $(button_ed).prop("type", "button");
        $(button_ed).on("click", function () {
            this_.activeInteraction('modify');
        });
        arrButton.push(button_ed);
    }
    /**
     * Draw Point
     */
    if (objInter["DP"] || objInter["point"]) {
        var button_dp = document.createElement("BUTTON");
        button_dp['innerHTML'] = "<span class='icon-point' aria-hidden='true'></span>";
        button_dp["className"] = "btn btn-success Map-btn map-Point";
        button_dp["title"] = this.oTranslates['COMPONENT_MAP_DRAW_POINT'];
        $(button_dp).prop("type", "button");
        $(button_dp).on("click", function () {
            this_.activeInteraction('point');
        });

        arrButton.push(button_dp);
    }
    /**
     * Draw Line
     */
    if (objInter["DL"] || objInter["line"]) {
        var button_dl = document.createElement("BUTTON");
        button_dl['innerHTML'] = "<span class='icon-line' aria-hidden='true'></span>";
        button_dl["className"] = "btn btn-success Map-btn map-Line";
        button_dl["title"] = this.oTranslates['COMPONENT_MAP_DRAW_LINE'];
        $(button_dl).prop("type", "button");
        $(button_dl).on("click", function () {
            this_.activeInteraction('line');
        });

        arrButton.push(button_dl);
    }
    /**
     * Draw Polygon
     */
    if (objInter["DPol"] || objInter["polygon"]) {
        var button_dpol = document.createElement("BUTTON");
        button_dpol['innerHTML'] = "<span class='icon-polygon' aria-hidden='true'></span>";
        button_dpol["className"] = "btn btn-success Map-btn map-Polygon";
        button_dpol["title"] = this.oTranslates['COMPONENT_MAP_DRAW_POLYGON'];
        $(button_dpol).prop("type", "button");
        $(button_dpol).on("click", function () {
            this_.activeInteraction('polygon');
        });
        arrButton.push(button_dpol);
    }
    /**
     * Divide segment
     */
    if (objInter["divide_segment"]) {
        if(goog.isArray(this.Features.getArray())){
            if(this.Features.getArray().length == 1){
                if (goog.isDefAndNotNull(this.Features.getArray()[0].getGeometry())) {
                    if (goog.isDefAndNotNull(this.Features.getArray()[0].getGeometry().getType)) {
                        if (this.Features.getArray()[0].getGeometry().getType() === "LineString") {
                            var button_dpol = document.createElement("BUTTON");
                            button_dpol['innerHTML'] = "<span class='icon-cut_line' aria-hidden='true'></span>";
                            button_dpol["className"] = "btn btn-success Map-btn map-divide_segment";
                            button_dpol["title"] = this.oTranslates['COMPONENT_MAP_DIVIDE_SEGMENT'];
                            $(button_dpol).prop("type", "button");
                            $(button_dpol).on("click", function () {
                                this_.activeInteraction('divide_segment');
                            });
                            arrButton.push(button_dpol);
                        }
                    }
                }
            }
        }
    }

    var element = document.createElement("DIV");
    element["className"] = "btn-group-vertical btn-group-md form-map-Group-btn-interact";

    for (var i = 0; i < arrButton.length; i++) {
        element.appendChild(arrButton[i]);
    }

    target.appendChild(element);

    // actions sur les interactions
    setTimeout(function () {
        var interactions = this_.MapObject.getInteractions();

        // active la bouton lorsque l'interaction est en cours d'utilisation
        interactions.on('change:length', function () {
            // enlève active et focus de tous les éléments
            $(element).children().removeClass('active');
            $(element).children().blur();
            // rajoute active aux éléments actifs
            interactions.forEach(function (element, index, array) {
                $(element.get('binded-button')).addClass('active');
            });
        });

        // Lorsqu'une interraction est supprimée
        interactions.on('remove', function () {
            if (goog.isDefAndNotNull(this_.selectHover_)) {
                this_.selectHover_.setActive(false);
            }
            this_.selectedFeature = null;
            for (var i = 0; i < this_.featureSelectedEvents.length; i++) {
                if (goog.isFunction(this_.featureSelectedEvents[i])) {
                    this_.featureSelectedEvents[i].call(this, this_.selectedFeature);
                }
            }
        });

        // désactive les interactions quand on appuie sur echap
        $(document).keydown(function (e) {
            if (e.keyCode === 27) {
                interactions.forEach(function (element, index, array) {
                    if (goog.isDef(element.get('binded-button'))) {
                        this_.MapObject.removeInteraction(element);
                    }
                });
            }
        });
    }, 1000);

    return element;
};

/**
 * Remove the passed interaction
 * @param {string} sInteraction
 * @export
 */
nsVitisComponent.Map.prototype.removeInteraction = function (sInteraction) {

    var button = null;
    switch (sInteraction) {
        case 'full_screen':
            button = $(this.Target).find('.map-FullScreen');
            break;
        case 'layer_tree':
            button = $(this.Target).find('.map-LayerTree');
            break;
        case 'select':
            button = $(this.Target).find('.map-Select');
            break;
        case 'remove_all':
        case 'RA':
            button = $(this.Target).find('.map-AllRemove');
            break;
        case 'remove':
        case 'RO':
            button = $(this.Target).find('.map-Remove');
            break;
        case 'modify':
        case 'ED':
            button = $(this.Target).find('.map-Modify');
            break;
        case 'point':
        case 'DP':
            button = $(this.Target).find('.map-Point');
            break;
        case 'line':
        case 'DL':
            button = $(this.Target).find('.map-Line');
            break;
        case 'polygon':
        case 'DPol':
            button = $(this.Target).find('.map-Polygon');
            break;
        default:
            break;
    }
    if (goog.isDefAndNotNull(button)) {
        $(button).remove();
    }
};

/**
 * Active the selectHover_ interaction
 */
nsVitisComponent.Map.prototype.activeSelectHover = function () {
    var this_ = this;
    if (goog.isDefAndNotNull(this.selectHover_)) {
        this.MapObject.removeInteraction(this.selectHover_);
    }
    this.selectHover_ = new ol.interaction.Select({
        source: this.FeatureOverlay.getSource(),
        condition: ol.events.condition.pointerMove
    });

    this.selectHover_.on("select", function (oEvent) {
        // Passe le curseur en mode pointeur
        this_.Target.style.cursor = '';
        if (goog.isArray(oEvent.selected)) {
            if (oEvent.selected.length > 0) {
                this_.Target.style.cursor = 'pointer';
            }
        }
    });
    this.MapObject.addInteraction(this.selectHover_);
    this.selectHover_.setActive(true);
};

/**
 * Active the interaction
 * @param {string} sInteraction point, line, polygon, select, remove, modify
 * @param {boolean} bForceActive if false and the botton is already active, unset the interaction
 * @export
 */
nsVitisComponent.Map.prototype.activeInteraction = function (sInteraction, bForceActive) {

    var this_ = this;
    switch (sInteraction) {
        case 'point':
            var button = $(this.Target).find('.map-Point');
            break;
        case 'line':
            var button = $(this.Target).find('.map-Line');
            break;
        case 'polygon':
            var button = $(this.Target).find('.map-Polygon');
            break;
        case 'select':
            var button = $(this.Target).find('.map-Select');
            break;
        case 'remove':
            var button = $(this.Target).find('.map-Remove');
            break;
        case 'modify':
            var button = $(this.Target).find('.map-Modify');
            break;
        case 'divide_segment':
            var button = $(this.Target).find('.map-divide_segment');
            break;
        default:
            break;
    }

    var isActive = $(button).hasClass('active');
    if (bForceActive) {
        isActive = false;
    }
    if (this.interaction !== "none") {
        this.MapObject.removeInteraction(this.interaction);
        // allow to toggle the interaction
        if (isActive)
            return 0;
    }

    // Désactive l'accrochage car il doit être ajouté après les interactions de dessin
    this.MapObject.removeInteraction(this.snapInteraction_);

    switch (sInteraction) {
        case 'point':
            this.interaction = new ol.interaction.Draw({
                source: this.FeatureOverlay.getSource(),
                type: "Point"
            });
            this.interaction.set('binded-button', button);
            this.interaction.set('type', 'draw_point');
            this.MapObject.addInteraction(this.interaction);
            this.interaction.on('drawend', function (event) {
                // Supprime si besoin les anciennes features
                if (this_.multiGeom !== true) {
                    var features = this_.FeatureOverlay.getSource().getFeaturesCollection().clear();
                }
                var featureID = featureID + 1;
                event.feature.d_ = featureID;
                this_.Features.changed();
                this_.callFeatureAddedEvents(event.feature);
            });
            break;
        case 'line':
            this.interaction = new ol.interaction.Draw({
                source: this.FeatureOverlay.getSource(),
                type: "LineString"
            });
            this.interaction.set('binded-button', button);
            this.interaction.set('type', 'draw_line');
            this.MapObject.addInteraction(this.interaction);
            this.interaction.on('drawend', function (event) {
                // Supprime si besoin les anciennes features
                if (this_.multiGeom !== true) {
                    var features = this_.FeatureOverlay.getSource().getFeaturesCollection().clear();
                }
                var featureID = featureID + 1;
                event.feature.d_ = featureID;
                this_.Features.changed();
                this_.callFeatureAddedEvents(event.feature);
            });
            break;
        case 'polygon':
            this.interaction = new ol.interaction.Draw({
                source: this.FeatureOverlay.getSource(),
                type: "Polygon"
            });
            this.interaction.set('binded-button', button);
            this.interaction.set('type', 'draw_polygon');
            this.MapObject.addInteraction(this.interaction);
            this.interaction.on('drawend', function (event) {
                // Supprime si besoin les anciennes features
                if (this_.multiGeom !== true) {
                    var features = this_.FeatureOverlay.getSource().getFeaturesCollection().clear();
                }
                var featureID = featureID + 1;
                event.feature.d_ = featureID;
                this_.Features.changed();
                this_.callFeatureAddedEvents(event.feature);
            });
            break;
        case 'select':
            var button = $(this.Target).find('.map-Select');
            this.setSelectInteraction(button);
            break;
        case 'remove':
            this.interaction = new ol.interaction.Select({
                condition: ol.events.condition.click,
                source: this.FeatureOverlay.getSource()
            });
            this.activeSelectHover();
            this.interaction.set('binded-button', button);
            this.interaction.set('type', 'remove');
            this.MapObject.addInteraction(this.interaction);
            this.interaction.getFeatures().on("add", function () {
                this_.removeFeature(this_.interaction.getFeatures().getArray()[0]);
            });
            break;
        case 'modify':
            this.interaction = new ol.interaction.Modify({
                features: this.FeatureOverlay.getSource().getFeaturesCollection()
            });
            this.interaction.set('binded-button', button);
            this.interaction.set('type', 'modify');
            this.MapObject.addInteraction(this.interaction);
            this.interaction.on('modifyend', function () {
                this_.Features.changed();
            });
            break;
        case 'divide_segment':
            this.interaction = new ol.interaction.Draw({
                source: this.FeatureDisplayOverlay.getSource(),
                type: "Point"
            });
            this.interaction.set('binded-button', button);
            this.interaction.set('type', 'draw_point');
            this.MapObject.addInteraction(this.interaction);
            this.interaction.on('drawend', function (event) {

                // Supprime les anciens points
                this_.FeatureDisplayOverlay.getSource().getFeaturesCollection().clear();

                var oPointGeom = event.feature.getGeometry();
                if(goog.isArray(this_.Features.getArray())){
                    if(this_.Features.getArray().length == 1){
                        if (goog.isDefAndNotNull(this_.Features.getArray()[0].getGeometry())) {


                            var oLineStringGeom = this_.Features.getArray()[0].getGeometry();


                            if (!this_.lineIntersectsPoint_(oLineStringGeom, oPointGeom) || oLineStringGeom.getType() !== 'LineString') {
                                $['notify'](this_.oTranslates['COMPONENT_MAP_DIVIDE_SEGMENT_POINT_NOT_VALID'], 'error');
                                setTimeout(function () {
                                    this_.FeatureDisplayOverlay.getSource().getFeaturesCollection().clear();
                                });
                            } else {

                                // Récupère les deux LineStrings résultants
                                var aDividedLineStrings = this_.divideLineString_(oLineStringGeom, oPointGeom);

                                bootbox.confirm("<h4>Voulez vous diviser le segment à cet endroit ?<h4>", function(result){
                                    this_.FeatureDisplayOverlay.getSource().getFeaturesCollection().clear();
                                    if (result) {

                                        var sFeature1, sFeature2;
                                        if (goog.isDefAndNotNull(this_.oFormValues)) {

                                            var sFeature1 = this_.getStringByFeatures([new ol.Feature({
                                                geometry: aDividedLineStrings[0]
                                            })]);

                                            var sFeature2 = this_.getStringByFeatures([new ol.Feature({
                                                geometry: aDividedLineStrings[1]
                                            })]);
                                        }
                                        if (goog.isDefAndNotNull(sFeature1) && goog.isDefAndNotNull(sFeature2)) {

                                            // Affiche un formulaire pour insérrer un enregistrement avec la feature2
                                            this_.showDividedSegmentForm_(sFeature1, sFeature2);
                                        }
                                    }
                                });
                            }
                        }
                    }
                }
            });
            break;
        default:
            break;
    }

    // Re-ajoute l'accrochage car il doit être ajouté après les interactions de dessin
    this_.MapObject.addInteraction(this.snapInteraction_);
};

nsVitisComponent.Map.prototype.getFormReaderScope_ = function (){
    return angular.element(this.Target).scope();
}

nsVitisComponent.Map.prototype.showDividedSegmentForm_ = function (sFeature1, sFeature2){

    var this_ = this;
    var sGeomField = this.sFormName;
    var oFormScope = this.getFormReaderScope_();
    var oSubformDefinition = angular.copy(oFormScope['oFormDefinition']);
    var sSubformDefinitionName = angular.copy(oFormScope['sFormDefinitionName']);
    var envSrvc = angular.element(vitisApp.appMainDrtv).injector().get(["envSrvc"]);
    var oSubformValues = {}
    var sEnvMode = angular.copy(envSrvc['sMode']);
    var oEnvFormValues = angular.copy(envSrvc["oFormValues"]);
    var oEnvFormDefinition = angular.copy(envSrvc["oFormDefinition"]);
    var sEnvFormDefName = angular.copy(envSrvc["sFormDefinitionName"]);
    var sEnvRessourceId = angular.copy(envSrvc["sId"]);
    var sModalId;

    // Affecte les nouvelles valeurs
    oSubformValues[sSubformDefinitionName] = angular.copy(this.oFormValues);
    oSubformValues[sSubformDefinitionName][sGeomField] = sFeature2;

    // Remplace les identifiants pour éviter les conflits
    if (goog.isDefAndNotNull(oSubformDefinition[sSubformDefinitionName]['rows'])) {
        for (var i = 0; i < oSubformDefinition[sSubformDefinitionName]['rows'].length; i++) {
            if (goog.isDefAndNotNull(oSubformDefinition[sSubformDefinitionName]['rows'][i]['fields'])) {
                for (var ii = 0; ii < oSubformDefinition[sSubformDefinitionName]['rows'][i]['fields'].length; ii++) {
                    if (goog.isDefAndNotNull(oSubformDefinition[sSubformDefinitionName]['rows'][i]['fields'][ii]['id'])) {
                        oSubformDefinition[sSubformDefinitionName]['rows'][i]['fields'][ii]['id'] += '_subform';
                    }
                }
            }
        }
    }

    // Id du formulaire
    oSubformDefinition[sSubformDefinitionName]['name'] = 'formreader_' + oFormScope['sFormUniqueName'] + '_standard_subform_form';

    // Suppime les boutons
    oSubformDefinition[sSubformDefinitionName]['rows'].pop();

    // Événements
    oSubformDefinition[sSubformDefinitionName]['beforeEvent'] = function(){

        envSrvc['sMode'] = 'insert';
        envSrvc["oFormValues"] = oSubformValues;
        envSrvc["oFormDefinition"] = oSubformDefinition;
        envSrvc["sFormDefinitionName"] = sSubformDefinitionName;

    }
    oSubformDefinition[sSubformDefinitionName]['afterEvent'] = function(){

        oFormScope['closeSubformModal']('#' + sModalId);

        this_.oFormValues[sGeomField] = sFeature1;
        oEnvFormValues[sSubformDefinitionName] = this_.oFormValues;

        envSrvc['sMode'] = sEnvMode;
        envSrvc["sId"] = sEnvRessourceId;
        oFormScope["oFormValues"] = envSrvc["oFormValues"] = oEnvFormValues;
        oFormScope["oFormDefinition"] = envSrvc["oFormDefinition"] = oEnvFormDefinition;
        oFormScope["sFormDefinitionName"] = envSrvc["sFormDefinitionName"] = sEnvFormDefName;

        oFormScope['sendForm']();
    }

    sModalId = oFormScope['displaySubformModal'](oSubformDefinition, sSubformDefinitionName, oSubformValues, 'insert', 'Nouveau segment');
}

/**
 * lineIntersectsPoint_ - Retourne true si le point intersecte la ligne
 *
 * @param  {ol.geom.LineString} olLineStringGeom
 * @param  {ol.geom.Point} olPointGeom
 * @return {boolean}
 */
nsVitisComponent.Map.prototype.lineIntersectsPoint_ = function (olLineStringGeom, olPointGeom){

    var aPointCoords = olPointGeom.getCoordinates();
    var oClosestPointCoords = olLineStringGeom.getClosestPoint(olPointGeom.getCoordinates());
    var iBuffer = 1;

    // Cercle de 1 mètre positionné au point le plus près de la ligne
    var olCircleGeom = new ol.geom.Circle(oClosestPointCoords, iBuffer);

    // Retourne true/false si le point se trouve sur le cercle
    var bIntersectsCircle = olCircleGeom.intersectsCoordinate(aPointCoords);
    return bIntersectsCircle;
}

/**
 * lineContainsPoint_ - Retourne true si le point fait partie des sommets de la ligne
 *
 * @param  {ol.geom.LineString} olLineStringGeom
 * @param  {ol.geom.Point} olPointGeom
 * @return {boolean}
 */
nsVitisComponent.Map.prototype.lineContainsPoint_ = function (olLineStringGeom, olPointGeom){

    var aPointCoords = olPointGeom.getCoordinates();
    var aLineStringCoords = olLineStringGeom.getCoordinates();

    for (var i = 0; i < aLineStringCoords.length; i++) {
        if(aLineStringCoords[i][0] === aPointCoords[0] &&
            aLineStringCoords[i][1] === aPointCoords[1]){
            return true;
        }
    }

    return false;
}

/**
 * divideLineString_ - Divise un segment en deux par rapport à un point
 * et renvoi les deux segments résultants
 *
 * @param  {ol.geom.LineString} olLineStringGeom
 * @param  {ol.geom.Point} olPointGeom
 * @return {array}
 */
nsVitisComponent.Map.prototype.divideLineString_ = function (olLineStringGeom, olPointGeom){

    // Si le point n'est pas un des sommets du segment
    if (!this.lineContainsPoint_(olLineStringGeom, olPointGeom)) {
        // Ajoute le sommet sur le segment
        olLineStringGeom = this.addPointOnTheLineString_(olLineStringGeom, olPointGeom);
    }

    var aPointCoords = olPointGeom.getCoordinates();
    var aLineStringCoords = olLineStringGeom.getCoordinates();
    var bDevidePointPassed = false;
    var oLinestring1 = new ol.geom.LineString([]);
    var oLinestring2 = new ol.geom.LineString([]);

    // Stoque les coordonnées avant et après le point de découpe
    // le point se découpe sera stoqué dans les deux segments
    for (var i = 0; i < aLineStringCoords.length; i++) {

        // Coordonnées avant le point de découpe
        if (!bDevidePointPassed) {
            oLinestring1.appendCoordinate(aLineStringCoords[i]);
        }

        if (aLineStringCoords[i][0] === aPointCoords[0] &&
            aLineStringCoords[i][1] === aPointCoords[1]) {
            bDevidePointPassed = true;
        }

        // Coordonnées après le point de découpe
        if (bDevidePointPassed) {
            oLinestring2.appendCoordinate(aLineStringCoords[i]);
        }

    }

    return [oLinestring1, oLinestring2]
}

/**
 * getLineStringClosestNode_ function - Retourne le sommet le plus proche de olPointGeom
 *
 * @param  {ol.geom.LineString} olLineStringGeom
 * @param  {ol.geom.Point} olPointGeom
 * @return {ol.geom.Point}
 */
nsVitisComponent.Map.prototype.getLineStringClosestNode_ = function (olLineStringGeom, olPointGeom){

    var aPointCoords = olPointGeom.getCoordinates();
    var aLineStringCoords = olLineStringGeom.getCoordinates(aPointCoords);
    var iMinLength = Infinity;
    var iPointsLength, iClosestPointIndex, oClosestPoint;

    for (var i = 0; i < aLineStringCoords.length; i++) {
        iPointsLength = new ol.geom.LineString([aLineStringCoords[i], aPointCoords]).getLength();
        if (iPointsLength < iMinLength) {
            iClosestPointIndex = angular.copy(i);
            iMinLength = iPointsLength;
        }
    }

    if (goog.isDefAndNotNull(aLineStringCoords[iClosestPointIndex])) {
        oClosestPoint = new ol.geom.Point(aLineStringCoords[iClosestPointIndex]);
    }

    return oClosestPoint;
}

/**
 * addPointOnTheLineString_ function - Ajoute un point sur une LineString
 *
 * @param  {ol.geom.LineString} olLineStringGeom
 * @param  {ol.geom.Point} olPointGeom
 * @return {ol.geom.LineString}
 */
nsVitisComponent.Map.prototype.addPointOnTheLineString_ = function (olLineStringGeom, olPointGeom){

    // Si le point est déjà sur la ligne, renvoie la ligne
    if (this.lineContainsPoint_(olLineStringGeom, olPointGeom)) {
        console.error('sommet existant');
        return olLineStringGeom;
    }

    var aPointCoords = olPointGeom.getCoordinates();
    var aLineStringCoords = olLineStringGeom.getCoordinates();
    var olClosestNode = this.getLineStringClosestNode_(olLineStringGeom, olPointGeom);
    var olClosestNodeCoords = olClosestNode.getCoordinates();
    var iClosestNodeIndex, bIsPointAfterClosestNode;

    // Cherche l'index du point le plus proche
    for (var i = 0; i < aLineStringCoords.length; i++) {
        if(aLineStringCoords[i][0] === olClosestNodeCoords[0] &&
            aLineStringCoords[i][1] === olClosestNodeCoords[1]){
            iClosestNodeIndex = angular.copy(i);
        }
    }

    // Cherche si le point à ajouter est situé avant ou après le point le plus proche
    if (goog.isDefAndNotNull(aLineStringCoords[iClosestNodeIndex])) {
        if (!goog.isDefAndNotNull(aLineStringCoords[iClosestNodeIndex + 1])) {
            // Si on se situe en bout de ligne
            bIsPointAfterClosestNode = false;
        } else if (!goog.isDefAndNotNull(aLineStringCoords[iClosestNodeIndex - 1])) {
            // Si on se situe en début de ligne
            bIsPointAfterClosestNode = true;
        } else {

            var oBeforeLineString = new ol.geom.LineString(
                [aLineStringCoords[iClosestNodeIndex - 1],
                aLineStringCoords[iClosestNodeIndex]]);

            var oAfterLineString = new ol.geom.LineString(
                [aLineStringCoords[iClosestNodeIndex],
                aLineStringCoords[iClosestNodeIndex + 1]]);

            if (this.lineIntersectsPoint_(oBeforeLineString, olPointGeom)) {
                bIsPointAfterClosestNode = false;
            } else if (this.lineIntersectsPoint_(oAfterLineString, olPointGeom)) {
                bIsPointAfterClosestNode = true;
            } else {
                bIsPointAfterClosestNode = true;
            }
        }
    }

    if (bIsPointAfterClosestNode) {
        aLineStringCoords.splice(iClosestNodeIndex + 1, 0, aPointCoords)
        olLineStringGeom.setCoordinates(aLineStringCoords);
    }else {
        aLineStringCoords.splice(iClosestNodeIndex, 0, aPointCoords)
        olLineStringGeom.setCoordinates(aLineStringCoords);
    }

    return olLineStringGeom;
}

/**
 * Set the select interaction
 * @param {type} button_select
 */
nsVitisComponent.Map.prototype.setSelectInteraction = function (button_select) {

    var this_ = this;

    this.interaction = new ol.interaction.Select({
        condition: ol.events.condition.click,
        source: this.FeatureOverlay.getSource()
    });
    if (goog.isDefAndNotNull(button_select)) {
        this.interaction.set('binded-button', button_select);
        this.interaction.set('type', 'select');
    }
    this.MapObject.addInteraction(this.interaction);

    this.interaction.getFeatures().on("change:length", function () {

        var aOverlayFeatures = this_.FeatureOverlay.getSource().getFeatures();
        var selectInteractionFeature = this_.interaction.getFeatures().getArray()[0];
        var oSelectedFeature = aOverlayFeatures[aOverlayFeatures.indexOf(selectInteractionFeature)];

        this_.selectedFeature = goog.isDefAndNotNull(oSelectedFeature) ? oSelectedFeature : null;

        // Lance les événements featureSelectedEvents
        for (var i = 0; i < this_.featureSelectedEvents.length; i++) {
            if (goog.isFunction(this_.featureSelectedEvents[i])) {
                this_.featureSelectedEvents[i].call(this, this_.selectedFeature);
            }
        }
    });

    this.interaction.displaySelectedFeatureContour = function () {
        var selectInteractionFeature = this_.interaction.getFeatures().getArray()[0];

        var fill = new ol.style.Fill({
            color: 'rgba(255,255,255,0)'
        });
        var stroke = new ol.style.Stroke({
            color: 'rgba(0,0,0,0.4)',
            width: 2,
            lineDash: [10]
        });
        var selectStyle = new ol.style.Style({
            image: new ol.style.Circle({
                fill: fill,
                stroke: stroke,
                radius: 50
            }),
            fill: fill,
            stroke: stroke
        });

        if (goog.isDefAndNotNull(selectInteractionFeature)) {
            var featureExtentArray = selectInteractionFeature.getGeometry().getExtent();
            if (selectInteractionFeature.getGeometry().getType() === 'Point') {
                var selectionFeature = new ol.Feature({
                    geometry: selectInteractionFeature.getGeometry()
                });
            } else {
                var xmin = featureExtentArray[0];
                var ymin = featureExtentArray[1];
                var xmax = featureExtentArray[2];
                var ymax = featureExtentArray[3];
                var featureExtentGeometry = new ol.geom.Polygon([[[xmin, ymin], [xmin, ymax], [xmax, ymax], [xmax, ymin]]]);
                var selectionFeature = new ol.Feature({
                    geometry: featureExtentGeometry
                });
            }
            selectionFeature.setStyle(selectStyle);
            this_.interaction.getFeatures().push(selectionFeature);
        }
    };

    this.interaction.on("select", function () {
        this_.interaction.displaySelectedFeatureContour();
    });

    this.activeSelectHover();
};

/**
 * Remove the passed feature
 * @param {ol.Feature} oFeature
 */
nsVitisComponent.Map.prototype.removeFeature = function (oFeature) {

    this.FeatureOverlay.getSource().removeFeature(oFeature);
    this.interaction.getFeatures().clear();
    if (goog.isDefAndNotNull(this.selectHover_)) {
        this.selectHover_.getFeatures().clear();
    }

    this.FeatureOverlay.getSource().changed();
    this.Features.changed();
};

/**
 * Load a Map.json to set layers and view of the map
 * @param {Object} tree Json tree to set the map (he's generated by Vmap)
 * @export
 */
nsVitisComponent.Map.prototype.loadTree = function (tree) {

    if (this.MapObject.getLayers().getArray().length > 0)
        this.MapObject.getLayers().clear();

    this.Layers.length = 0;
    var oLayer;
    var aTree = [];
    var tileSize = [256, 256];

    var oMapDefinition = angular.copy(tree);

    /**
     * Objet permettant de lancer des fonctions utiles
     * pour parser la définition de la carte
     * @type MapJSON
     */
    var oMapJSON = new MapJSON({
        'properties': this.$propertiesSrvc_
    });

    /**
     * Vue à utiliser
     * @type ol.View
     */
    var olView = oMapJSON.getViewFromDef(oMapDefinition, {
        'size': this.MapObject.getSize(),
        'tileSize': tileSize
    });

    /**
     * Tableau de couches ol
     * @type Array<ol.layer>
     */
    var olLayers = oMapJSON.getLayersFromDef(oMapDefinition, {
        'size': this.MapObject.getSize(),
        'tileSize': tileSize
    });

    // Définit la vue de la carte
    this.MapObject.setView(olView);

    // Définit les couches de la carte
    for (var i = 0; i < olLayers.length; i++) {
        var oLayer = {
            'index': i,
            'layer': olLayers[i]
        };
        this.Layers.push(oLayer);
        this.MapObject.addLayer(olLayers[i]);
    }

    // Définit l'arbre utilisé pour l'outil layer_tree
    for (var i = 0; i < oMapDefinition["children"].length; i++) {
        if (goog.isDef(oMapDefinition["children"][i]['children'])) {
            aTree.push({
                'service': oMapDefinition["children"][i]['name'],
                'layers': oMapDefinition["children"][i]['children']
            });
        }
    }

    return aTree;
};

/**
 * Calculates if the viewExtent is inside the projExtent
 * @param {ol.Extent} viewExtent
 * @param {ol.Extent} projExtent
 * @returns {Boolean} true if the viewExtent is inside the projExtent
 */
nsVitisComponent.Map.prototype.isInsideProjExtent = function (viewExtent, projExtent) {

    var bIsInside = true;

    if (projExtent[0] > viewExtent[0])
        bIsInside = false;
    if (projExtent[1] > viewExtent[1])
        bIsInside = false;
    if (projExtent[2] < viewExtent[2])
        bIsInside = false;
    if (projExtent[3] < viewExtent[3])
        bIsInside = false;

    return bIsInside;
};

/**
 * return the scal eof the map
 * @return {number} scale of the map
 * @export
 */
nsVitisComponent.Map.prototype.getScale = function () {
    var wgs84Sphere_ = new ol.Sphere(6378137);
    var projection = this.MapObject.getView().getProjection();

    var map = this.MapObject;

    // récupère les coordonnées d'une ligue de 1cm (avec le zoom en cours)
    var line = map.getView().calculateExtent([37.795275591, 0]);
    var c1 = ol.proj.transform([line[0], line[1]], projection, 'EPSG:4326');
    var c2 = ol.proj.transform([line[2], line[3]], projection, 'EPSG:4326');

    // Récuère la longueur sur la carte de la ligne de 1cm
    var length = wgs84Sphere_.haversineDistance(c1, c2);

    // donc 1m sur la carte correspond à (length mètres dans la réalité x 100)
    var scale = length * 100;
    // stringifier
    //scale = this.getPrettyScale(scale);

    return scale;
};

/**
 * set the scale of the map
 * @param {number} scale set the scale of the map and change the view
 * @export
 */
nsVitisComponent.Map.prototype.setScale = function (scale) {

    // calcule et va à l'échelle celon une règle de 3
    var currentScale = this["getScale"]();
    var currentResolution = this.MapObject.getView().getResolution();

    var scaleResolution = scale * currentResolution / currentScale;
    this.MapObject.getView().setResolution(scaleResolution);

    // Ajuste l'échelle en augementant la résolution
    currentScale = this["getScale"]();
};

/**
 * Set the map extent
 * @param {ol.Extent} extent
 */
nsVitisComponent.Map.prototype.setExtent = function (extent) {
    this.Extent = extent;
    this.MapObject.getView().fit(extent, {nearest: true});
};

/**
 * return the zoom value
 * @return {number} zoom Value
 * @export
 */
nsVitisComponent.Map.prototype.getZoom = function () {
    return this.MapObject.getView().getZoom();
};

/**
 * set the zoom value
 * @param {number} zoom value to resize Map
 * @export
 */
nsVitisComponent.Map.prototype.setZoom = function (zoom) {
    this.MapObject.getView().setZoom(zoom);
};

/**
 * set the coordinate of the View's center
 * @param {ol.Coordinate} coord coordinate of the center
 * @export
 */
nsVitisComponent.Map.prototype.setCenter = function (coord) {
    this.MapObject.getView().setCenter(coord);
};

/**
 * set the Controls on the map
 * @param {Object} obj object to configure Controls
 * @export
 */
nsVitisComponent.Map.prototype.setControls = function (obj) {
    var this_ = this;
    // on supprime tous les Controls
    this.MapObject.getControls().clear();
    //si le zoom est défini on le supprime
    if (this.ZoomGroup !== "undefined") {
        $(this.ZoomGroup).remove();
    }
    //on réinstancie les controls OK
    if (obj["ZO"]) {
        this.ZoomGroup = this.zoomControls(this.Target);
    }
    if (obj["MP"]) {
        this.MapObject.addControl(new ol.control.MousePosition({
            coordinateFormat: function (coordinate) {
                //var coord = ol.proj.transform(coordinate, "EPSG:3857", this_.proj_);
                return [coordinate[0].toPrecision(this_.Accuracy), coordinate[1].toPrecision(this_.Accuracy)];
            }
        }));
        $(".ol-mouse-position").css("bottom", "8px");
        $(".ol-mouse-position").css("top", "auto");
        $(".ol-mouse-position").css("background", "rgba(218, 94, 209, 0.58)");
        $(".ol-mouse-position").css("color", "#ffffff");
        $(".ol-mouse-position").css("border-radius", "4px");
    }
    if (obj["SL"]) {
        this.MapObject.addControl(new ol.control.ScaleLine());

        $(".ol-scale-line").css("background", "rgba(218, 94, 209, 0.58)");
    }
    if (obj["CP"]) {
        this.MapObject.addControl(new ol.control.Control({
            "element": this_.element,
            "render": function () {
                $(this_.element).html(this_.proj_);
            }
        }));
    }
};

/**
 * set the Interactions on the map
 * @param {Object} obj object to configure Interactions
 * @export
 */
nsVitisComponent.Map.prototype.setInteractions = function (obj) {
    this.MapObject.getInteractions().clear();
    if (this.IntractionsGroup !== "undefined") {
        $(this.IntractionsGroup).remove();
    }
    this.IntractionsGroup = this.addInteractions(this.Target, obj);

    ol.interaction.defaults().forEach(function (el, index, arr) {
        this.MapObject.addInteraction(el);
    }, this);
};

/**
 * set the Source for Bing's Map
 * @param {Object} objSource an object who contain key, culture, imageryset for the Map
 * @export
 */
nsVitisComponent.Map.prototype.setBingSource = function (objSource) {
    this.MapObject.getLayers().getArray()[0].setSource(
            new ol.source.BingMaps({
                culture: objSource["culture"],
                key: objSource["key"],
                imagerySet: objSource["style"]
            })
            );
};

/**
 * return the config object of the map
 * @return {object} an object who contain extent, coord of center and scale of the map
 * @export
 */
nsVitisComponent.Map.prototype.getConfig = function () {
    this.Config["extent"] = this.View.calculateExtent(this.MapObject.getSize());
    this.Config["coord"] = this.View.getCenter();
    this.Config["scale"] = Math.round(this["getScale"]());
    return this.Config;
};

/**
 * return the map Object
 * @return {Object} the Map object
 * @export
 */
nsVitisComponent.Map.prototype.getMap = function () {
    return this.MapObject;
};

/**
 * set the Map's View
 * @param {Object} options an object to configure the view of the map
 * @export
 */
nsVitisComponent.Map.prototype.setView = function (options) {

    this.proj_ = options["proj"];

    var center, extent;
    /*if (options["proj"] !== "EPSG:3857") {
     if (goog.isDef(options["center"]["coord"]))
     center = ol.proj.transform(options["center"]["coord"], options["proj"], "EPSG:3857");
     if (goog.isDef(options["center"]["extent"]))
     extent = ol.proj.transform(options["center"]["extent"], options["proj"], "EPSG:3857");
     } else {*/
    if (goog.isDef(options["center"]["coord"]))
        center = options["center"]["coord"];
    if (goog.isDef(options["center"]["extent"]))
        extent = options["center"]["extent"];
    //}

    if (options["center"]["coord"] && options["center"]["zoom"]) {
        this.View = new ol.View({
            "center": center,
            "zoom": options["center"]["zoom"],
            "projection": ol.proj.get(this.proj_)
        });
        this.MapObject.setView(this.View);
    } else {
        this.View = new ol.View({
            "center": [((extent[0] + extent[2]) / 2), ((extent[1] + extent[3]) / 2)],
            //"zoom": options["center"]["zoom"],
            "projection": ol.proj.get(this.proj_)
        });
        this.MapObject.setView(this.View);
        this.MapObject.getView().fit(extent, {nearest: true});
    }

    var currentProj = "";
    for (var i = 0; i < this.oProj_["projections"]; i++) {
        if (this.proj_ === this.oProj["projections"][i]["code"]) {
            currentProj = this.oProj["projections"][i]["name"];
        }
    }
    if (currentProj !== '') {
        $(".ol-current-projection-studio").html(currentProj);
    } else {
        $(".ol-current-projection-studio").html(this.proj_);
    }
};

/**
 * load a json tree to configure the map
 * @param {Object} tree  Json tree (create him with Vmap)
 * @export
 */
nsVitisComponent.Map.prototype.setJsonTree = function (tree) {
    this.loadTree(tree);
};

/**
 * return the Features draw on the map
 * @return {array.<ol.Feature>} the Feature on the map
 * @export
 */
nsVitisComponent.Map.prototype.getFeatures = function () {
    return this.Features.getArray();
};

/**
 * return the EPSG reference of the projection
 * @return {string} projection code EPSG
 * @export
 */
nsVitisComponent.Map.prototype.getProj = function () {
    return this.proj_;
};

/**
 * return extent of the view
 * @return {array <number>} coord of extent
 * @export
 */
nsVitisComponent.Map.prototype.getExtent = function () {
    return this.View.calculateExtent(this.MapObject.getSize());
};

/**
 * return the center of the view
 * @return {array <number>} coord of center
 * @export
 */
nsVitisComponent.Map.prototype.getCenter = function () {
    return this.View.getCenter();
};

/**
 * return reprojection of coordinate pass as data in a projection to another projection
 * @param {array.<float>} data data for the transformation
 * @param {string} epsgFrom origin's Code EPSG
 * @param {string} epsgTo destination's Code EPSG
 * @return {array.<float>} projection code EPSG
 * @export
 */
nsVitisComponent.Map.prototype.transformer = function (data, epsgFrom, epsgTo) {
    return ol.proj.transform(data, epsgFrom, epsgTo);
};

/**
 * return reprojection of extent pass as data in a projection to another projection
 * @param {array.<float>} extent extent for the transformation
 * @param {string} epsgFrom origin's Code EPSG
 * @param {string} epsgTo destination's Code EPSG
 * @return {array.<float>} projection code EPSG
 * @export
 */
nsVitisComponent.Map.prototype.transformExtent = function (extent, epsgFrom, epsgTo) {
    return ol.proj.transformExtent(extent, epsgFrom, epsgTo);
};

/**
 * return the Features draw on the map
 * @return {array.<ol.Feature>} the Feature on the map
 * @export
 */
nsVitisComponent.Map.prototype.on = function (event, handler, this_) {
    this.MapObject.on(event, handler, this_);
};

/**
 * Draw a point on the given coordinates
 * @param {number} x
 * @param {number} y
 * @private
 */
nsVitisComponent.Map.prototype.drawPoint = function (x, y) {
    var Map = this.getMap();
    var this_ = this;
    var Point = new ol.Feature({
        "geometry": new ol.geom.Point([x, y])
    });
    var array = Map.getLayers().getArray();

    array.forEach(function (item, index, array) {
        if (goog.isDef(item.getSource().getFeatures)) {
            var Features = item.getSource().getFeaturesCollection();

            if (this_.multiGeom !== true) {
                item.getSource().getFeaturesCollection().clear();
            }

            Features.push(Point);
        }
    });
    this.setCenter([x, y]);
};

/**
 * Add function to the featureSelectedEvents list
 * the function will be called when a feature will be selected
 * @param {function} event
 */
nsVitisComponent.Map.prototype.onFeatureSelected = function (event) {
    this.featureSelectedEvents.push(event);
};

/**
 * Add function to the featuresChangedEvents list
 * the function will be called when the features changed
 * @param {function} event
 */
nsVitisComponent.Map.prototype.onFeaturesChanged = function (event) {
    this.featuresChangedEvents.push(event);
};

/**
 * Add function to the featureAddedEvents list
 * the function will be called when a feature is added
 * @param {function} event
 */
nsVitisComponent.Map.prototype.onFeatureAdded = function (event) {
    this.featureAddedEvents.push(event);
};

/**
 * Add function to the featureAddedOnceEvents list
 * the function will be called oncewhen a feature is added
 * @param {function} event
 */
nsVitisComponent.Map.prototype.onceFeatureAdded = function (event) {
    this.featureAddedOnceEvents.push(event);
};

/**
 * Call the featureAddedEvents
 * @param {ol.Feature} feature
 */
nsVitisComponent.Map.prototype.callFeatureAddedEvents = function (feature) {
    // Lance les événements featureAddedEvents
    for (var i = 0; i < this.featureAddedEvents.length; i++) {
        if (goog.isFunction(this.featureAddedEvents[i])) {
            this.featureAddedEvents[i].call(this, feature);
        }
    }
    // Lance et vide les événements featureAddedOnceEvents
    for (var i = 0; i < this.featureAddedOnceEvents.length; i++) {
        if (goog.isFunction(this.featureAddedOnceEvents[i])) {
            this.featureAddedOnceEvents[i].call(this, feature);
        }
    }
    goog.array.clear(this.featureAddedOnceEvents);
};
