/* global goog, ol, nsVitisComponent, vitisApp, formReader, bootbox */

/**
 * @author: Anthony Borghi, Armand Bahi
 * @Description: Fichier contenant la classe nsVitisComponent.Map
 * Permet d'instancier un composant OpenLayers3
 */

'use strict';

goog.provide('nsVitisComponent.MapWorkbench');

goog.require('formReader');
goog.require('nsVitisComponent.Map');

/**
 * MapWorkbench Directive
 * @param {type} $timeout
 * @returns {nsVitisComponent.MapWorkbenchDirective}
 * @ngInject
 */
nsVitisComponent.MapWorkbenchDirective = function ($timeout) {
    return {
        restrict: 'A',
        controller: 'AppMapWorkbenchController',
        controllerAs: 'ctrl',
        templateUrl: window.location.protocol + "//" + window.location.hostname + (window.location.port ? ':' + window.location.port : '') + "/" + sessionStorage["appEnv"] + '/javascript/externs/formReader/component/map_workbench/map_workbench.html',
        compile: function (element, attributes) {
            return {
                pre: function (scope, element, attributes, controller, transcludeFn) {

                    var aColumns = [{name: 'map_workbench_index', visible: false}];
                    if (goog.isArray(scope['field']['attributes_def'])) {
                        for (var i = 0; i < scope['field']['attributes_def'].length; i++) {
                            aColumns.push({
                                name: scope['field']['attributes_def'][i]['name'],
                                displayName: scope['field']['attributes_def'][i]['label']
                            });
                        }
                    }
                    scope['gridOptions'] = {
                        'enableSorting': true,
                        'enableRowSelection': true,
                        'enableSelectAll': true,
                        'multiSelect': false,
                        'columnDefs': aColumns,
                        'data': []
                    };

                    // Valeurs par défaut obligatoires
                    scope['featureForm'] = {
                        'draw': {
                            'symbol': 'fa-home',
                            'color': 'red',
                            'outline_color': 'white',
                            'size': 4,
                            'rotation': 0,
                            'dash': 20
                        },
                        'text': {
                            'text': '',
                            'size': 14,
                            'font': 'Arial',
                            'color': 'white',
                            'outline_color': 'black',
                            'outline_size': 1,
                            'offsetX': 0,
                            'offsetY': 10,
                            'rotation': 0
                        },
                        'attributes': {}
                    };

                    scope['aIcons'] = [];
                    scope['olSelectedFeature'] = null;
                    scope['sSelectedFeatureType'] = null;
                    scope['bFormDisabled'] = true;
                    scope['aFeatures'] = [];
                    scope['custom_form'] = null;
                    scope['oSubformScope'] = null;

                    var lastRowChange = Date.now();
                    var lastRowChangeTimeout = 100;
                    var lastRowSelected = null;

                    /**
                     * Unselect the current feature
                     * @param {boolean} bremoveInteraction
                     */
                    scope['clearSelectedFeature'] = function (bremoveInteraction) {
                        scope['olSelectedFeature'] = null;
                        if (goog.isDefAndNotNull(scope['oMap'].interaction)) {
                            if (goog.isFunction(scope['oMap'].interaction.getFeatures)) {
                                if (goog.isDefAndNotNull(scope['oMap'].interaction.getFeatures())) {
                                    scope['oMap'].interaction.getFeatures().clear();
                                    if (bremoveInteraction) {
                                        scope['oMap'].MapObject.removeInteraction(scope['oMap'].interaction);
                                    }
                                }
                            }
                        }
                        lastRowSelected = null;
                        scope['gridApi']['selection']['clearSelectedRows']();
                    };

                    /**
                     * Allow to close a modal
                     * @param {type} identifier
                     */
                    scope['closeModal'] = function (identifier) {
                        $(identifier).modal('hide');
                    };

                    scope['gridOptions']['onRegisterApi'] = function (gridApi) {
                        scope['gridApi'] = gridApi;
                        scope['gridApi']['selection']['on']['rowSelectionChanged'](scope, function (row) {
                            // Temps minimum entre deux changements
                            if ((lastRowChange + lastRowChangeTimeout) > Date.now()) {
                                return null;
                            }
                            lastRowChange = Date.now();

                            var olGridSelectedFeature = null;
                            var selectedIndex = null;
                            var toSelectIndex = null;

                            // Récupère la feature dans la liste
                            for (var i = 0; i < scope['aFeatures'].length; i++) {
                                if (scope['aFeatures'][i].get('attributes')['map_workbench_index'] === row['entity']['map_workbench_index']) {
                                    olGridSelectedFeature = scope['aFeatures'][i];
                                    break;
                                }
                            }

                            // Tests si la feature est déjà sélectionnée dans la grille
                            if (goog.isDefAndNotNull(lastRowSelected)) {
                                if (lastRowSelected === row['entity']['map_workbench_index']) {
                                    scope['clearSelectedFeature'](true);
                                    return null;
                                }
                            }
                            lastRowSelected = row['entity']['map_workbench_index'];

                            // Tests si la feature est déjà sélectionnée dans la carte
                            if (goog.isDefAndNotNull(scope['oMap'].interaction)) {
                                if (goog.isFunction(scope['oMap'].interaction.get)) {
                                    if (scope['oMap'].interaction.get('type') === 'select') {
                                        var aFeatures = scope['oMap'].interaction.getFeatures().getArray();
                                        if (goog.isArray(aFeatures)) {
                                            if (aFeatures.length > 0) {
                                                var attributes = aFeatures[0].get('attributes');
                                                if (goog.isDefAndNotNull(attributes)) {
                                                    if (attributes['map_workbench_index'] === row['entity']['map_workbench_index']) {
                                                        return null;
                                                    }
                                                }
                                            }
                                        }
                                    }
                                }
                            }

                            // scope['oMap'].setSelectInteraction();
                            scope['oMap']['activeInteraction']('select', true);
                            scope['oMap'].interaction.getFeatures().push(olGridSelectedFeature);
                            setTimeout(function () {
                                scope['oMap'].interaction.displaySelectedFeatureContour();
                            });
                        });
                    };
                },
                post: function (scope, element, attributes, controller, transcludeFn) {

                    /**
                     * Get the subform scope
                     * @returns {object|undefined}
                     */
                    scope['getSubFormScope'] = function () {
                        return angular.element('#map_workbench_' + scope['sFormUniqueName'] + '_subform').scope();
                    };

                    var initMap = function () {

                        // Suppression de la carte si elle existe déjà
                        if (goog.isDef(scope['oMap'])) {
                            scope['oMap'].MapObject.setTarget(null);
                            delete scope['oMap'];
                        }

                        // Custom form ?
                        if (goog.isDefAndNotNull(scope['field']['custom_form'])) {

                            // Désactive la possibilité d'ajouter des features
                            scope['ctrl'].unactiveDrawInteractions_();
                        }

                        // Création de la carte
                        scope['oMap'] = new nsVitisComponent.Map({
                            'target': $(element).find('.form-map')[0],
                            'options': scope['field']['map_options'],
                            'hiddenFieldId': scope['field']['id'],
                            'oFormValues': scope['oFormValues'][scope['sFormDefinitionName']],
                            'sFormName': scope['field']['name'],
                            'oProj': scope.$root['proj']
                        });

                        scope['oMap'].onFeatureSelected(function (oFeature) {
                            scope.$applyAsync(function () {
                                scope['olSelectedFeature'] = oFeature;
                            });
                        });

                        scope['oMap'].onFeaturesChanged(function (aFeatures) {

                            if (!goog.isArray(aFeatures)) {
                                return null;
                            }

                            scope['aFeatures'] = aFeatures;
                            scope['ctrl'].fillGridFromFeatures_();
                        });

                        scope['oMap'].onFeatureAdded(function (oFeature) {
                            setTimeout(function () {
                                scope['olSelectedFeature'] = oFeature;
                                // scope['oMap'].setSelectInteraction();
                                scope['oMap']['activeInteraction']('select');
                                scope['oMap'].interaction.getFeatures().push(oFeature);
                                setTimeout(function () {
                                    // Affiche le contour
                                    scope['oMap'].interaction.displaySelectedFeatureContour();
                                    // Désactive la possibilité d'ajouter des features
                                    scope['ctrl'].unactiveDrawInteractions_();
                                });
                            });
                        });

                        setTimeout(function () {
                            scope['oMap'].saveFeatures();
                        });
                    };

                    var initSubform = function () {
                        scope['oSubformScope'] = scope['getSubFormScope']();
                        scope['oSubformScope']['loadSubForm']({
                            'sFormDefinitionName': 'update',
                            'oFormDefinition': scope['field']['custom_form']['form'],
                            'oFormValues': {
                                'update': {}
                            },
                            'oProperties': scope['oProperties']
                        });

                        var formSrvc = angular.element(vitisApp.appMainDrtv).injector().get(["formSrvc"]);

                        // Mode custom_form
                        if (goog.isDefAndNotNull(scope['oSubformScope'])) {
                            if (goog.isDefAndNotNull(scope['oSubformScope']['oSubformValues'])) {
                                scope.$watch('oSubformScope.oSubformValues', function (newValue, oldValue) {

                                    var newTextValue = formSrvc['getFormData']('update', true, {
                                        'oFormDefinition': scope['field']['custom_form']['form'],
                                        'oFormValues': goog.object.clone(newValue)
                                    });
                                    var oldTextValue = formSrvc['getFormData']('update', true, {
                                        'oFormDefinition': scope['field']['custom_form']['form'],
                                        'oFormValues': goog.object.clone(oldValue)
                                    });

                                    // vérifie le changement de type
                                    var sFieldName = scope['field']['custom_form']['featureStructure']['field'];
                                    var sFieldTypes = scope['field']['custom_form']['featureStructure']['types'];
                                    var newType = newTextValue[sFieldName];
                                    var oldType = oldTextValue[sFieldName];
                                    if (goog.isDefAndNotNull(newType) && oldType) {
                                        if (newType !== oldType) {
                                            if (goog.isDefAndNotNull(sFieldTypes[newType]) && sFieldTypes[oldType]) {
                                                var newGeomType = sFieldTypes[newType]['geometryType'];
                                                var oldGeomType = sFieldTypes[oldType]['geometryType'];
                                                if (newGeomType !== oldGeomType) {
                                                    $.notify('Ce type est incompatible avec la géométrie sélectionnée', 'error');
                                                    scope['oSubformScope']['setFormValues']({
                                                        'update': oldTextValue
                                                    });
                                                    return null;
                                                }
                                            }
                                        }
                                    }

                                    // Merge les attibuts
                                    goog.object.extend(scope['featureForm']['attributes'], newTextValue);

                                    // Édite le style
                                    if (goog.isDefAndNotNull(scope['olSelectedFeature'])) {
                                        scope['ctrl'].setStyleFromType_(scope['olSelectedFeature']);
                                    }
                                }, true);
                            }
                        }
                    };

                    // Instancie la carte
                    setTimeout(function () {
                        // Instanciation de la carte
                        initMap();

                        // Recharge la taille de la carte lorsqu'on modifie la taille du champ
                        scope.$watch('field.nb_cols', function (value, old) {
                            $timeout(function () {
                                scope['oMap'].MapObject.updateSize();
                            });
                        });

                        scope.$watch('field.style', function (value, old) {
                            $timeout(function () {
                                scope['oMap'].MapObject.updateSize();
                            });
                        }, true);

                        scope.$watch('oMap.bLayersTreeOpen', function (value, old) {
                            $timeout(function () {
                                scope['oMap'].MapObject.updateSize();
                            });
                        }, true);

                        scope.$on('studio resized', function () {
                            scope['oMap'].MapObject.updateSize();
                        });

                        scope.$on('updateMap', function (event, data) {
                            switch (data) {
                                case 'center':
                                    scope['oMap']['setCenter']([scope['field']['map_options']['center']['coord'][0], scope['field']['map_options']['center']['coord'][1]]);
                                    break;
                                case 'zoom':
                                    scope['oMap']['setZoom'](scope['field']['map_options']['center']['zoom']);
                                    break;
                                case 'source':
                                    scope['oMap']['setBingSource'](scope['field']['map_options']['source']);
                                    break;
                                case 'extent':
                                    scope['oMap']['setExtent'](scope['field']['map_options']['center']['extent']);
                                    break;
                                case 'controls':
                                    scope['oMap']['setControls'](scope['field']['map_options']['controls']);
                                    break;
                                case 'interact':
                                    scope['oMap']['setInteractions'](scope['field']['map_options']['interactions']);
                                    break;
                                case 'proj':
                                    var extent = scope['oMap']['getExtent']();
                                    var center = scope['oMap']['getCenter']();
                                    var proj = scope['oMap']['getProj']();

                                    scope['field']['map_options']['center']['extent'] = scope['oMap']['transformExtent'](extent, proj, scope['field']['map_options']['proj']);
                                    scope['field']['map_options']['center']['coord'] = scope['oMap']['transformer'](center, proj, scope['field']['map_options']['proj']);

                                    // Ré-initialise la carte
                                    initMap();
                                    break;
                                default:
                                    // Ré-initialise la carte
                                    initMap();
                                    break;
                            }
                        });
                    });

                    // Intancie le subform
                    setTimeout(function () {
                        if (goog.isDefAndNotNull(scope['field']['custom_form'])) {
                            if (goog.isDefAndNotNull(scope['field']['custom_form']['form'])) {
                                goog.object.extend(scope['field']['custom_form'], scope['oFormValues'][scope['sFormDefinitionName']]['custom_form']);
                                initSubform();
                            }
                        }
                    });
                }
            };
        }
    };
};
formReader.module.directive('appMapWorkbench', nsVitisComponent.MapWorkbenchDirective);

/**
 * Colorpicker directive
 * @returns {nsVitisComponent.appColorpickerDrtv}
 * @ngInject
 */
nsVitisComponent.appColorpickerDrtv = function () {
    return {
        scope: {
            'model': '='
        },
        link: function (scope, element, attrs) {
            $(element)['colorpicker']({
                format: "hex"
            });

            // Quand le colorpicker est changé
            var minTimeout = 100;
            var timeout;
            $(element).on("changeColor", function () {
                clearTimeout(timeout);
                timeout = setTimeout(function () {
                    scope['model'] = $(element)['colorpicker']('getValue');
                    scope.$parent.$apply();
                }, minTimeout);
            });
        }
    };
};
formReader.module.directive("appColorpicker", nsVitisComponent.appColorpickerDrtv);

/**
 * MapWorkbench Controller
 * @param {type} $scope
 * @param {type} $log
 * @returns {nsVitisComponent.MapWorkbenchController}
 * @ngInject
 */
nsVitisComponent.MapWorkbenchController = function ($scope, $log) {
    $log.info("MapWorkbenchController");
    var this_ = this;

    this.$log_ = $log;
    this.$scope_ = $scope;

    var featureFormTimeout = 50;
    var olSelectedFeatureTimeout = 200;

    var featureFormLastUpdate = Date.now();
    var olSelectedFeatureLastUpdate = Date.now();

    $scope.$watch('featureForm', function () {

        // Temps minimum entre deux mises à jour
        if ((featureFormLastUpdate + featureFormTimeout) > Date.now()) {
            return null;
        }
        featureFormLastUpdate = Date.now();

        if (!goog.isDefAndNotNull(this_.$scope_['oMap'])) {
            return 0;
        }
        if (!goog.isDefAndNotNull(this_.$scope_['oMap'].FeatureOverlay)) {
            return 0;
        }
        if (!goog.isDefAndNotNull(this_.$scope_['olSelectedFeature'])) {
            return 0;
        }
        this_.updateFeatureStyleFromForm_();
        this_.updateFeatureAttributes_();

        setTimeout(function () {
            this_.$scope_['oMap'].saveFeatures();
        });

    }, true);

    $scope.$watch('olSelectedFeature', function () {

        // Temps minimum entre deux mises à jour
        if ((olSelectedFeatureLastUpdate + olSelectedFeatureTimeout) > Date.now()) {
            return null;
        }
        olSelectedFeatureLastUpdate = Date.now();

        // bFormDisabled
        if (goog.isDefAndNotNull($scope['olSelectedFeature'])) {
            $scope['bFormDisabled'] = false;
        } else {
            $scope['bFormDisabled'] = true;
        }

        // sSelectedFeatureType
        if (goog.isDefAndNotNull($scope['olSelectedFeature'])) {
            switch (this_.$scope_['olSelectedFeature'].getGeometry().getType()) {
                case 'Point':
                case 'MultiPoint':
                    this_.$scope_['sSelectedFeatureType'] = 'Point';
                    break;
                case 'LineString':
                case 'MultiLineString':
                    this_.$scope_['sSelectedFeatureType'] = 'Line';
                    break;
                case 'Polygon':
                case 'MultiPolygon':
                    this_.$scope_['sSelectedFeatureType'] = 'Polygon';
                    break;
                default:
                    this_.$scope_['sSelectedFeatureType'] = null;
                    break;
            }
        }

        if (goog.isDefAndNotNull($scope['olSelectedFeature'])) {

            var oAttributes = goog.isDefAndNotNull($scope['olSelectedFeature'].get('attributes')) ? $scope['olSelectedFeature'].get('attributes') : {};
            $scope['featureForm']['attributes'] = oAttributes;

            // Mode custom_form
            if (goog.isDefAndNotNull($scope['oSubformScope'])) {

                // disableAttrs
                var sFieldName = $scope['field']['custom_form']['featureStructure']['field'];
                var sFieldTypes = $scope['field']['custom_form']['featureStructure']['types'];
                var oFeatureTypeSubform = angular.copy($scope['field']['custom_form']['form']);
                if (goog.isDefAndNotNull(oAttributes[sFieldName])) {
                    if (goog.isDefAndNotNull(sFieldTypes[oAttributes[sFieldName]])) {
                        if (goog.isArray(sFieldTypes[oAttributes[sFieldName]]['disableAttrs'])) {
                            var aDisableAttrs = sFieldTypes[oAttributes[sFieldName]]['disableAttrs'];
                            for (var i = 0; i < oFeatureTypeSubform['update']['rows'].length; i++) {
                                for (var ii = 0; ii < oFeatureTypeSubform['update']['rows'][i]['fields'].length; ii++) {
                                    if (aDisableAttrs.indexOf(oFeatureTypeSubform['update']['rows'][i]['fields'][ii]['name']) !== -1) {
                                        oFeatureTypeSubform['update']['rows'][i]['fields'][ii]['visible'] = false;
                                    }
                                }
                            }
                        }
                    }
                }

                // Valeurs
                $scope['oSubformScope']['loadSubForm']({
                    'sFormDefinitionName': 'update',
                    'oFormDefinition': oFeatureTypeSubform,
                    'oFormValues': {
                        'update': angular.copy(oAttributes)
                    },
                    'oProperties': $scope['oProperties']
                });
            }

            // Style
            var oStyle = $scope['olSelectedFeature'].getStyleAsJSON();

            if (goog.isDefAndNotNull(oStyle['draw'])) {
                $scope['featureForm']['draw'] = oStyle['draw'];
            }
            if (goog.isDefAndNotNull(oStyle['text'])) {
                $scope['featureForm']['text'] = oStyle['text'];
            }
        } else {
            // Mode dessin libre
            $scope['featureForm']['attributes'] = {};
            // Mode custom_form
            if (goog.isDefAndNotNull($scope['oSubformScope'])) {
                $scope['oSubformScope']['setFormValues']({
                    'update': {}
                });
            }
        }

        // Grille
        if (goog.isDefAndNotNull($scope['olSelectedFeature'])) {
            var featureIndex = $scope['aFeatures'].indexOf($scope['olSelectedFeature']);
            this_.$scope_['gridApi']['selection']['clearSelectedRows']();
            this_.$scope_['gridApi']['selection']['selectRowByVisibleIndex'](featureIndex);

        } else {
            $scope['clearSelectedFeature'](false);
        }
    });

    this.loadIconsList_();
};
formReader.module.controller('AppMapWorkbenchController', nsVitisComponent.MapWorkbenchController);

/**
 * Load the avaliable icons list
 * @private
 */
nsVitisComponent.MapWorkbenchController.prototype.loadIconsList_ = function () {
    this.$log_.info('MapWorkbenchController.loadIconsList_');

    var oGlyphs = ol.style.FontSymbol.prototype.defs.glyphs;
    var oFonts = ol.style.FontSymbol.prototype.defs.fonts;
    var aIcons = [];
    var sIcon;

    for (var key in oGlyphs) {
        if (goog.isDefAndNotNull(oGlyphs[key]['font'])) {
            if (goog.isDefAndNotNull(goog.isDefAndNotNull(oFonts[oGlyphs[key]['font']]))) {
                sIcon = '';
                if (goog.isDefAndNotNull(goog.isDefAndNotNull(oFonts[oGlyphs[key]['font']]['prefix']))) {
                    sIcon += oFonts[oGlyphs[key]['font']]['prefix'];
                    sIcon += ' ';
                }
                sIcon += key;
                aIcons.push({
                    'def': key,
                    'class': sIcon
                });
            }
        }
    }

    this.$scope_.$applyAsync(function (scope) {
        scope['aIcons'] = aIcons;
        scope['sSelectedIcon'] = aIcons[0]['class'];
        scope['featureForm']['draw']['symbol'] = aIcons[0]['def'];
    });
};

/**
 * Update the feature style
 * @private
 */
nsVitisComponent.MapWorkbenchController.prototype.updateFeatureStyleFromForm_ = function () {
    this.$log_.info('MapWorkbenchController.updateFeatureStyleFromForm_');
    this.$scope_['olSelectedFeature'].setStyleByJSON(this.$scope_['featureForm']);
};

/**
 * Update the feature attributes
 * @private
 */
nsVitisComponent.MapWorkbenchController.prototype.updateFeatureAttributes_ = function () {
    this.$log_.info('MapWorkbenchController.updateFeatureAttributes_');

    this.$scope_['olSelectedFeature'].set('attributes', this.$scope_['featureForm']['attributes']);
};

/**
 * Remplit la grille
 * @private
 */
nsVitisComponent.MapWorkbenchController.prototype.fillGridFromFeatures_ = function () {
    this.$log_.info("MapWorkbenchController.fillGridFromFeatures_");

    var aFeatures = this.$scope_['aFeatures'];
    var aGridData = [];
    var oAttributes;

    if (goog.isArray(aFeatures)) {
        for (var i = 0; i < aFeatures.length; i++) {
            if (goog.isDefAndNotNull(aFeatures[i].get('attributes'))) {
                oAttributes = aFeatures[i].get('attributes');
            } else {
                oAttributes = {};
            }
            oAttributes['map_workbench_index'] = angular.copy(i);
            aFeatures[i].set('attributes', oAttributes);
            aGridData.push(oAttributes);
        }
    }

    this.$scope_['gridOptions']['data'] = aGridData;
    this.$scope_['gridApi']['grid']['refresh']();
};

/**
 * Allow to add a feature
 * @export
 */
nsVitisComponent.MapWorkbenchController.prototype.showAddFeatureModal = function () {
    this.$log_.info("MapWorkbenchController.showAddFeatureModal");

    var this_ = this;
    var sFieldName = this.$scope_['field']['custom_form']['featureStructure']['field'];
    var oFeatureTypeSubform = angular.copy(this.$scope_['field']['custom_form']['form']);

    // Supprime les champs autre que sFieldName
    var formOk = false;
    for (var i = 0; i < oFeatureTypeSubform['update']['rows'].length; i++) {
        for (var ii = 0; ii < oFeatureTypeSubform['update']['rows'][i]['fields'].length; ii++) {
            if (oFeatureTypeSubform['update']['rows'][i]['fields'][ii]['name'] !== sFieldName) {
                oFeatureTypeSubform['update']['rows'][i]['fields'][ii]['visible'] = false;
                formOk = true;
            }
        }
    }

    // Vérifie que le formulaire soit bon
    if (!formOk) {
        console.error('Field ' + sFieldName + ' not present in the form');
        return null;
    }

    // Charge le sous formulire
    var oFeatureTypeSubformScope = angular.element('#map_workbench_' + this.$scope_['sFormUniqueName'] + '_feature_type_modal_subform').scope();
    oFeatureTypeSubformScope['loadSubForm']({
        'sFormDefinitionName': 'update',
        'oFormDefinition': oFeatureTypeSubform,
        'oFormValues': {
            'update': {}
        },
        'oProperties': this.$scope_['oProperties']
    });

    // Unselect la feature courrante
    this.$scope_['clearSelectedFeature']();

    // Affiche la modale
    setTimeout(function () {
        $('#map_workbench_' + this_.$scope_['sFormUniqueName'] + '_feature_type_modal').modal('show');
    });
};

/**
 * Allow to add a feature
 * @export
 */
nsVitisComponent.MapWorkbenchController.prototype.addFeature = function () {
    this.$log_.info("MapWorkbenchController.addFeature");

    var this_ = this;
    var formSrvc = angular.element(vitisApp.appMainDrtv).injector().get(["formSrvc"]);

    var sFieldName = this.$scope_['field']['custom_form']['featureStructure']['field'];
    var sFieldTypes = this.$scope_['field']['custom_form']['featureStructure']['types'];
    var oFeatureTypeSubform = angular.copy(this.$scope_['field']['custom_form']['form']);

    var oFeatureTypeSubformScope = angular.element('#map_workbench_' + this.$scope_['sFormUniqueName'] + '_feature_type_modal_subform').scope();

    var oFormTextValues = formSrvc['getFormData']('update', true, {
        'oFormDefinition': oFeatureTypeSubform,
        'oFormValues': goog.object.clone(oFeatureTypeSubformScope['oSubformValues'])
    });

    var sGeometryType = null;
    if (goog.isDefAndNotNull(oFormTextValues[sFieldName])) {
        if (goog.isDefAndNotNull(sFieldTypes[oFormTextValues[sFieldName]])) {
            if (goog.isDefAndNotNull(sFieldTypes[oFormTextValues[sFieldName]]['geometryType'])) {
                sGeometryType = sFieldTypes[oFormTextValues[sFieldName]]['geometryType'];
            } else {
                sGeometryType = 'all';
            }
        }
    }

    this.$scope_['oMap'].onceFeatureAdded(function (oFeature) {
        setTimeout(function () {
            // Ajout de l'attribut de type
            var oAttributes = oFeature.get('attributes');
            oAttributes = goog.isObject(oAttributes) ? oAttributes : {};
            oAttributes[sFieldName] = oFormTextValues[sFieldName];
            oFeature.set('attributes', oAttributes);
            // Set la style en fonction du type
            setTimeout(function () {
                this_.setStyleFromType_(oFeature);
            });
        });
    });

    // Désactive la possibilité d'ajouter des features    
    this.unactiveDrawInteractions_();

    // Ajoute la/les nouvelle interaction
    if (goog.isDefAndNotNull(sGeometryType)) {
        this.activeDrawInteraction_(sGeometryType);

        // cache la modale
        $('#map_workbench_' + this.$scope_['sFormUniqueName'] + '_feature_type_modal').modal('hide');
    } else {
        $.notify('Veuillez sélectionner un type', 'error');
    }
};

/**
 * Delete the selected feature from the map and the list
 * @export
 */
nsVitisComponent.MapWorkbenchController.prototype.deleteSelectedFeature = function () {
    this.$log_.info("MapWorkbenchController.deleteSelectedFeature");
    var this_ = this;

    if (!goog.isDefAndNotNull(this.$scope_['olSelectedFeature'])) {
        $.notify('Aucune géoétrie sélectionnée', 'error');
        return null;
    }

    bootbox.confirm("Supprimer définitivement la géométrie ?", function (result) {
        if (result) {
            this_.$scope_['oMap'].removeFeature(this_.$scope_['olSelectedFeature']);
            this_.$scope_['olSelectedFeature'] = null;
            setTimeout(function () {
                this_.$scope_['oMap'].saveFeatures();
            });
        }
    });

};

/**
 * Set the style of a feature from his type
 * @param {ol.Feature} oFeature
 */
nsVitisComponent.MapWorkbenchController.prototype.setStyleFromType_ = function (oFeature) {
    this.$log_.info("MapWorkbenchController.setStyleFromType_");

    if (!goog.isDefAndNotNull(oFeature)) {
        return null;
    }

    var sFieldName = this.$scope_['field']['custom_form']['featureStructure']['field'];
    var sFieldTypes = this.$scope_['field']['custom_form']['featureStructure']['types'];
    var oAttributes = oFeature.get('attributes');

    if (goog.isDefAndNotNull(oAttributes[sFieldName])) {
        if (goog.isDefAndNotNull(sFieldTypes[oAttributes[sFieldName]])) {
            if (goog.isDefAndNotNull(sFieldTypes[oAttributes[sFieldName]]['style'])) {

                // Remplace les balises par leurs valeurs respectives
                var oStyle = angular.copy(sFieldTypes[oAttributes[sFieldName]]['style']);
                for (var key1 in oStyle) {
                    for (var key2 in oStyle[key1]) {
                        if (goog.isString(oStyle[key1][key2])) {
                            // La chaine commence et se finit par {{ et }} ?
                            if (oStyle[key1][key2].substr(0, 2) === '{{' && oStyle[key1][key2].substr(-2, 2) === '}}') {

                                // Le contenu des balises est un des champs du formulaire ?                                
                                var sTag = oStyle[key1][key2].substr(2, oStyle[key1][key2].length - 4);
                                if (goog.isDefAndNotNull(oAttributes[sTag])) {
                                    oStyle[key1][key2] = oAttributes[sTag];
                                } else {
                                    oStyle[key1][key2] = '';
                                }
                            }
                        }
                    }
                }

                this.$scope_['featureForm']['draw'] = oStyle['draw'];
                this.$scope_['featureForm']['text'] = oStyle['text'];
            }
        }
    }
};

/**
 * Active a draw interactions on the map
 * @param {string} sGeometryType point|line|polygon|all
 */
nsVitisComponent.MapWorkbenchController.prototype.activeDrawInteraction_ = function (sGeometryType) {
    this.$log_.info("MapWorkbenchController.activeDrawInteraction_");

    switch (sGeometryType) {
        case 'all':
            this.$scope_['field']['map_options']['interactions']['point'] = true;
            this.$scope_['field']['map_options']['interactions']['line'] = true;
            this.$scope_['field']['map_options']['interactions']['polygon'] = true;
            break;
        case 'point':
            this.$scope_['field']['map_options']['interactions']['point'] = true;
            break;
        case 'line':
            this.$scope_['field']['map_options']['interactions']['line'] = true;
            break;
        case 'polygon':
            this.$scope_['field']['map_options']['interactions']['polygon'] = true;
            break;
        default:
            console.error('Bad sGeometryType: ', sGeometryType);
            return null;
            break;
    }

    this.$scope_['oMap']['setInteractions'](this.$scope_['field']['map_options']['interactions']);
    if (sGeometryType === 'point' || sGeometryType === 'line' || sGeometryType === 'polygon') {
        this.$scope_['oMap']['activeInteraction'](sGeometryType);
    }
};

/**
 * Unactive the draw interactions of the map
 */
nsVitisComponent.MapWorkbenchController.prototype.unactiveDrawInteractions_ = function () {
    this.$log_.info("MapWorkbenchController.unactiveDrawInteractions_");

    // Désactive la possibilité d'ajouter des features
    var aDrawInteractions = ['point', 'line', 'polygon', 'DP', 'DL', 'DPol', 'RA', 'remove_all', 'RO', 'remove'];
    for (var i = 0; i < aDrawInteractions.length; i++) {
        if (goog.isDefAndNotNull(this.$scope_['oMap'])) {
            if (goog.isDefAndNotNull(this.$scope_['oMap']['removeInteraction'])) {
                this.$scope_['oMap']['removeInteraction'](aDrawInteractions[i]);
            }
        }
        if (goog.isDefAndNotNull(this.$scope_['field']['map_options']['interactions'][aDrawInteractions[i]])) {
            this.$scope_['field']['map_options']['interactions'][aDrawInteractions[i]] = false;
        }
    }
//    if (goog.isDefAndNotNull(this.$scope_['oMap'])) {
//        if (goog.isDefAndNotNull(this.$scope_['oMap']['setInteractions'])) {
//            this.$scope_['oMap']['setInteractions'](this.$scope_['field']['map_options']['interactions']);
//        }
//    }
};