/* global formReader, goog, vitisApp, bootbox, deferred */

/**
 * @author: Armand Bahi
 * @Description: Form Reader service
 * @module formReader
 */
goog.provide('formReader.service.formReaderService');
goog.require('formReader');

/**
 * Form Reader Service
 * @param {service} $translate Translate service.
 * @param {angular.$rootScope} $rootScope Angular rootScope.
 * @param {service} $q Angular q service.
 * @param {angular.$log} $log Angular log service.
 * @param {service} $timeout
 * @param {service} envSrvc Paramètres d'environnement.
 * @ngInject
 * @constructor
 */
formReader.formReaderService = function ($translate, $rootScope, $q, $log, $timeout, envSrvc) {
    return {
        /**
         * reloadSelectField function.
         * Recharge les options du <select> (web service).
         * @param {object} oParentSelect Paramètres de definition du <select> parent.
         * @param {string} sFormDefinitionName Nom du formulaire.
         * @param {object} oFormValues Valeurs du formulaire.
         * @param {object} oFormDefinition Champs du formulaire.
         */
        "reloadSelectField": function (oParentSelect, sFormDefinitionName, oFormValues, oFormDefinition) {
            // Définition du <select> à recharger.
            var oChildSelect = this["getFormElementDefinition"](oParentSelect["child_select"], sFormDefinitionName, oFormDefinition);

            // Recharge le <select> enfant.
            if (goog.isDef(oChildSelect)) {
                var oParams = oChildSelect["web_service"]["parameters"];
                if (!goog.isDef(oParams))
                    oParams = {'filter': {}};

                // Si un filtre a déja été défini sous forme de chaine
                if (goog.isString(oParams['filter'])) {
                    oParams['filter'] = {
                        'stringFilter': oParams['filter']
                    };
                }

                if (!goog.isDef(oParams['filter']))
                    oParams['filter'] = {};

                oParams['filter'][oParentSelect["name"]] = oFormValues[sFormDefinitionName][oParentSelect["name"]]["selectedOption"]["value"];
                oChildSelect["web_service"]["parameters"] = oParams;

                this["setWebServiceSelectOptions"](oChildSelect, sFormDefinitionName, oFormValues, oChildSelect["web_service"]);
            }
        },
        /**
         * getFormElementDefinition function.
         * Retourne la définition d'un élément de form.
         * @param {string} sFormElementName Nom du champ de formulaire.
         * @param {string} sFormDefinitionName Nom du formulaire.
         * @param {object} oFormDefinition Champs du formulaire.
         * @return {object}
         */
        "getFormElementDefinition": function (sFormElementName, sFormDefinitionName, oFormDefinition) {
            var oFormDefinition = oFormDefinition[sFormDefinitionName];
            var i = 0, j, k;
            while (i < oFormDefinition["rows"].length) {
                j = 0;
                while (j < oFormDefinition["rows"][i]['fields'].length) {
                    if (oFormDefinition["rows"][i]['fields'][j]["name"] === sFormElementName)
                        return oFormDefinition["rows"][i]['fields'][j];
                    else if (typeof (oFormDefinition["rows"][i]['fields'][j]["buttons"]) != "undefined") {
                        k = 0;
                        while (k < oFormDefinition["rows"][i]['fields'][j]["buttons"].length) {
                            if (oFormDefinition["rows"][i]['fields'][j]["buttons"][k]["name"] === sFormElementName)
                                return oFormDefinition["rows"][i]['fields'][j]["buttons"][k];
                            k++;
                        }
                    }
                    j++;
                }
                i++;
            }
        },
        /**
         * getAllFormElementDefinition function.
         * Retourne la définition de tous les éléments de form.
         * @param {string} sFormDefinitionName Nom du formulaire.
         * @param {object} oFormDefinition Champs du formulaire.
         * @return {object}
         */
        "getAllFormElementDefinition": function (sFormDefinitionName, oFormDefinition) {
            var oFormDefinition = oFormDefinition[sFormDefinitionName];
            var oFormElementDefinition = [];
            var i = 0, j;
            while (i < oFormDefinition["rows"].length) {
                j = 0;
                while (j < oFormDefinition["rows"][i]['fields'].length) {
                    oFormElementDefinition.push(oFormDefinition["rows"][i]['fields'][j]);
                    j++;
                }
                i++;
            }
            return oFormElementDefinition;
        },
        /**
         * setWebServiceSelectOptions function.
         * Crée un objet de libellés et valeurs renvoyés par un web service et formatés pour un <select>.
         * @param {object} oFormElementDefinition Paramètres de definition du <select>.
         * @param {string} sFormDefinitionName Nom du formulaire.
         * @param {object} oFormValues Valeurs du formulaire.
         * @param {object} oWebService WebService à utiliser (peut provenir de oFormElementDefinition['web_service'], oFormElementDefinition['datasource'] etc..)
         * @param {object|undefined} oSelectedOption valeur initiale
         */
        "setWebServiceSelectOptions": function (oFormElementDefinition, sFormDefinitionName, oFormValues, oWebService, oSelectedOption) {

            var this_ = this;
            var sToken = this['sToken'];
            var oProperties = this['oProperties'];
            var oFormValues = oFormValues[sFormDefinitionName];
            var oParams = {};
            // Promesse pour le chargement des données du web service.
            var deferred = $q.defer();
            var promise = deferred.promise;
            var valueNumber;
            $rootScope["oFormPromises"][sFormDefinitionName].push(promise);
            //
            if (!goog.isDefAndNotNull(oFormElementDefinition)) {
                console.error('oFormElementDefinition not defined');
                return 0;
            }
            if (!goog.isDefAndNotNull(sFormDefinitionName)) {
                console.error('sFormDefinitionName not defined');
                return 0;
            }
            if (!goog.isDefAndNotNull(oFormValues)) {
                console.error('oFormValues not defined');
                return 0;
            }
            if (!goog.isDefAndNotNull(oWebService)) {
                console.error('oWebService not defined');
                return 0;
            }
            if (!goog.isDef(oFormValues)) {
                console.error('oFormValues[' + sFormDefinitionName + '] not defined');
                oFormValues = {};
            }

            // Charge les données du web service
            this['getWebServiceData'](oWebService).then(function (aData) {

                if (!goog.isDefAndNotNull(aData)) {
                    return null;
                }

                var j, oNotSelectedOptions, oSelectedOptions, sFromSelectName = "", sToSelectName;

                // Clés label / valeur du web service pour les <options>.
                var sLabelKey = oWebService["label_key"];
                var sValueKey = oWebService["id_key"];

                // Id des <options> à garder.
                var aSelectedOptions = [];
                //var selectedData = oFormValues[aRessourceId[1]];
                var selectedData = oFormValues[oFormElementDefinition["name"]];

                if (goog.isNumber(selectedData))
                    aSelectedOptions[0] = String(selectedData);
                else if (goog.isString(selectedData) && selectedData !== "") {
                    aSelectedOptions = selectedData.split("|");
                } else if (goog.isObject(selectedData) && goog.isArray(selectedData["options"])) {
                    for (var i in selectedData["options"]) {
                        aSelectedOptions.push(selectedData["options"][i]["value"]);
                    }
                }

                // "select" ou "double-select" ?
                j = 0;
                oNotSelectedOptions = {"selectedOption": [], "options": []};
                oSelectedOptions = {"selectedOption": [], "options": []};
                if (oFormElementDefinition["type"] === "double_select") {
                    sFromSelectName = oFormElementDefinition["name_from"];
                    sToSelectName = oFormElementDefinition["name_to"];
                } else
                    sToSelectName = oFormElementDefinition["name"];

                // Création du tableau pour le <select>.
                while (j < aData.length) {

                    if (!goog.isDef(aData[j][sLabelKey])) {
                        console.error('result.' + sLabelKey + ' does not exist, check the label_key param on the JSON file');
                        aData[j][sLabelKey] = null;
                        j++;
                        continue;
                    }
                    if (!goog.isDef(aData[j][sValueKey])) {
                        console.error('result.' + sValueKey + ' does not exist, check the id_key param on the JSON file');
                        aData[j][sValueKey] = null;
                        j++;
                        continue;
                    }
                    // Conversion d'une chaîne représentant une valeur numérique en type "number" (sauf pour les entiers commençants par "0").
                    valueNumber = Number(aData[j][sValueKey]);
                    if (!isNaN(valueNumber)) {
                        if (!parseInt(valueNumber, 10) === valueNumber || (parseInt(valueNumber, 10) === valueNumber && String(aData[j][sValueKey]).substr(0, 1) != 0))
                            aData[j][sValueKey] = valueNumber;
                    }
                    //
                    if (aSelectedOptions.indexOf(String(aData[j][sValueKey])) !== -1 || sFromSelectName === "")
                        oSelectedOptions["options"].push({"label": aData[j][sLabelKey], "value": aData[j][sValueKey]});
                    else
                        oNotSelectedOptions["options"].push({"label": aData[j][sLabelKey], "value": aData[j][sValueKey]});
                    j++;
                }

                // Définition de l'option sélectionnée sous forme d'objet
                if (sFromSelectName === "") {
                    if (goog.isDefAndNotNull(oSelectedOption)) {
                        oSelectedOptions["selectedOption"] = oSelectedOption;
                    } else {

                        // Listes déroulantes
                        if (oFormElementDefinition["type"] === 'select' || oFormElementDefinition["type"] === 'editable_select') {
                            // Si "select" : définition de l'option sélectionnée.
                            var selectedOptionValue = angular.copy(oFormValues[oFormElementDefinition["name"]]);
                            var selectedOptionLabel;
                            if (selectedOptionValue !== null && goog.isObject(selectedOptionValue)) {
                                selectedOptionValue = selectedOptionValue["selectedOption"]["value"];
                                selectedOptionLabel = oFormValues[oFormElementDefinition["name"]]["selectedOption"]["label"];
                            }
                            // Valeur par défaut si aucune valeur n'est définie
                            if (typeof (selectedOptionValue) === "undefined" && typeof (oFormElementDefinition["default_value"]) !== "undefined") {
                                selectedOptionValue = oFormElementDefinition["default_value"];
                                // La valeur par défaut est un appel de fonction ?
                                if (goog.isString(selectedOptionValue)) {
                                    if (this_["isFunctionCall"](selectedOptionValue)) {
                                        selectedOptionValue = this_['callFunction'](selectedOptionValue);
                                    }
                                }
                            }
                            // Si undefined, vaut null
                            selectedOptionValue = goog.isDef(selectedOptionValue) ? selectedOptionValue : null;
                            selectedOptionLabel = goog.isDef(selectedOptionLabel) ? selectedOptionLabel : "";
                            oSelectedOptions["selectedOption"] = {"value": selectedOptionValue, "label": selectedOptionLabel};
                        }

                        // Listes multiples
                        if (oFormElementDefinition["type"] === 'list') {

                            var selectedOptionValue = angular.copy(oFormValues[oFormElementDefinition["name"]]);
                            if (goog.isDefAndNotNull(selectedOptionValue)) {
                                if (goog.isArray(selectedOptionValue["selectedOption"])) {
                                    selectedOptionValue = selectedOptionValue["selectedOption"];
                                }
                            }

                            // Valeur par défaut si aucune valeur n'est définie
                            if (goog.isDefAndNotNull(selectedOptionValue) && typeof (oFormElementDefinition["default_value"]) !== "undefined") {
                                selectedOptionValue = oFormElementDefinition["default_value"];
                                // La valeur par défaut est un appel de fonction ?
                                if (goog.isString(selectedOptionValue)) {
                                    if (this_["isFunctionCall"](selectedOptionValue)) {
                                        selectedOptionValue = this_['callFunction'](selectedOptionValue);
                                    }
                                }
                            }

                            if (goog.isArray(selectedOptionValue) && selectedOptionValue.length > 0) {
                                oSelectedOptions["selectedOption"] = selectedOptionValue;
                            } else {
                                var aSelectedOptionValues = [];
                                if (goog.isString(selectedOptionValue)) {
                                    aSelectedOptionValues = selectedOptionValue.split('|');
                                }
                                if (goog.isNumber(selectedOptionValue)) {
                                    aSelectedOptionValues = [selectedOptionValue];
                                }
                                oSelectedOptions["selectedOption"] = [];
                                for (var i = 0; i < aSelectedOptionValues.length; i++) {
                                    for (var ii = 0; ii < oSelectedOptions["options"].length; ii++) {
                                        if (oSelectedOptions["options"][ii]['value'] == aSelectedOptionValues[i]) {
                                            // cas où la valeur soit un nombre
                                            if (goog.isNumber(oSelectedOptions["options"][ii]['value']) && goog.isString(aSelectedOptionValues[i])) {
                                                aSelectedOptionValues[i] = parseFloat(aSelectedOptionValues[i]);
                                            }
                                            // Push la valeur
                                            oSelectedOptions["selectedOption"].push({"value": aSelectedOptionValues[i], "label": oSelectedOptions["options"][ii]['label']});
                                        }
                                    }
                                }
                            }
                        }
                    }

                    // Options à ajouter à celles retournées par le web service ?
                    if (goog.isDefAndNotNull(oFormElementDefinition["options"])) {
                        var i = 0, aOptions, aOptionsKeys = [], aOptionsValues = [];
                        // Extraction des libellés et valeurs.
                        while (i < oFormElementDefinition["options"].length) {
                            aOptions = oFormElementDefinition["options"][i].split("|");
                            aOptionsKeys.push(aOptions[0]);
                            aOptionsValues.push(aOptions[1]);
                            i++;
                        }

                        // Traduction des libellés.
                        $translate(aOptionsKeys).then(function (translations) {
                            // Ajoute les options.
                            var aTranslationsKeys = Object.keys(translations);
                            i = 0;
                            while (i < aTranslationsKeys.length) {
                                oSelectedOptions["options"].unshift({"label": translations[aTranslationsKeys[i]], "value": aOptionsValues[i]});
                                i++;
                            }
                        });
                    }

                    // Ajoute une option vide si le champ n'est pas required et que la valeur "" n'est pas présente
                    if (oFormElementDefinition['required'] !== true && oFormElementDefinition['required'] !== 'true') {
                        var isEmptyValuePresent = false;
                        for (var i = 0; i < oSelectedOptions["options"].length; i++) {
                            if (oSelectedOptions["options"][i]['value'] === "")
                                isEmptyValuePresent = true;
                        }
                        if (!isEmptyValuePresent)
                            oSelectedOptions["options"].unshift({"label": "", "value": ""});
                    }
                }
                // Si "double-select" : définition des options du <select> "source".
                else {
                    oFormValues[sFromSelectName] = oNotSelectedOptions;
                }
                oFormValues[sToSelectName] = oSelectedOptions;

                // Sélectionne la valeur en cours si elle se trouve dans la liste des nouvelles options
                if (oFormElementDefinition["type"] !== 'editable_select') {

                    var selectedIsInOption = false;
                    for (var i = 0; i < oFormValues[sToSelectName]["options"].length; i++) {

                        // Prend la valeur des options au cas ou il y ait égalité
                        // Attention !!! le "==" à la place du "===" est obligatoire pour conserver l'égalité en cas de multiple chargement du formulaire
                        if (oSelectedOptions["selectedOption"]["value"] == oFormValues[sToSelectName]["options"][i]["value"]) {
                            oSelectedOptions["selectedOption"]["value"] = oFormValues[sToSelectName]["options"][i]["value"];
                            oSelectedOptions["selectedOption"]["label"] = oFormValues[sToSelectName]["options"][i]["label"];
                            selectedIsInOption = true;
                        }
                    }

                    // Supprime la valeur en cours si elle n'est pas dans les options
                    if (!selectedIsInOption) {
                        delete oFormValues[sToSelectName]["selectedOption"]["value"];
                        delete oFormValues[sToSelectName]["selectedOption"]["label"];
                    }
                }

                // Si aucune valeur n'existe, donne pour valeur "" (évite que Angular ajoute la valeur "?")
                if (!goog.isDefAndNotNull(oFormValues[sToSelectName])) {
                    oFormValues[sToSelectName] = {};
                }
                if (!goog.isDefAndNotNull(oFormValues[sToSelectName]["selectedOption"])) {
                    oFormValues[sToSelectName]["selectedOption"] = {};
                }
                if (!goog.isDefAndNotNull(oFormValues[sToSelectName]["selectedOption"]["value"])) {
                    oFormValues[sToSelectName]["selectedOption"]["value"] = "";
                    oFormValues[sToSelectName]["selectedOption"]["label"] = "";
                }
                if (oFormElementDefinition["type"] === "list") {
                    if (goog.isObject(oFormValues[sToSelectName]["selectedOption"]) && !goog.isArray(oFormValues[sToSelectName]["selectedOption"])) {
                        oFormValues[sToSelectName]["selectedOption"] = [oFormValues[sToSelectName]["selectedOption"]];
                    }
                }

                this['oFormValues'] = oFormValues;
                this['sFormDefinitionName'] = sFormDefinitionName;
                envSrvc["oFormDefaultValues"][sFormDefinitionName] = this_["copyFormValues"](oFormValues);

                if (typeof (oWebService["callback"]) !== "undefined") {
                    if (this_["isFunctionCall"](oWebService["callback"])) {
                        this_['callFunction'](oWebService["callback"]);
                    }
                }

                // Evènement de fin.
                $rootScope.$emit("webServiceSelectOptionsloaded");
                deferred.resolve();
            });
        },
        /**
         * getWebServiceData function.
         * @param {object} oWebService WebService à utiliser (peut provenir de oFormElementDefinition['web_service'], oFormElementDefinition['datasource'] etc..)
         */
        "getWebServiceData": function (oWebService) {

            var sToken = this['sToken'];
            var oProperties = this['oProperties'];
            var oWebService = oWebService;

            //var aRessourceId = oWebService["ressource_id"].split("/");
            var oParams = {};
            // Promesse pour le chargement des données du web service.
            var deferred = $q.defer();
            //
            if (!goog.isDefAndNotNull(oWebService)) {
                console.error('oWebService not defined');
                return 0;
            }
            // Paramètres à ajouter à la requête ?
            if (goog.isDef(oWebService["parameters"])) {
                oParams = oWebService["parameters"];

                // WebService type business object ?
                if (oWebService['dataType'] === 'businessObject') {
                    if (goog.isDefAndNotNull(oParams['attributs'])) {
                        delete oParams['attributs'];
                    }
                }

                // Filtres à ajouter à la requête ?
                if (goog.isDef(oWebService["parameters"]['filter']) || goog.isDef(oWebService["parameters"]['filter_extension'])) {

                    oWebService["parameters"]['filter'] = this["parseFilter"](oWebService["parameters"]['filter']);

                    if (oWebService["parameters"]['filter'] === "") {
                        delete oWebService["parameters"]['filter'];
                    }

                }
            }
            // Charge la liste du menu déroulant.
            ajaxRequest({
                "method": "POST",
                "url": oProperties["web_server_name"] + '/' + oProperties["services_alias"] + '/' + oWebService["ressource_id"],
                "data": oParams,
                "scope": $rootScope,
                "headers": {
                    'X-HTTP-Method-Override': 'GET',
                    'Accept': 'application/x-vm-json'
                },
                "success": function (response) {
                    // Remet le filtre origiel
                    if (goog.isDefAndNotNull(oWebService['parameters']))
                        oWebService['parameters']['filter'] = oWebService['parameters']['original_filter'];

                    // Récupération de la donnée
                    var data = response['data'];
                    data['data'] = goog.isDef(data['data']) ? data['data'] : [];

                    // Met à jour la liste des enregistrements.
                    if (data["status"] === 1) {

                        // Extraction des données.
                        var aData;
                        if (goog.isDefAndNotNull(oWebService['subObject'])) {
                            aData = [];
                            for (var i = 0; i < data['data'].length; i++) {
                                if (goog.isDefAndNotNull(data['data'][i][oWebService['subObject']]))
                                    aData.push(data['data'][i][oWebService['subObject']]);
                            }
                        } else {
                            aData = goog.isDef(data['data']) ? data['data'] : [];
                        }
                    }

                    deferred.resolve(aData);
                }
            });
            return deferred.promise;
        },
        /**
         * emptySelectOptions function.
         * Vide un select
         * @param {object} oFormElementDefinition Paramètres de definition du <select>.
         * @param {string} sFormDefinitionName Nom du formulaire.
         * @param {object} oFormValues Valeurs du formulaire.
         */
        "emptySelectOptions": function (oFormElementDefinition, sFormDefinitionName, oFormValues) {
            var sFromSelectName, sToSelectName;
            if (oFormElementDefinition["type"] === "double_select") {
                sFromSelectName = oFormElementDefinition["name_from"];
                sToSelectName = oFormElementDefinition["name_to"];
            } else
                sToSelectName = oFormElementDefinition["name"];

            if (goog.isDefAndNotNull(oFormValues[sFormDefinitionName])) {
                if (goog.isDefAndNotNull(oFormValues[sFormDefinitionName][sToSelectName])) {
                    if (goog.isDefAndNotNull(oFormValues[sFormDefinitionName][sToSelectName]['options'])) {
                        oFormValues[sFormDefinitionName][sToSelectName] = "";
                    }
                }
            }

            this['oFormValues'] = oFormValues;

            if (goog.isDefAndNotNull(envSrvc["oFormDefaultValues"])) {
                if (goog.isDefAndNotNull(envSrvc["oFormDefaultValues"][sFormDefinitionName])) {
                    envSrvc["oFormDefaultValues"][sFormDefinitionName] = this["copyFormValues"](oFormValues);
                }
            }
        },
        /**
         * Call the function if defined on this or $rootScope
         * @param {string} sFunction
         */
        "callFunction": function (sFunction) {

            if (!goog.isDefAndNotNull(sFunction)) {
                console.error('callFunction: no function defined');
                return 0;
            }
            if (!sFunction.length > 0) {
                console.error('callFunction: function === ""');
                return 0;
            }

            // Si il y a un appel à une fonction dans les paramètres de la fonction,
            // execute l'appel et renplace le résultat ex: "concat(getProperty('services_alias'), '_test')"
            var sParams = '';
            var aParams = [];
            if (sFunction.indexOf('(') !== -1) {
                sParams = sFunction.substr(sFunction.indexOf('('));
                sParams = sParams.substr(1, sParams.length - 2);
                if (sParams.length > 0) {
                    aParams = sParams.split(',');
                }
                for (var i = 0; i < aParams.length; i++) {
                    if (goog.isString(aParams[i])) {
                        aParams[i] = aParams[i].trim();
                    }
                }
            }
            for (var i = 0; i < aParams.length; i++) {
                if (this['isFunctionCall'](aParams[i])) {
                    aParams[i] = "'" + this['callFunction'](aParams[i]) + "'";
                }
            }
            if (aParams.length > 0) {
                sFunction = sFunction.replace(sParams, aParams.join(','));
            }

            var sFunctionName = sFunction.substr(0, sFunction.indexOf("("));
            var returnMessage;
            this['eventsContainer'] = this['oFormEventsContainer'];
            this['rootScope_'] = $rootScope;
            // Test sFunction tel quel
            if (typeof (eval("window." + sFunctionName)) == "function") {
                try {
                    $log.log(sFunction);
                    returnMessage = eval(sFunction);
                } catch (e) {
                    if (this['oProperties']['debug_mode'] === true)
                        console.error(e);
                }
            }
            // Test sFunction sur se qui a été défini dans eventsContainer
            else if (typeof (this["eventsContainer"]) != "undefined" && typeof (this["eventsContainer"][sFunctionName]) == "function") {
                try {
                    $log.log('eventsContainer.' + sFunction);
                    returnMessage = eval('this.eventsContainer.' + sFunction);
                } catch (e) {
                    if (this['oProperties']['debug_mode'] === true)
                        console.error(e);
                }
            }
            // Test sFunction sur this
            else if (typeof (this[sFunctionName]) == "function") {
                try {
                    $log.log('this.' + sFunction);
                    returnMessage = eval('this.' + sFunction);
                } catch (e) {
                    if (this['oProperties']['debug_mode'] === true)
                        console.error(e);
                }
            }
            // sFunction sur $rootScope (sans test pour voir les eventuelles erreurs)
            else if (typeof (this["rootScope_"]) != "undefined" && typeof (this["rootScope_"][sFunctionName]) == "function") {
                try {
                    $log.log('$rootScope.' + sFunction);
                    returnMessage = eval('this.rootScope_.' + sFunction);
                } catch (e) {
                    if (this['oProperties']['debug_mode'] === true)
                        console.error(e);
                }

            }
            return returnMessage;
        },
        /**
         * isFunctionCall function.
         * La chaine de caractère passée est un appel de fonction ?
         * @param {string} sString Chaine de caractère.
         * @param {boolean} bCheckFunction Vérification de l'existence de la fonction (vrai par défaut).
         * @return {boolean}
         */
        "isFunctionCall": function (sString, bCheckFunction) {
            if (typeof (bCheckFunction) == "undefined")
                bCheckFunction = true;
            var bReturn = false;
            if (typeof (sString) === "string") {
                aMatchResult = sString.match(/(.+)\((.*)\)/);
                if (aMatchResult !== null) {
                    if (bCheckFunction) {
                        var sFunctionName = sString.substr(0, sString.indexOf("("));
                        var returnMessage;
                        if (typeof (sFunctionName) == "function")
                            bReturn = true;
                        else if (typeof (this['oFormEventsContainer']) != "undefined" && typeof (this['oFormEventsContainer'][sFunctionName]) == "function")
                            bReturn = true;
                        else if (typeof (this[sFunctionName]) == "function")
                            bReturn = true;
                        else if (typeof ($rootScope[sFunctionName]) == "function")
                            bReturn = true;
                        if (!bReturn)
                            $log.warn("Function not found : " + sString);
                    } else
                        bReturn = true;
                }
            }

            return bReturn;
        },
        /**
         * isTernaryString function.
         * La chaine de caractère passée est une expression ternaire ?
         * @param {string} sString Chaine de caractère.
         * @return {boolean}
         */
        "isTernaryString": function (sString) {
            var ter = /([^?]{1,})\?([^:]{1,}):([^;]{1,})/gm;
            if (goog.isString(sString)) {
                return sString.match(ter) ? true : false;
            } else {
                return false;
            }
        },
        /**
         * selectFirstOption function.
         * Sélectionne la 1ere option d'un <select>.
         * @param {string} sFormElementName Nom du <select>.
         * @param {object} oFormValues Valeurs du formulaire.
         * @param {string} sFormDefinitionName Nom du formulaire.
         */
        "selectFirstOption": function (sFormElementName, oFormValues, sFormDefinitionName) {
            oFormValues = goog.isDef(oFormValues) ? oFormValues : this['oFormValues'];
            sFormDefinitionName = goog.isDef(sFormDefinitionName) ? sFormDefinitionName : this['sFormDefinitionName'];
            oFormValues = oFormValues[sFormDefinitionName];
            if (oFormValues[sFormElementName]["options"].length > 0)
                oFormValues[sFormElementName]["selectedOption"]["value"] = oFormValues[sFormElementName]["options"][0]["value"];
        },
        /**
         * setSelectOptions function.
         * Crée un objet de libellés et valeurs pour un <select>.
         * @param {object} oFormElementDefinition Paramètres de definition du <select>.
         * @param {string} sFormDefinitionName Nom du formulaire.
         * @param {object} oFormValues Valeurs du formulaire.
         * @param {array} aOptions Options à utiliser
         */
        "setSelectOptions": function (oFormElementDefinition, sFormDefinitionName, oFormValues, aOptions) {
            $log.log('setSelectOptions');

            var getFormatedOptions = function (aKeys, aValues) {

                var aOptions = [];

                if (!goog.isArray(aKeys)) {
                    console.error('getFormatedOptions: aKeys is not an array', aKeys);
                    return aOptions;
                }
                if (!goog.isArray(aValues)) {
                    console.error('getFormatedOptions: aKeys is not an array', aValues);
                    return aOptions;
                }
                if (aKeys.length !== aValues.length) {
                    console.error('getFormatedOptions: aKeys.length !== aValues.length', aKeys, aValues);
                    return aOptions;
                }

                for (var i = 0; i < aKeys.length; i++) {
                    aOptions.push({'label': aKeys[i], 'value': aValues[i]});
                }

                return aOptions;
            };

            var i = 0, sLabel, sValue, aOption;
            var oSelectOptions = {'selectedOption': {'value': ''}, 'options': []};
            var sSelectName, sFromSelectName, sToSelectName;

            aOptions = goog.isDef(aOptions) ? aOptions : oFormElementDefinition['options'];

            var aOptionsKeys = [];
            var aOptionsValues = [];

            if (goog.isString(aOptions))
                aOptions = aOptions.split(',');

            while (i < aOptions.length) {
                aOption = aOptions[i].split('|');
                sLabel = aOption[0];
                aOptionsKeys.push(sLabel);
                if (aOption.length > 1)
                    sValue = aOption[1];
                else
                    sValue = sLabel;
                aOptionsValues.push(sValue);
                i++;
            }

            // Valeur par défaut si aucune valeur n'est définie.
            var selectValue = oFormValues[sFormDefinitionName][oFormElementDefinition['name']];

            // Si il s'agit d'une liste, il faut parser les "|"
            if (oFormElementDefinition['type'] === 'list' && goog.isString(selectValue)) {
                var aSelectedValues = selectValue.split('|');
                if (aSelectedValues.length > 0) {
                    var aSelectedOptions = [];
                    for (var i = 0; i < aSelectedValues.length; i++) {
                        aSelectedOptions.push({
                            'label': '',
                            'value': aSelectedValues[i]
                        });
                    }
                    selectValue = {
                        'selectedOption': aSelectedOptions
                    };
                }
            }

            if (typeof (selectValue) === 'undefined' && typeof (oFormElementDefinition['default_value']) !== 'undefined') {

                selectValue = oFormElementDefinition['default_value'];

                // La valeur par défaut est un appel de fonction ?
                if (typeof (selectValue) === 'string')
                    if (this['isFunctionCall'](selectValue)) {
                        selectValue = this['callFunction'](selectValue);
                    }
            }

            // Utilisé pour liste simple
            if (goog.isDefAndNotNull(selectValue)) {
                if (goog.isDefAndNotNull(selectValue['selectedOption'])) {
                    if (goog.isDefAndNotNull(selectValue['selectedOption']['value']))
                        oSelectOptions = selectValue;
                    if (goog.isArray(selectValue['selectedOption']))
                        oSelectOptions = selectValue;
                } else
                    oSelectOptions['selectedOption']['value'] = String(selectValue);
            }

            // Traduction des libellés.
            $translate(aOptionsKeys).then(function (translations) {
                // Traduit les clés
                for (var i = 0; i < aOptionsKeys.length; i++) {
                    // Pas de traduction des labels numériques sinon bug.
                    if (isNaN(aOptionsKeys[i]))
                        aOptionsKeys[i] = translations[aOptionsKeys[i]];
                }

                // Ajoute les options.
                oSelectOptions['options'] = getFormatedOptions(aOptionsKeys, aOptionsValues);

                // Sauve la structure du <select>.
                if (oFormElementDefinition['type'] === 'double_select') {

                    sFromSelectName = oFormElementDefinition['name_from'];
                    sToSelectName = oFormElementDefinition['name_to'];

                    var aFromOptionsKeys = angular.copy(aOptionsKeys);
                    var aFromOptionsValues = angular.copy(aOptionsValues);
                    var aToOptionsKeys = [];
                    var aToOptionsValues = [];
                    var aValues = [];
                    if (goog.isDefAndNotNull(selectValue)) {
                        aValues = selectValue.split("|");
                    }

                    // Prend les valeurs sélectionnées dans ToOptions et les valeurs restantes dans FromOptions
                    for (var i = 0; i < aValues.length; i++) {
                        var index = aFromOptionsValues.indexOf(aValues[i]);
                        // Ajoute des valeurs à garder (à droite)
                        aToOptionsValues.unshift(aFromOptionsValues[index]);
                        aToOptionsKeys.unshift(aFromOptionsKeys[index]);
                        // Supprime les valeurs à ne pas garder (à gauche)
                        aFromOptionsValues.splice(index, 1);
                        aFromOptionsKeys.splice(index, 1);
                    }

                    var oFromSelectOptions = {
                        'options': getFormatedOptions(aFromOptionsKeys, aFromOptionsValues)
                    };
                    var oToSelectOptions = {
                        'options': getFormatedOptions(aToOptionsKeys, aToOptionsValues)
                    };

                    oFormValues[sFormDefinitionName][sFromSelectName] = oFromSelectOptions;
                    oFormValues[sFormDefinitionName][sToSelectName] = oToSelectOptions;

                } else {
                    sSelectName = oFormElementDefinition['name'];
                    oFormValues[sFormDefinitionName][sSelectName] = oSelectOptions;
                }
                //
                if (typeof (envSrvc['oFormDefaultValues'][sFormDefinitionName]) === 'undefined')
                    envSrvc['oFormDefaultValues'][sFormDefinitionName] = {};
                if (typeof (envSrvc["oFormDefaultValues"][sFormDefinitionName][sSelectName]) == "undefined")
                    envSrvc['oFormDefaultValues'][sFormDefinitionName][sSelectName] = angular.copy(oSelectOptions);
            });
        },
        /**
         * selectOptionAfterLoad function.
         * Sélectionne une option dans un <select>.
         * @param {object} oFormElementDefinition Paramètres de definition du <select>.
         * @param {string} sFormDefinitionName Nom du formulaire.
         * @param {object} oFormValues Valeurs du formulaire.
         * @param {string} sFieldName Nom du <select>
         * @param {mixed}  value valeur de l'option à sélectionner
         **/
        "selectOptionAfterLoad": function (oFormValues, oFormElementDefinition, sFormDefinitionName, sFieldName, value) {
            if (oFormElementDefinition["type"] === "double_select") {
                //voir si besoin
                console.log(value);
            } else if (oFormElementDefinition["type"] === "select") {
                var opt = oFormValues[sFormDefinitionName][sFieldName].options;
                for (var i = 0; i < opt.length; i++) {
                    if (opt[i].value === value) {
                        oFormValues[sFormDefinitionName][sFieldName].selectedOption = opt[i];
                        break;
                    }
                }

            } else if (oFormElementDefinition["type"] === "list") {
                //pareil avec un tableau de selected options
            } else {
                console.error("this element is not a select");
            }


        },
        /**
         * extractWebServiceData function.
         * Extrait les données renvoyées par un web service.
         * @param {string} sController Nom du contrôleur du web service.
         * @param {object} oWebServiceData Objet retourné par le web service.
         * @return {array}
         **/
        "extractWebServiceData": function (sController, oWebServiceData) {
            var sKey = sController;
            var aData = [];
            if (typeof (oWebServiceData[sKey]) !== "undefined") {
                //
                if (Array.isArray(oWebServiceData[sKey])) {
                    if (oWebServiceData[sKey].length > 0)
                        aData = oWebServiceData[sKey];
                } else
                    aData = oWebServiceData[sKey];
            }
            return aData;
        },
        /**
         * parseFilter function
         * récupère le filtre en chaine de caractères
         * @param {string|object} filter
         * @return {string} filter
         */
        "parseFilter": function (filter) {

            if (goog.isObject(filter)) {

                var isJSONFilter = false;
                if (goog.isDefAndNotNull(filter['operators']) && goog.isDefAndNotNull(filter['relation'])) {
                    isJSONFilter = true;
                }
                if (goog.isDefAndNotNull(filter['column']) && goog.isDefAndNotNull(filter['compare_operator']) && goog.isDefAndNotNull(filter['value'])) {
                    isJSONFilter = true;
                }

                if (!isJSONFilter) {
                    var aFilters = [];
                    var sFilterValue;
                    for (var key in filter) {
                        if (key === 'stringFilter') {
                            aFilters.push(filter[key]);
                        } else {
                            sFilterValue = filter[key];
                            if (sFilterValue !== "") {
                                if (this["isFunctionCall"](sFilterValue)) {
                                    sFilterValue = this['callFunction'](sFilterValue);
                                }
                                if (!goog.isDefAndNotNull(sFilterValue)) {
                                    sFilterValue = 'is null';
                                }
                                if (goog.isDef(sFilterValue)) {

                                    var sCompareOperator = '=';

                                    if (goog.isString(sFilterValue)) {
                                        // NOT NULL
                                        if (sFilterValue.toLowerCase().trim() === 'not null') {
                                            sCompareOperator = '!=';
                                            sFilterValue = 'NULL';
                                        }
                                        // IS NULL
                                        if (sFilterValue.toLowerCase().trim() === 'is null') {
                                            sCompareOperator = 'IS NULL';
                                            sFilterValue = '';
                                        }
                                        // IS NOT NULL
                                        if (sFilterValue.toLowerCase().trim() === 'is not null') {
                                            sCompareOperator = 'IS NOT NULL';
                                            sFilterValue = '';
                                        }
                                    }

                                    aFilters.push({
                                        "column": key,
                                        "compare_operator": sCompareOperator,
                                        "value": sFilterValue
                                    });
                                }
                            }
                        }
                    }
                    // Formatage des filtres ou suppression du paramètre.
                    if (aFilters.length > 0) {
                        filter = {
                            "relation": "AND",
                            "operators": []
                        };
                        for (var i = 0; i < aFilters.length; i++) {
                            filter['operators'].push(aFilters[i]);
                        }
                    } else {
                        filter = "";
                    }
                }
            }

            return filter;
        },
        /**
         * setWebServiceTags function.
         * Crée un objet de libellés et valeurs renvoyés par un web service et formatés pour un champ de "tags".
         * @param {object} oFormElementDefinition Paramètres de definition du <select>.
         * @param {string} sFormDefinitionName Nom du formulaire.
         * @param {object} oFormValues Valeurs du formulaire.
         */
        "setWebServiceTags": function (oFormElementDefinition, sFormDefinitionName, oFormValues) {
            var oWebService = angular.copy(oFormElementDefinition["web_service"]);
            var aRessourceId = oWebService["ressource_id"].split("/");
            var oFormValues = oFormValues[sFormDefinitionName], selectedOptionValue;
            var oFormSrvc = this;
            var oParams = {};
            // Paramètres à ajouter à la requête ?
            if (typeof (oWebService["parameters"]) !== "undefined") {
                oParams = oWebService["parameters"];
                // Filtres à ajouter à la requête ?
                if (typeof (oWebService["parameters"]['filter']) !== "undefined") {

                    oWebService["parameters"]['filter'] = this["parseFilter"](oWebService["parameters"]['filter']);
                    if (oWebService["parameters"]['filter'] === "")
                        delete oWebService["parameters"]['filter'];
                }
            }
            // Charge la liste du menu déroulant.
            ajaxRequest({
                "method": "POST",
                "url": this['oProperties']["web_server_name"] + '/' + this['oProperties']["services_alias"] + '/' + oWebService["ressource_id"],
                "headers": {
                    "X-HTTP-Method-Override": "GET"
                },
                "data": oParams,
                "scope": $rootScope,
                "success": function (response) {
                    var data = response['data'];
                    var j, oNotSelectedOptions, oSelectedOptions, sFromSelectName = "", sToSelectName;
                    // Met à jour la liste des enregistrements.
                    if (data["status"] === 1) {
                        // Extraction des données.
                        var aData = oFormSrvc["extractWebServiceData"](aRessourceId[1], data);

                        // Tags sélectionnés.
                        var aTags = [];
                        if (typeof (oFormValues[oFormElementDefinition["name"]]) === "string")
                            aTags = oFormValues[oFormElementDefinition["name"]].split(",");

                        // Clés label / valeur du web service pour les <options>.
                        var sLabelKey = oWebService["label_key"];
                        var sValueKey = oWebService["id_key"];

                        // Tags disponibles.
                        var i = 0;
                        var aAvailableTags = [];
                        while (i < aData.length) {
                            if (aTags.indexOf(aData[i][sValueKey]) === -1)
                                aAvailableTags.push(aData[i][sValueKey]);
                            i++;
                        }
                        oFormValues[oFormElementDefinition[["name_available_tags"]]] = aAvailableTags.join(",");
                        if (typeof (envSrvc["oFormDefaultValues"][sFormDefinitionName][oFormElementDefinition[["name_available_tags"]]]) == "undefined")
                            envSrvc["oFormDefaultValues"][sFormDefinitionName][oFormElementDefinition[["name_available_tags"]]] = angular.copy(oFormValues[oFormElementDefinition[["name_available_tags"]]]);

                        // Lance la création des tags.
                        $timeout(function () {
                            $rootScope.$emit("webServiceTagsloaded", oFormElementDefinition);
                        });
                    }
                }
            });
        },
        /**
         * checkFormModifications function.
         * Vérifie si un formulaire a été modifié et si oui affiche un warning.
         * @param {string} sFormDefinitionName Nom du formulaire.
         * @return {promise}
         */
        "checkFormModifications": function (sFormDefinitionName) {
            var deferred = $q.defer();
            var promise = deferred.promise;
            // Vérifie si un formulaire a été modifié.
            if ((envSrvc["sMode"] == "insert" || envSrvc["sMode"] == "update") && typeof (sFormDefinitionName) != "undefined") {
                var oFormDefinition = envSrvc["oFormDefinition"][sFormDefinitionName];

                if (typeof (oFormDefinition) != "undefined" && typeof (oFormDefinition["event"]) != "undefined") {
                    var sFormName = oFormDefinition["name"];
                    var oFormElement = angular.element("form[name='" + sFormName + "']");
                    if (oFormElement.length > 0) {
                        var formScope = oFormElement.scope();
                        if (formScope[sFormName]["$dirty"] && !formScope[sFormName]["$submitted"]) {
                            // Paramètres de la fenêtre modale.
                            var oOptions = {
                                "className": "modal-warning",
                                "message": "FORM_MODIFIED_WARNING",
                                "callback": function (bResponse) {
                                    if (bResponse)
                                        deferred.resolve();
                                }
                            };
                            // Affichage de la fenêtre modale.
                            formScope.$root["modalWindow"]("confirm", "FORM_MODIFIED_WARNING_TITLE", oOptions);
                        } else
                            deferred.resolve();
                    } else
                        deferred.resolve();
                } else
                    deferred.resolve();
            } else
                deferred.resolve();
            return promise;
        },
        /**
         * checkStudioFormModifications function.
         * Vérifie si un formulaire du studio a été modifié et si oui et que ses modifications n'ont pas été modifiées affiche un warning.
         * @param {string} sSelectedObjectName
         * @return {promise}
         */
        "checkStudioFormModifications": function (sSelectedObjectName) {
            var deferred = $q.defer();
            var promise = deferred.promise;
            // Vérifie l'application en cour
            var AppName = oVFB.getApplication();
            if (AppName === "vmap") {
                var Update = oVFB.getUpdate();
                if (Update === true) {

                    // Affichage de la Modal d'Alert/Confirm
                    var oOptions = {
                        "className": "modal-warning",
                        "message": "FORM_MODIFIED_WARNING",
                        "callback": function (bResponse) {
                            if (bResponse)
                                deferred.resolve();
                        }
                    };
                    $rootScope["modalWindow"]("confirm", "FORM_MODIFIED_WARNING_TITLE", oOptions);
                } else {
                    deferred.resolve();
                }
            } else {
                deferred.resolve();
            }
            //oVFB.Update = false;
            return promise;
        },
        /**
         * getFormValueFromObject function.
         * Retourne la valeur d'un champ de formulaire contenu dans un objet.
         * @param {object} oFormValues Valeurs du formulaire.
         * @param {string} sFormFieldName Nom du champ de formulaire.
         **/
        "getFormValueFromObject": function (oFormValues, sFormFieldName) {
            $log.info("getFormValueFromObject");
            var sReturn = this["copyFormValues"](oFormValues);
            var aFormFieldName = sFormFieldName.split(".");
            aFormFieldName.forEach(function (sParameter) {
                sReturn = sReturn[sParameter];
            })
            return sReturn;
        },
        /**
         * Copy the form values
         * @param {object} oFormValues
         * @returns {object}
         */
        "copyFormValues": function (oFormValues) {
            $log.info("copyFormValues");
            var oReturn = {};
            try {
                oReturn = angular.copy(oFormValues);
            } catch (e) {
                for (var key in oFormValues) {
                    try {
                        oReturn[key] = angular.copy(oFormValues[key]);
                    } catch (e) {
                        if (goog.isObject(oFormValues[key])) {
                            oReturn[key] = {};
                            for (var key2 in oFormValues[key]) {
                                try {
                                    oReturn[key][key2] = angular.copy(oFormValues[key][key2]);
                                } catch (e) {

                                }
                            }
                        }
                    }
                }
            }

            return oReturn;
        },
        /**
         * Get a form value
         * @param {string} sName
         * @returns {string}
         */
        "getFormValue": function (sName) {
            $log.info("getFormValue");

            if (!goog.isString(sName)) {
                return null;
            }

            sName = sName.trim();

            var oFormValues = this['oFormValues'][this['sFormDefinitionName']];

            if (!goog.isDefAndNotNull(oFormValues)) {
                return null;
            }

            var sValue = oFormValues[sName.trim()];

            if (goog.isDefAndNotNull(sValue['selectedOption'])) {
                if (goog.isDefAndNotNull(sValue['selectedOption']['value'])) {
                    sValue = sValue['selectedOption']['value'];
                }
            }

            return sValue;

        },
        /**
         * Concats all the arguments passed
         */
        "concat": function () {
            $log.info("concat");

            var sResult = '';
            for (var i = 0; i < arguments.length; i++) {
                if (goog.isString(arguments[i])) {
                    sResult += arguments[i];
                }
            }
            return sResult;
        },
        /**
         * Get the business object definition
         * @param {string} sBusinessObject
         * @returns {promise}
         */
        "getBusinessObjectDef": function (sBusinessObject) {
            var deferred = $q.defer();
            showAjaxLoader();
            ajaxRequest({
                "method": "GET",
                "url": this['oProperties']["web_server_name"] + "/" + this['oProperties']["services_alias"] + "/vmap/businessobjects/" + sBusinessObject,
                "scope": $rootScope,
                'headers': {
                    'Accept': 'application/x-vm-json'
                },
                "success": function (response) {
                    hideAjaxLoader();
                    if (!goog.isDef(response['data'])) {
                        $translate('NO_VALUES_RETURNED').then(function (sTranslation) {
                            bootbox.alert(sTranslation);
                        });
                        return 0;
                    }
                    if (!goog.isDef(response['data']['data'])) {
                        return 0;
                    }
                    if (!goog.isDef(response['data']['data'][0])) {
                        return 0;
                    }
                    if (!goog.isDef(response['data']['data'][0]['business_object_id'])) {
                        return 0;
                    }
                    if (goog.isDef(response['data']['errorMessage'])) {
                        bootbox.alert(response['data']['errorMessage']);
                        return 0;
                    }

                    deferred.resolve(response['data']['data'][0]);
                },
                "error": function (response) {
                    hideAjaxLoader();
                    console.error(response);
                    bootbox.alert(response);
                }
            });

            return deferred.promise;
        },
        /**
         * Show a modal witch contains a subform
         * @param {string} sModalId
         * @param {object} oSubformScope
         * @param {object} oSubformDefinition
         * @param {object} oSubformDefinitionName
         * @param {object} oSubformValues
         */
        "showModalSubform": function (sModalId, oSubformScope, oSubformDefinition, oSubformDefinitionName, oSubformValues) {
            $log.log("showModalSubform");

            var sToken = this['sToken'];
            var oProperties = this['oProperties'];

            if (goog.isDefAndNotNull(oSubformScope['loadSubForm'])) {

                oSubformScope['loadSubForm']({
                    'sFormDefinitionName': oSubformDefinitionName,
                    'oFormDefinition': oSubformDefinition,
                    'oFormValues': oSubformValues,
                    'oProperties': oProperties,
                    'sToken': sToken
                });

                $('#' + sModalId).modal('show');
            }
        },
        /**
         * Update a business object element
         * @param {string} sBusinessObject
         * @param {object} oObject
         * @param {object} oSubformDefinition
         * @param {string} sSubformDefinitionName
         * @param {object} oFormValuesResulted
         * @param {funciton} successCallback
         */
        "updateBoElement": function (sBusinessObject, oObject, oSubformDefinition, sSubformDefinitionName, oFormValuesResulted, successCallback) {
            var oProperties = this['oProperties'];
            var envSrvc = angular.element(vitisApp.appMainDrtv).injector().get(["envSrvc"]);
            var formSrvc = angular.element(vitisApp.appMainDrtv).injector().get(["formSrvc"]);
            var aParamsToSave = [oObject['bo_id_field']];

            envSrvc["oFormValues"] = oFormValuesResulted;
            envSrvc["oFormDefinition"] = oSubformDefinition;
            envSrvc["sFormDefinitionName"] = sSubformDefinitionName;

            // Récupère les valeurs à envoyer
            var oValues = formSrvc['getFormData'](sSubformDefinitionName, true);
            for (var i = 0; i < aParamsToSave.length; i++) {
                if (goog.isDefAndNotNull(oFormValuesResulted[sSubformDefinitionName][aParamsToSave[i]])) {
                    oValues[aParamsToSave[i]] = oFormValuesResulted[sSubformDefinitionName][aParamsToSave[i]];
                }
            }

            // Met tout sous forme de FormData
            var oFormData = new FormData();

            for (var key in oValues) {
                // Fichier ?
                if (goog.isDefAndNotNull(oValues[key]['aFiles'])) {
                    oFormData.append(key + '_attached_content', oValues[key]['aFiles'][0]);
                    oFormData.append(key, oValues[key]['aFiles'][0]['name']);
                } else {
                    oFormData.append(key, oValues[key]);
                }
            }

            ajaxRequest({
                "method": "PUT",
                "url": this['oProperties']["web_server_name"] + '/' + this['oProperties']["services_alias"] + '/vmap/querys/' + sBusinessObject,
                "data": oFormData,
                "scope": $rootScope,
                "success": function (response) {
                    if (!goog.isDef(response['data'])) {
                        $translate('NO_VALUES_RETURNED').then(function (sTranslation) {
                            bootbox.alert(sTranslation);
                        });
                        return 0;
                    }

                    if (goog.isDef(response['data']['errorMessage'])) {
                        bootbox.alert(response['data']['errorMessage']);
                        return 0;
                    }

                    if (goog.isDef(response['status'])) {
                        if (response['status'] === 500) {
                            bootbox.alert(response['statusText']);
                            return 0;
                        }
                    }

                    successCallback.call();
                },
                "error": function (response) {
                    bootbox.alert(response);
                    console.error(response);
                }
            });
        },
        /**
         * Insert a business object element
         * @param {string} sBusinessObject
         * @param {object} oBusinessObject
         * @param {object} oSubformDefinition
         * @param {string} sSubformDefinitionName
         * @param {object} oFormValuesResulted
         * @param {function} successCallback
         */
        "insertBoElement": function (sBusinessObject, oBusinessObject, oSubformDefinition, sSubformDefinitionName, oFormValuesResulted, successCallback) {
            var oProperties = this['oProperties'];
            var envSrvc = angular.element(vitisApp.appMainDrtv).injector().get(["envSrvc"]);
            var formSrvc = angular.element(vitisApp.appMainDrtv).injector().get(["formSrvc"]);
            var aParamsToSave = [oBusinessObject['id_field']];

            envSrvc["oFormValues"] = oFormValuesResulted;
            envSrvc["oFormDefinition"] = oSubformDefinition;
            envSrvc["sFormDefinitionName"] = sSubformDefinitionName;

            // Récupère les valeurs à envoyer
            var oValues = formSrvc['getFormData'](sSubformDefinitionName, true);
            for (var i = 0; i < aParamsToSave.length; i++) {
                if (goog.isDefAndNotNull(oFormValuesResulted[sSubformDefinitionName][aParamsToSave[i]])) {
                    oValues[aParamsToSave[i]] = oFormValuesResulted[sSubformDefinitionName][aParamsToSave[i]];
                }
            }

            // Met tout sous forme de FormData
            var oFormData = new FormData();

            for (var key in oValues) {
                // Fichier ?
                if (goog.isDefAndNotNull(oValues[key]['aFiles'])) {
                    oFormData.append(key + '_attached_content', oValues[key]['aFiles'][0]);
                    oFormData.append(key, oValues[key]['aFiles'][0]['name']);
                } else {
                    oFormData.append(key, oValues[key]);
                }
            }

            ajaxRequest({
                "method": "POST",
                "url": this['oProperties']["web_server_name"] + '/' + this['oProperties']["services_alias"] + '/vmap/querys/' + sBusinessObject,
                "data": oFormData,
                "scope": $rootScope,
                "success": function (response) {
                    if (!goog.isDef(response['data'])) {
                        $translate('NO_VALUES_RETURNED').then(function (sTranslation) {
                            bootbox.alert(sTranslation);
                        });
                        return 0;
                    }

                    if (goog.isDef(response['data']['errorMessage'])) {
                        bootbox.alert(response['data']['errorMessage']);
                        return 0;
                    }

                    if (goog.isDef(response['status'])) {
                        if (response['status'] === 500) {
                            bootbox.alert(response['statusText']);
                            return 0;
                        }
                    }

                    successCallback.call();
                },
                "error": function (response) {
                    bootbox.alert(response);
                    console.error(response);
                }
            });
        },
        /**
         * Delete the elements defined in aIds
         * @param {string} sBusinessObject
         * @param {array} aIds
         * @param {function} successCallback
         */
        "deleteBoElements": function (sBusinessObject, aIds, successCallback) {
            var oProperties = this['oProperties'];
            ajaxRequest({
                "method": "DELETE",
                "url": this['oProperties']["web_server_name"] + '/' + this['oProperties']["services_alias"] + '/vmap/querys/' + sBusinessObject,
                "params": {
                    'idList': aIds.join('|')
                },
                "scope": $rootScope,
                "success": function (response) {
                    if (!goog.isDef(response['data'])) {
                        $translate('NO_VALUES_RETURNED').then(function (sTranslation) {
                            bootbox.alert(sTranslation);
                        });
                        return 0;
                    }

                    if (goog.isDef(response['data']['errorMessage'])) {
                        bootbox.alert(response['data']['errorMessage']);
                        return 0;
                    }

                    if (goog.isDef(response['status'])) {
                        if (response['status'] === 500) {
                            bootbox.alert(response['statusText']);
                            return 0;
                        }
                    }

                    successCallback.call();
                },
                "error": function (response) {
                    bootbox.alert(response);
                    console.error(response);
                }
            });
        },
        /**
         * Gets a section definition
         * @param {string} sSection
         * @returns {promise}
         */
        "getSectionDef": function (sSection) {
            var oProperties = this['oProperties'];
            var deferred = $q.defer();
            ajaxRequest({
                "method": "GET",
                "url": this['oProperties']["web_server_name"] + '/' + this['oProperties']["services_alias"] + '/vitis/vitissections/' + sSection,
                "scope": $rootScope,
                "success": function (response) {
                    if (!goog.isDef(response['data'])) {
                        $translate('NO_VALUES_RETURNED').then(function (sTranslation) {
                            bootbox.alert(sTranslation);
                        });
                        return 0;
                    }
                    if (!goog.isDef(response['data']['data'])) {
                        $translate('NO_SECTION_FOUNDED').then(function (sTranslation) {
                            bootbox.alert(sTranslation);
                        });
                        return 0;
                    }
                    if (!goog.isDef(response['data']['data'][0])) {
                        $translate('NO_SECTION_FOUNDED').then(function (sTranslation) {
                            bootbox.alert(sTranslation);
                        });
                        return 0;
                    }
                    if (!goog.isDef(response['data']['data'][0]['ressource_id'])) {
                        $translate('NO_SECTION_FOUNDED').then(function (sTranslation) {
                            bootbox.alert(sTranslation);
                        });
                        return 0;
                    }
                    if (goog.isDef(response['data']['errorMessage'])) {
                        bootbox.alert(response['data']['errorMessage']);
                        return 0;
                    }

                    deferred.resolve(response['data']['data'][0]);

                },
                "error": function (response) {
                    bootbox.alert(response);
                    console.error(response);
                }
            });

            return deferred.promise;
        },
        /**
         * Gets the tab from the given ressource
         * @param {string} sRessourceId
         * @returns {promise}
         */
        "getSectionRessourceTab": function (sRessourceId) {
            var oProperties = this['oProperties'];
            var deferred = $q.defer();
            ajaxRequest({
                "method": "GET",
                "url": this['oProperties']["web_server_name"] + '/' + this['oProperties']["services_alias"] + '/vitis/ressources/' + sRessourceId,
                "headers": {
                    "Accept": 'application/json'
                },
                "scope": $rootScope,
                "success": function (response) {
                    if (!goog.isDef(response['data'])) {
                        $translate('NO_VALUES_RETURNED').then(function (sTranslation) {
                            bootbox.alert(sTranslation);
                        });
                        return 0;
                    }
                    if (!goog.isDef(response['data']['ressources'])) {
                        $translate('NO_VALUES_RETURNED').then(function (sTranslation) {
                            bootbox.alert(sTranslation);
                        });
                        return 0;
                    }
                    if (!goog.isDef(response['data']['ressources'][0])) {
                        $translate('NO_VALUES_RETURNED').then(function (sTranslation) {
                            bootbox.alert(sTranslation);
                        });
                        return 0;
                    }
                    if (goog.isDef(response['data']['errorMessage'])) {
                        bootbox.alert(response['data']['errorMessage']);
                        return 0;
                    }

                    deferred.resolve(response['data']['ressources'][0]);

                },
                "error": function (response) {
                    bootbox.alert(response);
                    console.error(response);
                }
            });

            return deferred.promise;
        },
        /**
         * Gets the data from the given ressource with the given filter
         * @param {string} sRessourceId
         * @param {string} sFilter
         * @returns {promise}
         */
        "getSectionRessourceData": function (sRessourceId, sFilter) {
            var oProperties = this['oProperties'];
            var deferred = $q.defer();
            var oParams = {
                'limit': 100
            };

            if (goog.isDefAndNotNull(sFilter) && sFilter !== '') {
                oParams['filter'] = sFilter;
            }
            ajaxRequest({
                "method": "GET",
                "url": this['oProperties']["web_server_name"] + "/" + this['oProperties']["services_alias"] + "/" + sRessourceId,
                "params": oParams,
                "scope": $rootScope,
                "success": function (response) {
                    if (!goog.isDef(response['data'])) {
                        $translate('NO_VALUES_RETURNED').then(function (sTranslation) {
                            bootbox.alert(sTranslation);
                        });
                        return 0;
                    }
                    if (!goog.isDef(response['data']['data'])) {
                        $translate('NO_VALUES_RETURNED').then(function (sTranslation) {
                            bootbox.alert(sTranslation);
                        });
                        return 0;
                    }
                    if (!goog.isDef(response['data']['data'][0])) {
                        $translate('NO_VALUES_RETURNED').then(function (sTranslation) {
                            bootbox.alert(sTranslation);
                        });
                        return 0;
                    }
                    if (goog.isDef(response['data']['errorMessage'])) {
                        bootbox.alert(response['data']['errorMessage']);
                        return 0;
                    }

                    deferred.resolve(response['data']['data']);

                },
                "error": function (response) {
                    bootbox.alert(response);
                    console.error(response);
                }
            });

            return deferred.promise;
        },
        /**
         * Gets the given section form, can omit the buttons if bHideButtons is true
         * @param {object} oSection
         * @param {boolean} bHideButtons
         * @returns {promise}
         */
        "getSectionForm": function (oSection, bHideButtons) {

            bHideButtons = goog.isDefAndNotNull(bHideButtons) ? bHideButtons : false;

            var deferred = $q.defer();
            var sFileName;

            if (oSection['index'] > 1) {
                sFileName = oSection['mode_id'] + '_' + oSection['tab_name'] + '_' + oSection['name'] + '.json';
            } else {
                sFileName = oSection['mode_id'] + '_' + oSection['tab_name'] + '.json';
            }
            ajaxRequest({
                "method": "GET",
                "url": "modules/" + oSection['module_name'] + "/forms/" + oSection['mode_id'] + "/" + sFileName,
                "scope": $rootScope,
                "success": function (response) {
                    if (!goog.isDef(response['data'])) {
                        $translate('NO_FORM_FOUNDED').then(function (sTranslation) {
                            bootbox.alert(sTranslation);
                        });
                        return 0;
                    }

                    var oForm = response['data'];

                    // Supprime les boutons si bHideButtons
                    if (bHideButtons) {
                        for (var key in oForm) {
                            if (goog.isDefAndNotNull(oForm[key])) {
                                if (goog.isDefAndNotNull(oForm[key]['rows'])) {
                                    for (var i = oForm[key]['rows'].length - 1; i >= 0; i--) {
                                        if (goog.isDefAndNotNull(oForm[key]['rows'][i]['fields'])) {
                                            for (var ii = oForm[key]['rows'][i]['fields'].length - 1; ii >= 0; ii--) {
                                                if (oForm[key]['rows'][i]['fields'][ii]['type'] === 'button') {
                                                    oForm[key]['rows'][i]['fields'].splice(ii, 1);
                                                }
                                            }
                                        }
                                    }
                                }
                            }
                        }
                    }

                    deferred.resolve(oForm);
                },
                "error": function (response) {
                    bootbox.alert(response);
                    console.error(response);
                }
            });

            return deferred.promise;
        },
        /**
         * Update the element of a section
         * @param {string} sRessourceId
         * @param {string} sIdField
         * @param {object} oSubformDefinition
         * @param {string} sSubformDefinitionName
         * @param {object} oFormValuesResulted
         * @returns {promise}
         */
        "updateSectionElement": function (sRessourceId, sIdField, oSubformDefinition, sSubformDefinitionName, oFormValuesResulted) {

            var deferred = $q.defer();
            var oProperties = this['oProperties'];
            var sId = oFormValuesResulted[sSubformDefinitionName][sIdField];
            var sUrl;

            var envSrvc = angular.element(vitisApp.appMainDrtv).injector().get(["envSrvc"]);
            var formSrvc = angular.element(vitisApp.appMainDrtv).injector().get(["formSrvc"]);

            envSrvc["oFormValues"] = oFormValuesResulted;
            envSrvc["oFormDefinition"] = oSubformDefinition;
            envSrvc["sFormDefinitionName"] = sSubformDefinitionName;

            var oFormData = formSrvc["getFormData"](envSrvc["sFormDefinitionName"]);

            if (goog.isDefAndNotNull(sId)) {
                sUrl = this['oProperties']["web_server_name"] + '/' + this['oProperties']["services_alias"] + '/' + sRessourceId + '/' + sId;
            } else {
                sUrl = this['oProperties']["web_server_name"] + '/' + this['oProperties']["services_alias"] + '/' + sRessourceId;
            }
            ajaxRequest({
                "method": "PUT",
                "url": sUrl,
                "data": oFormData,
                "scope": $rootScope,
                "success": function (response) {
                    if (!goog.isDef(response['data'])) {
                        $translate('NO_VALUES_RETURNED').then(function (sTranslation) {
                            bootbox.alert(sTranslation);
                        });
                        return 0;
                    }

                    if (goog.isDef(response['data']['errorMessage'])) {
                        bootbox.alert(response['data']['errorMessage']);
                        return 0;
                    }

                    if (goog.isDef(response['status'])) {
                        if (response['status'] === 500) {
                            bootbox.alert(response['statusText']);
                            return 0;
                        }
                    }

                    deferred.resolve(response['data']);

                },
                "error": function (response) {
                    console.error(response);
                    bootbox.alert(response);
                }
            });

            return deferred.promise;
        },
        /**
         * Insert the element of a section
         * @param {string} sRessourceId
         * @param {string} sIdField
         * @param {object} oSubformDefinition
         * @param {string} sSubformDefinitionName
         * @param {object} oFormValuesResulted
         * @returns {promise}
         */
        "insertSectionElement": function (sRessourceId, sIdField, oSubformDefinition, sSubformDefinitionName, oFormValuesResulted) {
            var deferred = $q.defer();
            var oProperties = this['oProperties'];
            var sId = oFormValuesResulted[sSubformDefinitionName][sIdField];
            var sUrl;

            var envSrvc = angular.element(vitisApp.appMainDrtv).injector().get(["envSrvc"]);
            var formSrvc = angular.element(vitisApp.appMainDrtv).injector().get(["formSrvc"]);

            envSrvc["oFormValues"] = oFormValuesResulted;
            envSrvc["oFormDefinition"] = oSubformDefinition;
            envSrvc["sFormDefinitionName"] = sSubformDefinitionName;

            var oFormData = formSrvc["getFormData"](envSrvc["sFormDefinitionName"]);
            ajaxRequest({
                "method": "POST",
                "url": this['oProperties']["web_server_name"] + "/" + this['oProperties']["services_alias"] + "/" + sRessourceId,
                "data": oFormData,
                "scope": $rootScope,
                "success": function (response) {
                    if (!goog.isDef(response['data'])) {
                        $translate('NO_VALUES_RETURNED').then(function (sTranslation) {
                            bootbox.alert(sTranslation);
                        });
                        return 0;
                    }

                    if (goog.isDef(response['data']['errorMessage'])) {
                        bootbox.alert(response['data']['errorMessage']);
                        return 0;
                    }

                    if (goog.isDef(response['status'])) {
                        if (response['status'] === 500) {
                            bootbox.alert(response['statusText']);
                            return 0;
                        }
                    }

                    deferred.resolve(response['data']);

                },
                "error": function (response) {
                    console.error(response);
                    bootbox.alert(response);
                }
            });

            return deferred.promise;
        },
        /**
         * Delete the section elements given in aIds
         * @param {string} sRessourceId
         * @param {array} aIds
         * @returns {promise}
         */
        "deleteSectionElements": function (sRessourceId, aIds) {

            var deferred = $q.defer();
            var sToken = this['sToken'];
            var oProperties = this['oProperties'];
            ajaxRequest({
                "method": "DELETE",
                "url": this['oProperties']["web_server_name"] + '/' + this['oProperties']["services_alias"] + '/' + sRessourceId,
                "params": {
                    "idList": aIds.join("|")
                },
                "scope": $rootScope,
                "success": function (response) {
                    if (!goog.isDef(response['data'])) {
                        $translate('NO_VALUES_RETURNED').then(function (sTranslation) {
                            bootbox.alert(sTranslation);
                        });
                        return 0;
                    }

                    if (goog.isDef(response['data']['errorMessage'])) {
                        bootbox.alert(response['data']['errorMessage']);
                        return 0;
                    }

                    if (goog.isDef(response['status'])) {
                        if (response['status'] === 500) {
                            bootbox.alert(response['statusText']);
                            return 0;
                        }
                    }

                    deferred.resolve(response['data']);

                },
                "error": function (response) {
                    console.error(response);
                    bootbox.alert(response);
                }
            });

            return deferred.promise;
        },
        /**
         * Disable the elements defined by sElement in the definition defined by oFormDefinition[sFormDefinitionName]
         * if sFormDefinitionName is not defined, the function will browse all the form names
         * @param {object} oFormDefinition
         * @param {srting} sElement
         * @param {string|undefined} sFormDefinitionName
         * @returns {object}
         */
        "disableFormElement": function (oFormDefinition, sElement, sFormDefinitionName) {

            if (!goog.isDefAndNotNull(oFormDefinition)) {
                console.error('oFormDefinition not defined');
                return null;
            }
            if (!goog.isDefAndNotNull(sElement)) {
                console.error('sElement not defined');
                return null;
            }

            // Noms des formulaires dans lesquels aller "disabler" l'élément
            var aFormNames = [];

            if (goog.isDefAndNotNull(sFormDefinitionName)) {
                aFormNames = [sFormDefinitionName];
            } else {
                aFormNames = Object.keys(oFormDefinition);
            }

            for (var i = 0; i < aFormNames.length; i++) {
                if (goog.isDefAndNotNull(oFormDefinition[aFormNames[i]])) {
                    if (goog.isArray(oFormDefinition[aFormNames[i]]['rows'])) {
                        for (var ii = 0; ii < oFormDefinition[aFormNames[i]]['rows'].length; ii++) {
                            if (goog.isDefAndNotNull(oFormDefinition[aFormNames[i]]['rows'][ii]['fields'])) {
                                for (var iii = 0; iii < oFormDefinition[aFormNames[i]]['rows'][ii]['fields'].length; iii++) {
                                    if (oFormDefinition[aFormNames[i]]['rows'][ii]['fields'][iii]['name'] === sElement) {
                                        oFormDefinition[aFormNames[i]]['rows'][ii]['fields'][iii]['disabled'] = true;
                                    }
                                }
                            }
                        }
                    }
                }
            }

            return oFormDefinition;
        },
        /**
         * Make ajax requests etc..
         * @param {string} sFormName
         */
        "extractFormDefinitionInfos": function (sFormName) {
            var scope = angular.element("form[name='" + sFormName + "']").scope();
            scope['ctrl'].extractFormDefinitionInfos();
        },
        /**
         * Get the Geolocation object
         * @returns {ol.Geolocation}
         */
        "getGPS": function () {
            return goog.isDefAndNotNull(vitisApp['oGeoLocation']) ? vitisApp['oGeoLocation'] : null;
        },
        /**
         * Get the GPS Position
         * @returns {array|undefined}
         */
        "getGPSPosition": function () {
            var oGPS = vitisApp['oGeoLocation'];
            if (goog.isDefAndNotNull(oGPS)) {
                return oGPS.getPosition();
            }
        },
        /**
         * Get the GPS Latitude
         * @returns {number|undefined}
         */
        "getGPSLat": function () {
            var oGPS = vitisApp['oGeoLocation'];
            if (goog.isDefAndNotNull(oGPS)) {
                var aPosition = oGPS.getPosition();
                if (goog.isDefAndNotNull(aPosition)) {
                    return oGPS.getPosition()[1];
                } else {
                    console.error('GPS position undefined');
                }
            } else {
                console.error('GPS undefined');
            }
        },
        /**
         * Get the GPS Longitude
         * @returns {number|undefined}
         */
        "getGPSLong": function () {
            var oGPS = vitisApp['oGeoLocation'];
            if (goog.isDefAndNotNull(oGPS)) {
                var aPosition = oGPS.getPosition();
                if (goog.isDefAndNotNull(aPosition)) {
                    return oGPS.getPosition()[0];
                }
            } else {
                console.error('GPS undefined');
            }
        },
        /**
         * Get the GPS Altitude
         * @returns {number|undefined}
         */
        "getGPSAltitude": function () {
            var oGPS = vitisApp['oGeoLocation'];
            if (goog.isDefAndNotNull(oGPS)) {
                return oGPS.getAltitude();
            } else {
                console.error('GPS undefined');
            }
        },
        /**
         * Get the GPS Accuracy
         * @returns {number|undefined}
         */
        "getGPSAccuracy": function () {
            var oGPS = vitisApp['oGeoLocation'];
            if (goog.isDefAndNotNull(oGPS)) {
                return oGPS.getAccuracy();
            } else {
                console.error('GPS undefined');
            }
        },
        /**
         * Get the GPS Heading
         * @returns {number|undefined}
         */
        "getGPSHeading": function () {
            var oGPS = vitisApp['oGeoLocation'];
            if (goog.isDefAndNotNull(oGPS)) {
                return oGPS.getHeading();
            } else {
                console.error('GPS undefined');
            }
        }
    };
};
formReader.module.service("formReaderService", formReader.formReaderService);
